/* extension.js
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

/* exported init */

const Main = imports.ui.main;
const PopupMenu = imports.ui.popupMenu;
const Mainloop = imports.mainloop;
const Util = imports.misc.util;
const ExtensionUtils = imports.misc.extensionUtils;
const Me = ExtensionUtils.getCurrentExtension();

const {GLib, Gio} = imports.gi;

const FAN_PROCFILE = '/proc/acpi/ibm/fan';

function readFile(path) {
    return imports.byteArray.toString(GLib.file_get_contents(path)[1]);
}

function readFanStatus() {
    const file = readFile(FAN_PROCFILE)
        .split('\n')
        .map(line => line.replace(/\s*/g, '').split(':', 2))
    ;

    return Object.fromEntries(file);
}

class FanMenu {
    constructor() {
        this._popup = new PopupMenu.PopupSubMenuMenuItem('Loading', true);
        this._popup.icon.gicon = Gio.icon_new_for_string(Me.path + '/icons/fan-symbolic.svg');
        Main.panel.statusArea.aggregateMenu.menu.addMenuItem(this._popup, 11 - 2);

        this._index = 0;
        this._items = {};
    }

    addEntry(str, callback) {
        if(!this._popup) {
            return;
        }

        const menuItem = new PopupMenu.PopupMenuItem(str);

        if(callback) {
            menuItem.connect('activate', callback);
        }
    
        this._popup.menu.addMenuItem(menuItem);
    }

    setStatus(status) {
        if(!this._popup) {
            return;
        }
    
        this._popup.label.set_text('Fan: ' + status);
    }

    destroy() {
        if(this._popup) {
            this._popup.destroy();
            this._popup = null;
        }
    }
}

class Extension {
    enable() {
        this.disable();

        this._fanMenu = new FanMenu();
        this._fanMenu.setStatus('Loading');

        for(const item of ['auto', ...Array.from(Array(10).keys()), 'full-speed']) {
            this._fanMenu.addEntry('level '+item, () => {
                Util.trySpawnCommandLine(`pkexec bash -c 'echo level ${item} > ${FAN_PROCFILE}'`);
                this._tryRefresh();
            });
        }

        this._timeout = Mainloop.timeout_add_seconds(2, () => {
            this._tryRefresh();

            return true;
        });
    }

    disable() {
        if(this._fanMenu) {
            this._fanMenu.destroy();
            this._fanMenu = null;
        }

        if (this._timeout) {
            Mainloop.source_remove(this._timeout);
            this._timeout = null;
        }
    }

    
    _tryRefresh() {
        try {
            this._refresh();
        } catch(error) {
            logError(error);
        }
    }

    _refresh() {
        if(!this._fanMenu) {
            return;
        }

        const status = readFanStatus();

        if(status.level === 'disengaged') {
            status.level = 'full-speed';
        }

        this._fanMenu.setStatus(`${status.level || '?'} (${status.speed || '?'} RPM)`);
    }
}

function init() {
    return new Extension();
}
