/**
 * ====================================================
 *   Serenity Desktop Gnome Extension
 * ====================================================
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * Homepage: https://gitlab.com/sisujs/serenity-desktop
 *
 * Contributors:
 *  - Marko Lavikainen: Main Developer
 *
 * This code was generated by Rollup-bundler
 *
 */
import { Extension } from "resource:///org/gnome/shell/extensions/extension.js";
import * as Main from "resource:///org/gnome/shell/ui/main.js";
import Meta from "gi://Meta";
import Shell from "gi://Shell";
import St from "gi://St";
import * as PanelMenu from "resource:///org/gnome/shell/ui/panelMenu.js";
import * as PopupMenu from "resource:///org/gnome/shell/ui/popupMenu.js";
import Clutter from "gi://Clutter";
class Logger {
  static PREFIX="serenity-desktop@sisujs.fi: ";
  static isDebug;
  static log(msg) {
    console.log(Logger.PREFIX + msg);
  }
  static error(msg, err) {
    console.error(Logger.PREFIX + msg, err);
  }
  static debug(msg, obj = null) {
    if (Logger.isDebug) {
      if (obj) {
        console.log(Logger.PREFIX + msg, JSON.stringify(obj));
      } else {
        console.log(Logger.PREFIX + msg);
      }
    }
  }
}
class WindowsDB {
  nativeEvents;
  events;
  windows=new Map;
  metaWindows=new Map;
  constructor(nativeEvents, events) {
    this.nativeEvents = nativeEvents;
    this.events = events;
  }
  enable() {
    this.updateWindows();
    this.nativeEvents.connect(global.display, "window-created", ((_display, window) => this.onWindowCreate(window)));
    this.nativeEvents.connect(global.window_manager, "destroy", ((_shellwm, actor) => this.onWindowDestroy(actor.meta_window)));
  }
  onWindowCreate(mw) {
    this.updateWindows();
  }
  initWindow(id) {
    const mw = this.metaWindows.get(id);
    const uw = new UDWindow(id, this.getAppKey(mw), (() => this.metaWindows.get(id)));
    this.nativeEvents.connect(mw, "notify::title", (() => this.events.notifyTitleChanged(mw.get_id())));
    return uw;
  }
  refresh() {
    this.updateWindows();
  }
  updateWindows() {
    const mws = this.getMetaWindows();
    const ids = this.getIds();
    const positions = this.getWindowIdStack();
    for (let mw of mws) {
      const id = mw.get_id();
      this.metaWindows.set(id, mw);
      ids.delete(id);
      const uw = this.get(id);
      uw.stackPos = positions.get(id) ?? -1;
    }
    ids.forEach((id => this.windows.delete(id)));
  }
  onWindowDestroy(mw) {
    this.updateWindows();
    this.nativeEvents.disconnectTarget(mw);
    this.events.notifyWindowDestroyed(mw.get_id());
  }
  clear() {
    this.windows.clear();
  }
  getIds() {
    return new Set(this.windows.keys());
  }
  get(id) {
    if (id == null) {
      return null;
    }
    if (!this.windows.has(id)) {
      this.windows.set(id, this.initWindow(id));
    }
    return this.windows.get(id);
  }
  exists(id) {
    return this.windows.has(id);
  }
  remove(id) {
    this.windows.delete(id);
  }
  getAppKey(mw) {
    let app = Shell.WindowTracker.get_default().get_window_app(mw);
    return `${app.get_name()}.${app.get_id()}`;
  }
  query(query) {
    return [ ...this.windows.values() ].filter((uw => {
      if (query.monitorIndex != null && query.workspaceIndex != null) {
        return uw.monitorIndex() == query.monitorIndex && uw.workspaceIndex() == query.workspaceIndex;
      } else if (query.monitorIndex != null) {
        return uw.monitorIndex() == query.monitorIndex;
      } else if (query.monitorIndex != null) {
        return uw.workspaceIndex() == query.workspaceIndex;
      }
      return true;
    }));
  }
  getMetaWindows() {
    return global.get_window_actors().map((w => w.metaWindow)).filter((w => w.get_window_type() == Meta.WindowType.NORMAL && !w.is_skip_taskbar() && !w.is_always_on_all_workspaces()));
  }
  getWindowIdStack() {
    const ids = new Map;
    const list = global.display.get_tab_list(Meta.TabList.NORMAL_ALL, null);
    for (let c = 0; c < list.length; c++) {
      ids.set(list[c].get_id(), c);
    }
    return ids;
  }
}
class WindowSettings {
  maximized=false;
  static fromPlain(obj) {
    const ws = new WindowSettings;
    ws.maximized = !!(obj.maximized ?? false);
    return ws;
  }
}
class UDWindow {
  id;
  appKey;
  windowGetter;
  stackPos;
  windowCreateTime;
  windowMazimizedOnCreate=false;
  constructor(id, appKey, windowGetter) {
    this.id = id;
    this.appKey = appKey;
    this.windowGetter = windowGetter;
  }
  metaWindow() {
    return this.windowGetter();
  }
  isAlive() {
    const mw = this.metaWindow();
    return mw?.isAlive ?? false;
  }
  activate() {
    this.metaWindow()?.activate(global.get_current_time());
  }
  monitorIndex() {
    return this.metaWindow()?.get_monitor();
  }
  workspaceIndex() {
    return this.metaWindow()?.get_workspace().index();
  }
  title() {
    return this.metaWindow().get_title();
  }
  moveTo(query) {
    if (query.workspaceIndex != null) {
      this.metaWindow().change_workspace_by_index(query.workspaceIndex, false);
    }
  }
}
class ShortcutsDB {
  events;
  static shortcuts=new Array(10);
  constructor(events) {
    this.events = events;
  }
  setShortcut(index, id) {
    if (index == null || id == null || ShortcutsDB.shortcuts[index] == id) {
      return;
    }
    if (ShortcutsDB.shortcuts[index] != null) {
      ShortcutsDB.shortcuts[index] = null;
    }
    for (let i = 0; i < ShortcutsDB.shortcuts.length; i++) {
      if (ShortcutsDB.shortcuts[i] == id) {
        ShortcutsDB.shortcuts[i] = null;
      }
    }
    ShortcutsDB.shortcuts[index] = id;
    this.events.notifyShortcutAdded(index, id);
  }
  removeByWindowId(id) {
    const index = this.getIndexByWindowId(id);
    if (index != null) {
      ShortcutsDB.shortcuts[index] = null;
      this.events.notifyShortcutRemoved(index, id);
    }
  }
  removeByIndex(index) {
    if (ShortcutsDB.shortcuts[index] != null) {
      ShortcutsDB.shortcuts[index] = null;
      this.events.notifyShortcutRemoved(index, ShortcutsDB.shortcuts[index]);
    }
  }
  getIndexByWindowId(id) {
    for (let i = 0; i < ShortcutsDB.shortcuts.length; i++) {
      if (ShortcutsDB.shortcuts[i] == id) {
        return i;
      }
    }
    return null;
  }
  getWindowIdByIndex(index) {
    return ShortcutsDB.shortcuts[index];
  }
}
class Db {
  windows;
  shortcuts;
  constructor(nativeEvents, events) {
    this.shortcuts = new ShortcutsDB(events);
    this.windows = new WindowsDB(nativeEvents, events);
  }
  enable() {
    this.windows.enable();
  }
  disable() {
    this.windows.clear();
  }
}
class Events {
  shortcutAddListeners=[];
  shortcutRemoveListeners=[];
  titleChangedListeners=[];
  windowDestroyedListeners=[];
  addTitleChangedListener(listener) {
    this.titleChangedListeners.push(listener);
  }
  addWindowDestroyedListener(listener) {
    this.windowDestroyedListeners.push(listener);
  }
  addShortcutAddListener(listener) {
    this.shortcutAddListeners.push(listener);
  }
  addShortcutRemoveListener(listener) {
    this.shortcutRemoveListeners.push(listener);
  }
  notifyShortcutAdded(index, windowId) {
    this.shortcutAddListeners.forEach((l => l(index, windowId)));
  }
  notifyShortcutRemoved(index, windowId) {
    this.shortcutRemoveListeners.forEach((l => l(index, windowId)));
  }
  notifyTitleChanged(windowId) {
    this.titleChangedListeners.forEach((l => l(windowId)));
  }
  notifyWindowDestroyed(windowId) {
    this.windowDestroyedListeners.forEach((l => l(windowId)));
  }
}
class NativeEventHandler {
  handlers=new Map;
  connect(target, event, handler) {
    if (!this.handlers.has(target)) {
      this.handlers.set(target, new Map);
    }
    const oldId = this.handlers.get(target).get(event);
    if (oldId) {
      target.disconnect(oldId);
    }
    this.handlers.get(target).set(event, target.connect(event, ((...args) => handler(...args))));
  }
  disconnectTarget(target) {
    if (this.handlers.has(target)) {
      for (let id of this.handlers.get(target).values()) {
        target.disconnect(id);
      }
    }
    this.handlers.delete(target);
  }
  enable() {}
  disable() {
    for (let target of [ ...this.handlers.keys() ]) {
      this.disconnectTarget(target);
    }
  }
}
class WorkspaceMgr {
  db;
  nativeEvents;
  events;
  settings;
  static SETTINGS_KEY="window-settings";
  monitorCount=0;
  workspaceCount=0;
  currentWorkspaceIndex=-1;
  originalGetNeighbor=null;
  windowSettings;
  constructor(db, nativeEvents, events, settings) {
    this.db = db;
    this.nativeEvents = nativeEvents;
    this.events = events;
    this.settings = settings;
  }
  enable() {
    this.readSettings();
    this.addKeyBindings();
    this.initWorkspaces(true);
    this.nativeEvents.connect(global.workspace_manager, "active-workspace-changed", (() => this.workspaceChanged(null, null)));
    this.nativeEvents.connect(global.workspace_manager, "notify::n-workspaces", (() => this.initWorkspaces(true)));
    this.nativeEvents.connect(global.display, "notify::n-monitors", (() => this.initWorkspaces(true)));
    this.workspaceChanged(null, null);
    this.events.addWindowDestroyedListener((id => this.onWindowDestroyed(id)));
  }
  onWindowDestroyed(id) {
    this.db.shortcuts.removeByWindowId(id);
  }
  enableWrapAround() {
    const self = this;
    this.originalGetNeighbor = Meta.Workspace.prototype.get_neighbor;
    Meta.Workspace.prototype.get_neighbor = function(direction) {
      if (self.switchWorkspace(direction)) {
        return self.originalGetNeighbor.call(this, direction);
      } else {
        return this;
      }
    };
  }
  switchWorkspace(direction) {
    let newIndex = null;
    if (direction) {
      let activeIndex = global.workspace_manager.get_active_workspace_index();
      let lastIndex = global.workspace_manager.n_workspaces - 1;
      if (direction == Meta.MotionDirection.LEFT && activeIndex == 0) {
        newIndex = lastIndex;
      } else if (direction == Meta.MotionDirection.RIGHT && activeIndex == lastIndex) {
        newIndex = 0;
      }
    }
    if (newIndex != null) {
      this.activateWorkspace(newIndex);
    }
    return newIndex == null;
  }
  async activateWorkspace(index) {
    global.workspace_manager.get_workspace_by_index(index).activate(global.get_current_time());
  }
  findSuitableWorkspace(workspaceIndex, monitorIndex) {
    Logger.debug(`findFreeWorkspace: monitorIndex=${monitorIndex}, workspaceCount=${this.workspaceCount}`);
    let windowCount = this.db.windows.query({
      workspaceIndex: workspaceIndex,
      monitorIndex: monitorIndex
    }).length;
    if (windowCount <= 1) {
      return workspaceIndex;
    }
    for (let i = 0; i < this.workspaceCount; i++) {
      const windowCount = this.db.windows.query({
        workspaceIndex: i,
        monitorIndex: monitorIndex
      }).length;
      if (i == workspaceIndex && windowCount == 1 || windowCount == 0) {
        return i;
      }
    }
    return null;
  }
  disableWrapAround() {
    if (this.originalGetNeighbor) {
      Meta.Workspace.prototype.get_neighbor = this.originalGetNeighbor;
      this.originalGetNeighbor = null;
    }
  }
  addKeyBindings() {
    for (let i = 0; i < 10; i++) {
      Main.wm.addKeybinding(`open-shortcut-${i}`, this.settings, Meta.KeyBindingFlags.NONE, Shell.ActionMode.ALL, (() => this.openShortcut(i)));
      Main.wm.addKeybinding(`add-shortcut-${i}`, this.settings, Meta.KeyBindingFlags.NONE, Shell.ActionMode.ALL, (() => this.addShortcut(i)));
    }
  }
  removeKeyBindings() {
    for (let i = 0; i < 10; i++) {
      Main.wm.removeKeybinding(`open-shortcut-${i}`);
      Main.wm.removeKeybinding(`add-shortcut-${i}`);
    }
  }
  addShortcut(index) {
    try {
      const window = global.display.get_focus_window();
      if (window) {
        this.db.shortcuts.setShortcut(index, window.get_id());
        Logger.debug("Add shortcut " + index + ":" + (window?.get_title() ?? "No window"));
      } else {
        Logger.debug("addShortcut: No window in focus");
      }
    } catch (err) {
      Logger.error("Error while adding a shortcut", err);
    }
  }
  openShortcut(index) {
    Logger.debug("Open shortcut " + index);
    const uw = this.db.windows.get(this.db.shortcuts.getWindowIdByIndex(index));
    if (uw) {
      this.db.windows.refresh();
      const diff = global.workspace_manager.get_active_workspace_index() - uw.workspaceIndex();
      const repositioned = new Set;
      for (let i = 0; i < this.workspaceCount; i++) {
        this.updateWindowPositions(repositioned, i, uw.monitorIndex(), diff);
      }
      uw.activate();
    } else {
      this.db.shortcuts.removeByIndex(index);
      Logger.debug("No shortcut for index " + index);
    }
  }
  readSettings() {
    this.windowSettings = new Map;
    const values = this.settings.get_strv(WorkspaceMgr.SETTINGS_KEY);
    if (values) {
      for (let value of values) {
        try {
          const splits = value.split("|");
          if (splits.length == 2) {
            this.windowSettings.set(splits[0], WindowSettings.fromPlain(JSON.parse(splits[1])));
          }
        } catch (err) {
          Logger.error("Failed to read settings value: " + value, err);
        }
      }
    }
  }
  initWorkspaces(force) {
    const mCount = global.display.get_n_monitors();
    const wCount = global.workspace_manager.get_n_workspaces();
    if (force || this.monitorCount != mCount || this.workspaceCount != wCount) {
      this.db.windows.clear();
      this.monitorCount = mCount;
      this.workspaceCount = wCount;
      return true;
    }
    return false;
  }
  workspaceChanged(workspaceIndex, monitorIndex) {
    const newIndex = global.workspace_manager.get_active_workspace_index();
    if (this.currentWorkspaceIndex == newIndex) {
      Logger.debug("No workspace change");
      return;
    } else {
      Logger.debug("Workspace changed");
    }
    try {
      if (this.initWorkspaces(false)) {
        return;
      }
      this.updatePositions(workspaceIndex ?? global.workspace_manager.get_active_workspace_index(), monitorIndex ?? global.display.get_current_monitor());
    } catch (err) {
      Logger.log(err);
    } finally {
      this.currentWorkspaceIndex = newIndex;
    }
  }
  updatePositions(activeWorkspaceIndex, activeMonitorIndex) {
    this.db.windows.refresh();
    const diff = activeWorkspaceIndex - this.currentWorkspaceIndex;
    const repositioned = new Set;
    for (let monitorIndex = 0; monitorIndex < this.monitorCount; monitorIndex++) {
      if (monitorIndex != activeMonitorIndex) {
        for (let workspaceIndex = 0; workspaceIndex < this.workspaceCount; workspaceIndex++) {
          this.updateWindowPositions(repositioned, workspaceIndex, monitorIndex, diff);
        }
      }
    }
    this.currentWorkspaceIndex = activeWorkspaceIndex;
  }
  updateWindowPositions(repositioned, workspaceIndex, monitorIndex, diff) {
    const monitorWindows = this.db.windows.query({
      workspaceIndex: workspaceIndex,
      monitorIndex: monitorIndex
    });
    const newIndex = (this.workspaceCount + workspaceIndex + diff) % this.workspaceCount;
    const updateable = [];
    for (let uw of monitorWindows) {
      if (!repositioned.has(uw.id)) {
        repositioned.add(uw.id);
        if (uw.workspaceIndex() != newIndex) {
          updateable.push(uw);
        }
      }
    }
    updateable.sort(((uw1, uw2) => uw2.stackPos - uw1.stackPos));
    for (let uw of updateable) {
      try {
        Logger.log(`Reposition window: ${uw.id} ${uw.title()}, monitor = ${monitorIndex}, workspace = ${newIndex}`);
        uw.moveTo({
          workspaceIndex: newIndex
        });
      } catch (err) {
        Logger.error("Error", err);
        this.db.windows.remove(uw.id);
      }
    }
  }
  disable() {
    this.removeKeyBindings();
  }
}
class Notifier {
  static notify(msg) {
    Main.notify("Serenity Desktop", msg);
  }
}
class UDIndicator {
  db;
  uuid;
  button;
  shortcuts;
  constructor(db, events, uuid) {
    this.db = db;
    this.uuid = uuid;
    events.addShortcutAddListener(((si, wi) => this.shortcutAdded(si, wi)));
    events.addShortcutRemoveListener((() => this.shortcutsChanged()));
    events.addTitleChangedListener((id => this.titleChanged(id)));
  }
  async shortcutAdded(shorcutIndex, windowId) {
    this.shortcutsChanged();
    const uw = this.db.windows.get(windowId);
    if (uw) {
      Notifier.notify(`Shortcut ${shorcutIndex} added: ${uw.title()}`);
    }
  }
  async shortcutsChanged() {
    try {
      for (let i = 0; i < 10; i++) {
        const shortcut = this.shortcuts[i];
        const uw = this.db.windows.get(this.db.shortcuts.getWindowIdByIndex(i));
        if (uw) {
          shortcut.button.reactive = true;
          shortcut.button.can_focus = true;
          shortcut.label.text = `${i}: ${uw.title()}`;
        } else {
          shortcut.button.reactive = false;
          shortcut.button.can_focus = false;
          shortcut.label.text = `${i}: [No shortcut]`;
        }
      }
    } catch (err) {
      Logger.error("shortcutsChanged", err);
    }
  }
  titleChanged(windowId) {
    const index = this.db.shortcuts.getIndexByWindowId(windowId);
    const uw = this.db.windows.get(windowId);
    if (index != null && uw) {
      this.shortcuts[index].label.text = `${index}: ${uw.title()}`;
    }
  }
  enable() {
    this.shortcuts = new Array(10);
    this.createStatusAreaButton();
    this.addShortcutsHeader();
    for (let i = 1; i <= 10; i++) {
      this.addShortcutItem(i % 10);
    }
    Main.panel.addToStatusArea(this.uuid, this.button);
    this.shortcutsChanged();
  }
  addMenuItem(item) {
    this.button.menu.addMenuItem(item);
  }
  createStatusAreaButton() {
    this.button = new PanelMenu.Button(0, "Serenity Desktop");
    this.button.add_child(new St.Icon({
      icon_name: "video-display-symbolic",
      style_class: "system-status-icon"
    }));
  }
  addShortcutsHeader() {
    let header = new PopupMenu.PopupBaseMenuItem({
      reactive: false,
      can_focus: false
    });
    let label = new St.Label({
      text: "Serenity Desktop - Shortcuts",
      x_align: Clutter.ActorAlign.START,
      style_class: "header-label"
    });
    header.add_child(label);
    this.addMenuItem(header);
  }
  addShortcutItem(index) {
    let menuItem = new PopupMenu.PopupBaseMenuItem({
      reactive: false
    });
    this.shortcuts[index] = {
      label: new St.Label({
        text: `${index}: [No shortcut]`,
        x_align: Clutter.ActorAlign.START,
        x_expand: true,
        y_align: Clutter.ActorAlign.CENTER
      }),
      button: new St.Button({
        label: "Remove",
        style_class: "button",
        x_align: Clutter.ActorAlign.END,
        y_align: Clutter.ActorAlign.CENTER,
        reactive: false,
        can_focus: false
      })
    };
    this.shortcuts[index].button.connect("clicked", (() => {
      this.db.shortcuts.removeByIndex(index);
    }));
    let layout = new St.BoxLayout({
      vertical: false,
      x_expand: true,
      y_align: Clutter.ActorAlign.CENTER
    });
    layout.add_child(this.shortcuts[index].label);
    layout.add_child(new St.Widget({
      width: 10
    }));
    layout.add_child(this.shortcuts[index].button);
    menuItem.add_child(layout);
    this.addMenuItem(menuItem);
  }
  disable() {
    try {
      if (this.button) {
        this.button.destroy();
      }
    } catch (err) {
      Logger.error("Failed to disable service", err);
    } finally {
      this.button = null;
      this.shortcuts = null;
    }
  }
}
class SerenityDesktopExtension extends Extension {
  db;
  indicator;
  workspaceMgr;
  originalShouldAnimate;
  nativeEvents;
  enable() {
    try {
      Logger.isDebug = false;
      const events = new Events;
      const nativeEvents = new NativeEventHandler;
      this.db = new Db(nativeEvents, events);
      this.db.windows.enable();
      this.indicator = new UDIndicator(this.db, events, this.uuid);
      this.workspaceMgr = new WorkspaceMgr(this.db, nativeEvents, events, this.getSettings());
      this.workspaceMgr.enable();
      this.indicator.enable();
      this.disableSwitchAnimations();
    } catch (err) {
      Logger.error("Failed to enable extension", err);
      this.disable();
    }
  }
  disable() {
    try {
      this.disableService(this.nativeEvents);
      this.disableService(this.db);
      this.disableService(this.workspaceMgr);
      this.disableService(this.indicator);
      this.restoreSwitchAnimations();
    } catch (err) {
      Logger.error("Failed to disable extension", err);
    } finally {
      this.db = null;
      this.workspaceMgr = null;
      this.indicator = null;
      this.originalShouldAnimate = null;
    }
  }
  disableSwitchAnimations() {
    if (Main.wm["_shouldAnimate"]) {
      this.originalShouldAnimate = Main.wm["_shouldAnimate"];
      Main.wm["_shouldAnimate"] = function() {
        return false;
      };
    }
  }
  restoreSwitchAnimations() {
    if (Main.wm["_shouldAnimate"] && this.originalShouldAnimate) {
      Main.wm["_shouldAnimate"] = this.originalShouldAnimate;
    }
  }
  disableService(service) {
    if (service) {
      try {
        service.disable();
      } catch (err) {
        Logger.error("Failed to disable local service", err);
      }
    }
  }
}
export { SerenityDesktopExtension as default };
