/**
 * ====================================================
 *   Serenity Desktop Gnome Extension
 * ====================================================
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * Homepage: https://gitlab.com/sisujs/serenity-desktop
 *
 * Contributors:
 *  - Marko Lavikainen: Main Developer
 *
 * This code was generated by Rollup-bundler
 *
 */
import { Extension } from "resource:///org/gnome/shell/extensions/extension.js";
import Gio from "gi://Gio";
import * as Main from "resource:///org/gnome/shell/ui/main.js";
import Meta from "gi://Meta";
import Shell from "gi://Shell";
import St from "gi://St";
import * as PanelMenu from "resource:///org/gnome/shell/ui/panelMenu.js";
import * as PopupMenu from "resource:///org/gnome/shell/ui/popupMenu.js";
import Clutter from "gi://Clutter";
class Logger {
  static PREFIX="serenity-desktop@sisujs.fi:";
  static isDebug;
  static info(msg) {
    console.log(`${Logger.PREFIX} (INFO) ${msg}`);
  }
  static error(msg, err) {
    console.error(`${Logger.PREFIX} (ERROR) ${msg}`, err);
  }
  static debug(msg, obj = null) {
    if (Logger.isDebug) {
      if (obj) {
        const lines = `${msg} ${JSON.stringify(obj, null, 2)}`.split("\n");
        lines.forEach((l => Logger.debug(l)));
      } else {
        console.log(`${Logger.PREFIX} (DEBUG) ${msg}`);
      }
    }
  }
}
var Theme;
(function(Theme) {
  Theme["DARK"] = "dark";
  Theme["LIGHT"] = "light";
})(Theme || (Theme = {}));
class WindowSettings {
  maximized=false;
  static fromPlain(obj) {
    const ws = new WindowSettings;
    ws.maximized = !!(obj.maximized ?? false);
    return ws;
  }
}
class SWindow {
  id;
  appKey;
  windowGetter;
  stackPos;
  windowCreateTime;
  windowMazimizedOnCreate=false;
  constructor(id, appKey, windowGetter) {
    this.id = id;
    this.appKey = appKey;
    this.windowGetter = windowGetter;
  }
  metaWindow() {
    return this.windowGetter();
  }
  isAlive() {
    const mw = this.metaWindow();
    return mw?.isAlive ?? false;
  }
  activate() {
    this.metaWindow()?.activate(global.get_current_time());
  }
  monitorIndex() {
    return this.metaWindow()?.get_monitor();
  }
  workspaceIndex() {
    return this.metaWindow()?.get_workspace().index();
  }
  title() {
    return this.metaWindow().get_title();
  }
  moveTo(query) {
    if (query.workspaceIndex != null) {
      this.metaWindow().change_workspace_by_index(query.workspaceIndex, false);
    }
  }
  isOnAllWorkspaces() {
    const mw = this.metaWindow();
    return mw == null ? false : mw.is_always_on_all_workspaces() || mw.is_on_all_workspaces();
  }
}
class RecursionDetector {
  increment;
  treshold;
  currentTime=0;
  constructor(increment, treshold) {
    this.increment = increment;
    this.treshold = treshold;
  }
  isRecursive() {
    const now = Date.now();
    this.currentTime += this.increment;
    if (this.currentTime < now) {
      this.currentTime = now;
      return false;
    }
    const recursive = this.currentTime >= now + this.treshold;
    if (recursive) {
      Logger.info("Recursion detected");
    } else {
      Logger.debug("Not recursive", {
        currentTime: this.currentTime,
        now: now
      });
    }
    return recursive;
  }
}
class WorkspaceMgr {
  db;
  nativeEvents;
  events;
  settings;
  static SETTINGS_KEY="window-settings";
  monitorCount=0;
  workspaceCount=0;
  workspaceIndex=-1;
  rd;
  windowSettings;
  addWorkspaceToLeft=false;
  constructor(db, nativeEvents, events, settings) {
    this.db = db;
    this.nativeEvents = nativeEvents;
    this.events = events;
    this.settings = settings;
  }
  enable() {
    this.rd = new RecursionDetector(10, 100);
    this.readSettings();
    this.addKeyBindings();
    this.initWorkspaces();
    this.nativeEvents.connect(global.workspace_manager, "active-workspace-changed", (() => this.workspaceChanged()));
    this.nativeEvents.connect(global.workspace_manager, "notify::n-workspaces", (() => this.workspaceCountChanged()));
    this.nativeEvents.connect(global.display, "notify::n-monitors", (() => this.initWorkspaces()));
    this.nativeEvents.connect(Main.layoutManager, "monitors-changed", (() => this.initWorkspaces()));
    this.events.addWindowDestroyedListener((id => this.onWindowDestroyed(id)));
  }
  onWindowDestroyed(id) {
    this.db.shortcuts.removeByWindowId(id);
  }
  addKeyBindings() {
    for (let i = 0; i < 10; i++) {
      Main.wm.addKeybinding(`open-shortcut-${i}`, this.settings, Meta.KeyBindingFlags.NONE, Shell.ActionMode.ALL, (() => this.openShortcut(i)));
      Main.wm.addKeybinding(`add-shortcut-${i}`, this.settings, Meta.KeyBindingFlags.NONE, Shell.ActionMode.ALL, (() => this.addShortcut(i)));
    }
  }
  removeKeyBindings() {
    for (let i = 0; i < 10; i++) {
      Main.wm.removeKeybinding(`open-shortcut-${i}`);
      Main.wm.removeKeybinding(`add-shortcut-${i}`);
    }
  }
  addShortcut(index) {
    try {
      const window = global.display.get_focus_window();
      if (window) {
        this.db.shortcuts.setShortcut(index, window.get_id());
        Logger.debug("Add shortcut " + index + ":" + (window?.get_title() ?? "No window"));
      } else {
        Logger.debug("addShortcut: No window in focus");
      }
    } catch (err) {
      Logger.error("Error while adding a shortcut", err);
    }
  }
  openShortcut(index) {
    Logger.debug("Open shortcut " + index);
    const uw = this.db.windows.get(this.db.shortcuts.getWindowIdByIndex(index));
    if (uw) {
      this.db.windows.refresh();
      const diff = global.workspace_manager.get_active_workspace_index() - uw.workspaceIndex();
      const repositioned = new Set;
      for (let i = 0; i < this.workspaceCount; i++) {
        this.updateWindowPositions(repositioned, i, uw.monitorIndex(), diff);
      }
      uw.activate();
    } else {
      this.db.shortcuts.removeByIndex(index);
      Logger.debug("No shortcut for index " + index);
    }
  }
  readSettings() {
    this.windowSettings = new Map;
    const values = this.settings.get_strv(WorkspaceMgr.SETTINGS_KEY);
    if (values) {
      for (let value of values) {
        try {
          const splits = value.split("|");
          if (splits.length == 2) {
            this.windowSettings.set(splits[0], WindowSettings.fromPlain(JSON.parse(splits[1])));
          }
        } catch (err) {
          Logger.error("Failed to read settings value: " + value, err);
        }
      }
    }
  }
  initWorkspaces() {
    this.db.windows.refresh();
    this.monitorCount = global.display.get_n_monitors();
    this.workspaceCount = global.workspace_manager.get_n_workspaces();
    this.workspaceIndex = global.workspace_manager.get_active_workspace_index();
    this.addWorkspaceToLeft = false;
  }
  workspaceCountChanged() {
    const wCount = global.workspace_manager.get_n_workspaces();
    const activeIndex = global.workspace_manager.get_active_workspace_index();
    try {
      Logger.debug("workspaceCountChanged()", {
        workspaces: `Workspace change: ${this.workspaceCount} -> ${wCount}`,
        activeIndex: activeIndex
      });
      if (this.workspaceCount != wCount) {
        if (wCount > 2 && wCount > this.workspaceCount && activeIndex == 0) {
          this.addWorkspaceToLeft = true;
        } else {
          this.addWorkspaceToLeft = false;
          this.updatePositions(activeIndex, global.display.get_current_monitor());
        }
      }
    } catch (err) {
      Logger.error(`workspaceCountChanged`, err);
    } finally {
      this.workspaceIndex = activeIndex;
      this.workspaceCount = wCount;
    }
  }
  workspaceChanged() {
    const newIndex = global.workspace_manager.get_active_workspace_index();
    const wCount = global.workspace_manager.get_n_workspaces();
    try {
      Logger.debug(`WorkspaceMgr.workspaceChanged()`, {
        newIndex: newIndex,
        currentWorkspaceIndex: this.workspaceIndex,
        currentWorkspaceCount: this.workspaceCount,
        newWorkspaceCount: wCount,
        addWorkspaceToLeft: this.addWorkspaceToLeft
      });
      if (this.workspaceIndex == newIndex) {
        Logger.debug("No workspace change");
        return;
      } else if (this.rd.isRecursive()) {
        this.initWorkspaces();
        return;
      }
      const offset = this.workspaceCount - wCount + (this.addWorkspaceToLeft ? -1 : 0);
      this.updatePositions(newIndex + offset, global.display.get_current_monitor());
    } catch (err) {
      Logger.error("workspaceChanged", err);
    } finally {
      this.workspaceIndex = newIndex;
      this.addWorkspaceToLeft = false;
    }
  }
  updatePositions(activeWorkspaceIndex, activeMonitorIndex) {
    this.db.windows.refresh();
    const diff = activeWorkspaceIndex - this.workspaceIndex;
    const repositioned = new Set;
    for (let monitorIndex = 0; monitorIndex < this.monitorCount; monitorIndex++) {
      if (monitorIndex != activeMonitorIndex) {
        for (let workspaceIndex = 0; workspaceIndex < this.workspaceCount; workspaceIndex++) {
          this.updateWindowPositions(repositioned, workspaceIndex, monitorIndex, diff);
        }
      }
    }
  }
  updateWindowPositions(repositioned, workspaceIndex, monitorIndex, diff) {
    const monitorWindows = this.db.windows.query({
      workspaceIndex: workspaceIndex,
      monitorIndex: monitorIndex
    });
    const newIndex = (this.workspaceCount + workspaceIndex + diff) % this.workspaceCount;
    const updateable = [];
    for (let uw of monitorWindows) {
      if (!repositioned.has(uw.id)) {
        repositioned.add(uw.id);
        if (uw.workspaceIndex() != newIndex && !uw.isOnAllWorkspaces()) {
          updateable.push(uw);
        }
      }
    }
    updateable.sort(((uw1, uw2) => uw2.stackPos - uw1.stackPos));
    for (let uw of updateable) {
      try {
        Logger.debug("Reposition window", {
          id: uw.id,
          title: uw.title(),
          monitor: monitorIndex,
          workspace: newIndex
        });
        uw.moveTo({
          workspaceIndex: newIndex
        });
      } catch (err) {
        Logger.error("Error", err);
        this.db.windows.remove(uw.id);
      }
    }
  }
  disable() {
    this.removeKeyBindings();
    this.rd = null;
  }
}
class Notifier {
  static notify(msg) {
    Main.notify("Serenity Desktop", msg);
  }
}
class UDIndicator {
  db;
  uuid;
  path;
  button;
  shortcuts;
  constructor(db, events, uuid, path) {
    this.db = db;
    this.uuid = uuid;
    this.path = path;
    events.addShortcutAddListener(((si, wi) => this.shortcutAdded(si, wi)));
    events.addShortcutRemoveListener((() => this.shortcutsChanged()));
    events.addTitleChangedListener((id => this.titleChanged(id)));
  }
  async shortcutAdded(shorcutIndex, windowId) {
    this.shortcutsChanged();
    const uw = this.db.windows.get(windowId);
    if (uw) {
      Notifier.notify(`Shortcut ${shorcutIndex} added: ${uw.title()}`);
    }
  }
  async shortcutsChanged() {
    if (this.shortcuts == null) {
      return;
    }
    try {
      for (let i = 0; i < 10; i++) {
        const shortcut = this.shortcuts[i];
        const uw = this.db.windows.get(this.db.shortcuts.getWindowIdByIndex(i));
        if (uw) {
          shortcut.button.reactive = true;
          shortcut.button.can_focus = true;
          shortcut.label.text = this.toTitle(uw.title());
        } else {
          shortcut.button.reactive = false;
          shortcut.button.can_focus = false;
          shortcut.label.text = `[No shortcut]`;
        }
      }
    } catch (err) {
      Logger.error("shortcutsChanged", err);
    }
  }
  toTitle(title) {
    return title.length <= 64 ? title : title.substring(0, 64) + "…";
  }
  titleChanged(windowId) {
    if (this.shortcuts == null) {
      return;
    }
    const index = this.db.shortcuts.getIndexByWindowId(windowId);
    const uw = this.db.windows.get(windowId);
    if (index != null && uw && this.shortcuts[index].label) {
      this.shortcuts[index].label.text = this.toTitle(uw.title());
    }
  }
  enable() {
    this.shortcuts = new Array(10);
    this.createStatusAreaButton();
    this.addShortcutsHeader();
    for (let i = 1; i <= 10; i++) {
      this.addShortcutItem(i % 10);
    }
    Main.panel.addToStatusArea(this.uuid, this.button);
    this.shortcutsChanged();
  }
  addMenuItem(item) {
    this.button.menu.addMenuItem(item);
  }
  createStatusAreaButton() {
    const iconPath = `${this.path}/icons/desktop-svgrepo-com.light.svg`;
    Gio.icon_new_for_string(iconPath);
    this.button = new PanelMenu.Button(0, "Serenity Desktop");
    this.button.add_child(new St.Icon({
      icon_name: "video-display-symbolic",
      style_class: "system-status-icon"
    }));
  }
  addShortcutsHeader() {
    let header = new PopupMenu.PopupBaseMenuItem({
      reactive: false,
      can_focus: false
    });
    let label = new St.Label({
      text: "Serenity Desktop - Shortcuts",
      x_align: Clutter.ActorAlign.START,
      style_class: "header-label"
    });
    header.add_child(label);
    this.addMenuItem(header);
  }
  addShortcutItem(index) {
    if (this.shortcuts == null) {
      return;
    }
    let menuItem = new PopupMenu.PopupBaseMenuItem({
      reactive: false,
      style_class: "menu-item"
    });
    const numberLabel = new St.Label({
      text: index.toString(),
      x_align: Clutter.ActorAlign.START,
      y_align: Clutter.ActorAlign.CENTER
    });
    this.shortcuts[index] = {
      label: new St.Label({
        text: `[No shortcut]`,
        x_align: Clutter.ActorAlign.START,
        x_expand: true,
        y_align: Clutter.ActorAlign.CENTER
      }),
      button: new St.Button({
        child: new St.Icon({
          icon_name: "window-close-symbolic",
          icon_size: 16
        }),
        style_class: "button button-remove",
        x_align: Clutter.ActorAlign.END,
        y_align: Clutter.ActorAlign.CENTER,
        reactive: false,
        can_focus: false
      })
    };
    this.shortcuts[index].button.connect("clicked", (() => {
      this.db.shortcuts.removeByIndex(index);
    }));
    let layout = new St.BoxLayout({
      vertical: false,
      x_expand: true,
      y_align: Clutter.ActorAlign.CENTER
    });
    layout.add_child(numberLabel);
    layout.add_child(new St.Widget({
      width: 15
    }));
    layout.add_child(this.shortcuts[index].label);
    layout.add_child(new St.Widget({
      width: 15
    }));
    layout.add_child(this.shortcuts[index].button);
    menuItem.add_child(layout);
    this.addMenuItem(menuItem);
  }
  disable() {
    try {
      if (this.button) {
        this.button.destroy();
      }
    } catch (err) {
      Logger.error("Failed to disable service", err);
    } finally {
      this.button = null;
      this.shortcuts = null;
    }
  }
}
class ShortcutsDB {
  events;
  static shortcuts=new Array(10);
  constructor(events) {
    this.events = events;
  }
  setShortcut(index, id) {
    if (index == null || id == null || ShortcutsDB.shortcuts[index] == id) {
      return;
    }
    if (ShortcutsDB.shortcuts[index] != null) {
      ShortcutsDB.shortcuts[index] = null;
    }
    for (let i = 0; i < ShortcutsDB.shortcuts.length; i++) {
      if (ShortcutsDB.shortcuts[i] == id) {
        ShortcutsDB.shortcuts[i] = null;
      }
    }
    ShortcutsDB.shortcuts[index] = id;
    this.events.notifyShortcutAdded(index, id);
  }
  removeByWindowId(id) {
    const index = this.getIndexByWindowId(id);
    if (index != null) {
      ShortcutsDB.shortcuts[index] = null;
      this.events.notifyShortcutRemoved(index, id);
    }
  }
  removeByIndex(index) {
    if (ShortcutsDB.shortcuts[index] != null) {
      ShortcutsDB.shortcuts[index] = null;
      this.events.notifyShortcutRemoved(index, ShortcutsDB.shortcuts[index]);
    }
  }
  getIndexByWindowId(id) {
    for (let i = 0; i < ShortcutsDB.shortcuts.length; i++) {
      if (ShortcutsDB.shortcuts[i] == id) {
        return i;
      }
    }
    return null;
  }
  getWindowIdByIndex(index) {
    return ShortcutsDB.shortcuts[index];
  }
}
class WindowsDB {
  nativeEvents;
  events;
  windows=new Map;
  metaWindows=new Map;
  constructor(nativeEvents, events) {
    this.nativeEvents = nativeEvents;
    this.events = events;
  }
  enable() {
    this.updateWindows();
    this.nativeEvents.connect(global.display, "window-created", ((_display, window) => this.onWindowCreate(window)));
    this.nativeEvents.connect(global.window_manager, "destroy", ((_shellwm, actor) => this.onWindowDestroy(actor.meta_window)));
  }
  onWindowCreate(mw) {
    this.updateWindows();
  }
  initWindow(id) {
    const mw = this.metaWindows.get(id);
    if (!mw) {
      return null;
    }
    const uw = new SWindow(id, this.getAppKey(mw), (() => this.metaWindows.get(id)));
    this.nativeEvents.connect(mw, "notify::title", (() => this.events.notifyTitleChanged(mw.get_id())));
    return uw;
  }
  refresh() {
    this.updateWindows();
  }
  updateWindows() {
    const mws = this.getMetaWindows();
    const ids = this.getIds();
    const positions = this.getWindowIdStack();
    for (let mw of mws) {
      const id = mw.get_id();
      this.metaWindows.set(id, mw);
      ids.delete(id);
      const uw = this.get(id);
      uw.stackPos = positions.get(id) ?? -1;
    }
    ids.forEach((id => this.windows.delete(id)));
  }
  onWindowDestroy(mw) {
    this.updateWindows();
    this.nativeEvents.disconnectTarget(mw);
    this.events.notifyWindowDestroyed(mw.get_id());
  }
  clear() {
    this.windows.clear();
  }
  getIds() {
    return new Set(this.windows.keys());
  }
  get(id) {
    if (id == null) {
      return null;
    }
    if (!this.windows.has(id)) {
      const uw = this.initWindow(id);
      if (uw) {
        this.windows.set(id, uw);
      }
    }
    return this.windows.get(id);
  }
  exists(id) {
    return this.windows.has(id);
  }
  remove(id) {
    this.windows.delete(id);
  }
  getAppKey(mw) {
    let app = Shell.WindowTracker.get_default().get_window_app(mw);
    return `${app.get_name()}.${app.get_id()}`;
  }
  query(query) {
    return [ ...this.windows.values() ].filter((uw => {
      if (query.monitorIndex != null && query.workspaceIndex != null) {
        return uw.monitorIndex() == query.monitorIndex && uw.workspaceIndex() == query.workspaceIndex;
      } else if (query.monitorIndex != null) {
        return uw.monitorIndex() == query.monitorIndex;
      } else if (query.monitorIndex != null) {
        return uw.workspaceIndex() == query.workspaceIndex;
      }
      return true;
    }));
  }
  getMetaWindows() {
    return global.get_window_actors().map((w => w.metaWindow)).filter((w => w.get_window_type() == Meta.WindowType.NORMAL && !w.is_skip_taskbar()));
  }
  getWindowIdStack() {
    const ids = new Map;
    const list = global.display.get_tab_list(Meta.TabList.NORMAL_ALL, null);
    for (let c = 0; c < list.length; c++) {
      ids.set(list[c].get_id(), c);
    }
    return ids;
  }
}
class Db {
  windows;
  shortcuts;
  constructor(nativeEvents, events) {
    this.shortcuts = new ShortcutsDB(events);
    this.windows = new WindowsDB(nativeEvents, events);
  }
  enable() {
    this.windows.enable();
  }
  disable() {
    this.windows.clear();
  }
}
class NativeEventHandler {
  handlers=new Map;
  connect(target, event, handler) {
    if (target == null) {
      Logger.info("Target does not exist for signal " + event);
    }
    if (!this.handlers.has(target)) {
      this.handlers.set(target, new Map);
    }
    const oldId = this.handlers.get(target).get(event);
    if (oldId) {
      target.disconnect(oldId);
    }
    this.handlers.get(target).set(event, target.connect(event, ((...args) => {
      try {
        return handler(...args);
      } catch (err) {
        Logger.error(`Error on native event '${event}'`, err);
      }
    })));
  }
  disconnectTarget(target) {
    if (this.handlers.has(target)) {
      for (let id of this.handlers.get(target).values()) {
        target.disconnect(id);
      }
    }
    this.handlers.delete(target);
  }
  enable() {}
  disable() {
    for (let target of [ ...this.handlers.keys() ]) {
      this.disconnectTarget(target);
    }
  }
}
class Events {
  shortcutAddListeners=[];
  shortcutRemoveListeners=[];
  titleChangedListeners=[];
  windowDestroyedListeners=[];
  addTitleChangedListener(listener) {
    this.titleChangedListeners.push(listener);
  }
  addWindowDestroyedListener(listener) {
    this.windowDestroyedListeners.push(listener);
  }
  addShortcutAddListener(listener) {
    this.shortcutAddListeners.push(listener);
  }
  addShortcutRemoveListener(listener) {
    this.shortcutRemoveListeners.push(listener);
  }
  notifyShortcutAdded(index, windowId) {
    this.shortcutAddListeners.forEach((l => {
      try {
        l(index, windowId);
      } catch (err) {
        Logger.error("Error on notifyShortcutAdded", err);
      }
    }));
  }
  notifyShortcutRemoved(index, windowId) {
    this.shortcutRemoveListeners.forEach((l => {
      try {
        l(index, windowId);
      } catch (err) {
        Logger.error("Error on notifyShortcutRemoved", err);
      }
    }));
  }
  notifyTitleChanged(windowId) {
    this.titleChangedListeners.forEach((l => {
      try {
        l(windowId);
      } catch (err) {
        Logger.error("Error on notifyTitleChanged", err);
      }
    }));
  }
  notifyWindowDestroyed(windowId) {
    this.windowDestroyedListeners.forEach((l => {
      try {
        l(windowId);
      } catch (err) {
        Logger.error("Error on notifyWindowDestroyed", err);
      }
    }));
  }
}
class SerenityDesktopExtension extends Extension {
  db;
  indicator;
  workspaceMgr;
  originalShouldAnimate;
  nativeEvents;
  enable() {
    try {
      Logger.isDebug = false;
      const events = new Events;
      this.nativeEvents = new NativeEventHandler;
      this.db = new Db(this.nativeEvents, events);
      this.db.windows.enable();
      this.indicator = new UDIndicator(this.db, events, this.uuid, this.path);
      this.workspaceMgr = new WorkspaceMgr(this.db, this.nativeEvents, events, this.getSettings());
      this.workspaceMgr.enable();
      this.indicator.enable();
      this.disableSwitchAnimations();
    } catch (err) {
      Logger.error("Failed to enable extension", err);
      this.disable();
    }
  }
  disable() {
    try {
      this.disableService(this.nativeEvents);
      this.disableService(this.db);
      this.disableService(this.workspaceMgr);
      this.disableService(this.indicator);
      this.restoreSwitchAnimations();
    } catch (err) {
      Logger.error("Failed to disable extension", err);
    } finally {
      this.db = null;
      this.workspaceMgr = null;
      this.indicator = null;
      this.originalShouldAnimate = null;
    }
  }
  disableSwitchAnimations() {
    if (Main.wm["_shouldAnimate"]) {
      this.originalShouldAnimate = Main.wm["_shouldAnimate"];
      Main.wm["_shouldAnimate"] = function() {
        return false;
      };
    }
  }
  restoreSwitchAnimations() {
    if (Main.wm["_shouldAnimate"] && this.originalShouldAnimate) {
      Main.wm["_shouldAnimate"] = this.originalShouldAnimate;
    }
  }
  disableService(service) {
    if (service) {
      try {
        service.disable();
      } catch (err) {
        Logger.error("Failed to disable local service", err);
      }
    }
  }
  getTheme() {
    const schema = Gio.Settings.new("org.gnome.desktop.interface");
    return schema.get_string("gtk-theme").includes("dark") ? Theme.DARK : Theme.LIGHT;
  }
}
export { SerenityDesktopExtension as default };
