/*
  To reviewers of this code:
  
  I highly recommend starting your code review with the [PatchManager] class since it is used all
  around the extension – briefly, it allows to define a nested tree of [PatchManager] instances that
  each are responsible for applying and cleaning up patches to Gnome Shell code. A patch can be
  anything, for example overwriting a method in some prototype, connecting to a signal or adding an
  actor to the shell. [PatchManager] allows to easily define a patch and the necessary cleanup
  in-place, which facilitates a good overview and mitigates the risk of forgetting to clean something
  up/to unapply a patch. When a parent [PatchManager] is destroyed or disabled, all its children are
  automatically destroyed/disabled too.
  
  Next, it's probably best to start with the main [GnomeTouchExtension] class since this is where
  the root PatchManager is defined – all other [PatchManager]s are children of this [PatchManager]
  and are automatically destroyed/unapplied when the root [PatchManager] is.
  
  Thanks for the time you put into this!
*/

import {
  AssetIcon,
  Delay,
  __reExport,
  assert,
  assetsGResourceFile,
  clamp,
  debugLog,
  filterObject,
  findActorBy,
  initSettings,
  log as log2,
  randomChoice,
  settings,
  uninitSettings
} from "./chunk-TSRMGQWY.js";

// src/utils/ui/widgets.ts
import St from "gi://St";
import GObject from "gi://GObject";
import Clutter from "gi://Clutter";
var Widgets;
((Widgets2) => {
  class Ref4 {
    _destroySignalId;
    /**
     * Create a reference with an optional initial value.
     */
    constructor(initialValue) {
      this.set(initialValue ?? null);
    }
    /**
     * Get the actor the reference points to, or `null` if the actor has been
     * destroyed or unset.
     */
    get current() {
      return this._value;
    }
    /**
     * Update the reference to point to the given actor, unset the reference if
     * `null` is passed.
     */
    set(value) {
      if (this._destroySignalId !== void 0 && this._value) {
        this._value.disconnect(this._destroySignalId);
      }
      this._value = value;
      this._destroySignalId = value?.connect("destroy", () => this.set(null));
    }
    /**
     * Convenience method to call the given function or closure on the referenced
     * actor only if there is a referenced actor at the moment.
     *
     * Example:
     * ```typescript
     * const myRef = new Ref(myWidget);
     *
     * // Set the widget's opacity only if it has not been destroyed or in another way unset yet:
     * myRef.apply(w => w.opacity = 0.8);
     * ```
     */
    apply(fn) {
      if (this.current) {
        fn(this.current);
      }
    }
    _value = null;
  }
  Widgets2.Ref = Ref4;
  function filterConfig(config, filterOut) {
    filterOut ??= [
      "ref",
      "children",
      "child",
      "onCreated",
      "constraints",
      /^(on|notify)[A-Z]/
    ];
    return filterObject(
      config,
      //@ts-ignore
      (entry) => typeof entry[0] === "string" && !filterOut.some((filter) => filter instanceof RegExp ? filter.test(entry[0]) : filter === entry[0])
    );
  }
  function initWidget(w, props) {
    if (props.ref)
      props.ref.set(w);
    props.constraints?.forEach((c) => w.add_constraint(c));
    for (const [key, value] of Object.entries(props)) {
      if (/^(on|notify)[A-Z]/.test(key) && typeof value === "function" && key !== "onCreated") {
        const signalName = key.replace(/^on/, "").replace(/^notify/, "notify::").replace(/(\w)([A-Z])/g, "$1-$2").toLowerCase();
        w.connect(signalName, value);
      }
    }
    const onCreatedRes = props.onCreated?.(w);
    if (onCreatedRes)
      w.connect("destroy", onCreatedRes);
  }
  class Button extends St.Button {
    static {
      GObject.registerClass(this);
    }
    constructor(config) {
      super(filterConfig(config));
      initWidget(this, filterConfig(config, config.onLongPress ? ["onLongPress", "onClicked"] : []));
      if (config.onLongPress) {
        this._setupLongPress(config.onLongPress, config.onClicked);
      }
      if (config.child)
        this.child = config.child;
    }
    // A simple long press implementation, that is triggered after holding the button for 500ms
    // and cancelled when moving up earlier or when moving the finger too much.
    _setupLongPress(onLongPress, onClicked) {
      const pressEvents = [Clutter.EventType.TOUCH_BEGIN, Clutter.EventType.BUTTON_PRESS, Clutter.EventType.PAD_BUTTON_PRESS];
      const releaseEvents = [Clutter.EventType.TOUCH_END, Clutter.EventType.BUTTON_RELEASE, Clutter.EventType.PAD_BUTTON_RELEASE];
      const cancelEvents = [Clutter.EventType.TOUCH_CANCEL, Clutter.EventType.LEAVE];
      let downAt;
      const handleEvent = (_, evt) => {
        if (pressEvents.includes(evt.type())) {
          let thisDownAt = downAt = { t: evt.get_time(), x: evt.get_coords()[0], y: evt.get_coords()[1] };
          Delay.ms(500).then(() => {
            if (this.pressed && downAt?.t === thisDownAt.t && downAt?.x === thisDownAt.x && downAt?.y === thisDownAt.y) {
              onLongPress(this);
              downAt = void 0;
            }
          });
        } else if (releaseEvents.includes(evt.type()) && downAt) {
          if (evt.get_time() - downAt.t < 500)
            onClicked?.(this);
          downAt = void 0;
        } else if (cancelEvents.includes(evt.type())) {
          downAt = void 0;
        } else if (evt.type() == Clutter.EventType.TOUCH_UPDATE && downAt) {
          let dist = Math.sqrt((evt.get_coords()[0] - downAt.x) ** 2 + (evt.get_coords()[1] - downAt.y) ** 2);
          if (dist > 15 * St.ThemeContext.get_for_stage(global.stage).scaleFactor) {
            downAt = void 0;
          }
        }
      };
      this.connect("touch-event", handleEvent);
      this.connect("button-press-event", handleEvent);
      this.connect("button-release-event", handleEvent);
      this.connect("leave-event", handleEvent);
    }
  }
  Widgets2.Button = Button;
  class Icon extends St.Icon {
    static {
      GObject.registerClass(this);
    }
    constructor(config) {
      super(filterConfig(config));
      initWidget(this, config);
    }
  }
  Widgets2.Icon = Icon;
  class Label extends St.Label {
    static {
      GObject.registerClass(this);
    }
    constructor(config) {
      super(filterConfig(config));
      initWidget(this, config);
    }
  }
  Widgets2.Label = Label;
  class Bin extends St.Bin {
    static {
      GObject.registerClass(this);
    }
    constructor(config) {
      super(filterConfig(config));
      initWidget(this, config);
      if (config.child)
        this.set_child(config.child);
    }
  }
  Widgets2.Bin = Bin;
  class Box extends St.BoxLayout {
    static {
      GObject.registerClass(this);
    }
    constructor(config) {
      super(filterConfig(config));
      initWidget(this, config);
      config.children?.forEach((c) => this.add_child(c));
    }
  }
  Widgets2.Box = Box;
  class Row extends St.BoxLayout {
    static {
      GObject.registerClass(this);
    }
    constructor(config) {
      super({
        ...filterConfig(config),
        vertical: false
      });
      initWidget(this, config);
      config.children?.forEach((c) => this.add_child(c));
    }
  }
  Widgets2.Row = Row;
  class Column extends St.BoxLayout {
    static {
      GObject.registerClass(this);
    }
    constructor(config) {
      super({
        ...filterConfig(config),
        vertical: true
      });
      initWidget(this, config);
      config.children?.forEach((c) => this.add_child(c));
    }
  }
  Widgets2.Column = Column;
  class ScrollView extends St.ScrollView {
    static {
      GObject.registerClass(this);
    }
    constructor(config) {
      super({
        ...filterConfig(config)
      });
      initWidget(this, config);
      if (config.child) {
        if ("vadjustment" in config.child) {
          this.set_child(config.child);
        } else {
          const s = new St.BoxLayout();
          s.add_child(config.child);
          this.set_child(s);
        }
      }
    }
  }
  Widgets2.ScrollView = ScrollView;
})(Widgets || (Widgets = {}));

// src/utils/patchManager.ts
import GObject2 from "gi://GObject";
import { InjectionManager } from "resource:///org/gnome/shell/extensions/extension.js";
var Ref = Widgets.Ref;
var PatchManager = class _PatchManager {
  debugName;
  _parent;
  _children = [];
  _injectionManager = new InjectionManager();
  _patches = [];
  _isDestroyed = false;
  constructor(debugName) {
    this.debugName = debugName;
  }
  /**
   * Apply a patch. The callback peforming the patch is called immediately and must
   * return another function to undo the patch again.
  */
  patch(func, debugName) {
    const patch = this.registerPatch(func, debugName);
    patch.enable();
    return patch;
  }
  /**
   * Add a patch without automatically applying it. Otherwise, same
   * as [PatchManager.patch(...)]
   */
  registerPatch(func, debugName) {
    this._patches.push(new Patch({
      enable: func,
      debugName: this._generatePatchDebugName(debugName)
    }));
    return this._patches.at(-1);
  }
  /**
   * Automatically destroy any object with a [destroy] method when the [PatchManager] is
   * disabled or destroyed.
   */
  autoDestroy(instance, debugName) {
    this.patch(() => {
      let ref = new Ref(instance);
      return () => ref.current?.destroy();
    }, debugName ?? `autoDestroy:${instance.constructor.name}`);
    return instance;
  }
  /**
   * Connect to a signal from any GObject/widget and automatically disconnect when the [PatchManager]
   * is disabled or destroyed.
   */
  connectTo(instance, signal, handler, debugName) {
    return this.patch(() => {
      const signalId = instance.connect(signal, handler);
      return () => instance.disconnect(signalId);
    }, debugName ?? `connectTo(${instance.constructor.name}:${signal})`);
  }
  patchSignalHandler(instance, signalId, handler, debugName) {
    if (Array.isArray(signalId)) {
      return new MultiPatch({
        patches: signalId.map((s) => this.patchSignalHandler(
          instance,
          s,
          handler,
          `${debugName}#signal(${instance.constructor.name}:${signalId})`
        )),
        debugName: this._generatePatchDebugName(debugName)
      });
    } else {
      return this.patch(() => {
        const originalHandler = GObject2.signal_handler_find(instance, { signalId });
        GObject2.signal_handler_block(instance, originalHandler);
        const newHandler = instance.connect(signalId, handler);
        return () => {
          GObject2.signal_handler_disconnect(instance, newHandler);
          GObject2.signal_handler_unblock(instance, originalHandler);
        };
      }, debugName);
    }
  }
  patchMethod(prototype, methodName, method, debugName) {
    if (Array.isArray(methodName)) {
      return new MultiPatch({
        patches: methodName.map((m) => this.patchMethod(
          prototype,
          m,
          method,
          `${debugName}#method(${prototype.constructor.name}:${m})`
        )),
        debugName: this._generatePatchDebugName(debugName)
      });
    } else {
      return this.patch(() => {
        this._injectionManager.overrideMethod(prototype, methodName, (orig) => {
          return function(...args) {
            return method.call(this, orig.bind(this), ...args);
          };
        });
        return () => this._injectionManager.restoreMethod(prototype, methodName);
      }, debugName);
    }
  }
  appendToMethod(prototype, methodName, method, debugName) {
    if (Array.isArray(methodName)) {
      return new MultiPatch({
        patches: methodName.map((m) => this.appendToMethod(
          prototype,
          m,
          method,
          `${debugName}#append-to-method(${prototype.constructor.name}:${m})`
        )),
        debugName: this._generatePatchDebugName(debugName)
      });
    } else {
      return this.patchMethod(prototype, methodName, function(orig, ...args) {
        const res = orig.call(this, ...args);
        method.call(this, ...args);
        return res;
      }, debugName);
    }
  }
  /**
   * Undo and delete all patches made so far.
   *
   * This function should only be called if the [PatchManager] is not going to be used anymore.
   */
  destroy() {
    if (this._isDestroyed)
      return;
    debugLog(`Destroying PatchManager ${this.debugName ?? ""}`.trim());
    if (this._parent?._children.includes(this)) {
      this._parent?._children.splice(this._parent._children.indexOf(this), 1);
    }
    while (this._children.length > 0) {
      this._children.pop().destroy();
    }
    this._patches.toReversed().forEach((p) => p.disable());
    this._patches = [];
    this._isDestroyed = true;
  }
  /**
   * Undo all patches made so far, but keep them in store for a potential call to [enable]
   */
  disable() {
    if (this._isDestroyed)
      return;
    debugLog(`Disabling PatchManager ${this.debugName ?? ""}`.trim());
    this._children.toReversed().forEach((c) => c.disable());
    this._patches.toReversed().forEach((p) => p.disable());
  }
  /**
   * Enable all disabled patches.
   */
  enable() {
    debugLog(`Enabling PatchManager ${this.debugName ?? ""}`.trim());
    this._patches.forEach((p) => p.enable());
    this._children.forEach((c) => c.enable());
  }
  /**
   * Create a descendent [PatchManager].
   *
   * This child [PatchManager] will react to any call to [destroy], [disable] and [enable]
   * on any parent [PatchManager] and will forward those calls to its own descendents, should
   * it be forked again. This allows for a nice, tree structure and a consistent interface
   * for managing patches.
   * @param debugName An optional label used for debug log messages
   */
  fork(debugName) {
    const instance = new _PatchManager(
      this.debugName ? `${this.debugName}/${debugName ?? this._children.length + 1}` : debugName
    );
    instance._parent = this;
    this._children.push(instance);
    return instance;
  }
  _patchNameCounter = 0;
  _generatePatchDebugName(debugName) {
    return `${this.debugName}:${debugName ?? `#${this._patchNameCounter++}`}`;
  }
};
var Patch = class {
  debugName;
  _enableCallback;
  _disableCallback;
  _isEnabled = false;
  constructor(props) {
    this._enableCallback = props.enable;
    this.debugName = props.debugName ?? null;
  }
  disable(force = false) {
    if (!force && !this.isEnabled)
      return;
    debugLog(` - Disabling patch ${this.debugName}`);
    this._disableCallback?.call(this);
    this._isEnabled = false;
  }
  enable(force = false) {
    if (!force && this.isEnabled)
      return;
    debugLog(` - Enabling patch ${this.debugName}`);
    this._disableCallback = this._enableCallback();
    this._isEnabled = true;
  }
  get isEnabled() {
    return this._isEnabled;
  }
};
var MultiPatch = class extends Patch {
  _patches;
  constructor(props) {
    super({
      enable: () => {
        props.patches.forEach((p) => p.enable());
        return () => props.patches.forEach((p) => p.disable());
      },
      debugName: props.debugName
    });
    this._patches = props.patches;
  }
  get isEnabled() {
    return this._patches.every((p) => p.isEnabled);
  }
  enable(force = false) {
    this._patches.forEach((p) => p.enable(force));
  }
  disable(force = false) {
    this._patches.forEach((p) => p.disable(force));
  }
};

// src/utils/extensionFeature.ts
var ExtensionFeature = class {
  pm;
  constructor(patchManager) {
    this.pm = patchManager;
  }
  destroy() {
    this.pm.destroy();
  }
};

// src/utils/intervalRunner.ts
import GLib from "gi://GLib";
var IntervalRunner = class {
  callback;
  timeoutId = null;
  _interval;
  _priority;
  constructor(interval, callback, priority = GLib.PRIORITY_DEFAULT) {
    this.callback = callback;
    this._interval = interval;
    this._priority = priority;
  }
  /**
   * Start the interval runner or restart it if it is running already.
   */
  start() {
    this.stop();
    const tid = GLib.timeout_add(this._priority, this._interval, () => {
      this.callback(this.stop.bind(this));
      return this.timeoutId === tid ? GLib.SOURCE_CONTINUE : GLib.SOURCE_REMOVE;
    });
    this.timeoutId = tid;
  }
  /**
   * Stop the interval. Can be resumed using `start()`.
   */
  stop() {
    if (this.timeoutId !== null) {
      GLib.source_remove(this.timeoutId);
      this.timeoutId = null;
    }
  }
  /**
   * Declarative way of starting/stopping the interval runner.
   *
   * Calling this with `active=true` while the timeout is running or with `active=false`
   * while it is not running is a no-op.
   */
  setActive(active) {
    if (!active && this.timeoutId !== null) {
      this.stop();
    } else if (active && this.timeoutId === null) {
      this.start();
    }
  }
  /**
   * Run the callback once after the given delay (unless `stop()` is called before that)
   */
  scheduleOnce(delayMs = 0) {
    GLib.timeout_add(this._priority, delayMs, () => {
      if (this.timeoutId != null) {
        this.callback(this.stop.bind(this));
      }
      return GLib.SOURCE_REMOVE;
    });
  }
  /**
   * Change the interval. Restarts the callback automatically if a different interval than the
   * previous one is given.
   */
  setInterval(interval) {
    if (interval !== this._interval) {
      this._interval = interval;
      this.start();
    }
  }
  /**
   * Change the callback priority. Restarts the callback automatically if a different priority than
   * the previous one is given.
   */
  setPriority(priority) {
    if (priority != this._priority) {
      this._priority = priority;
      this.start();
    }
  }
  get priority() {
    return this._priority;
  }
  get interval() {
    return this._interval;
  }
};

// src/utils/ui/gestureRecognizer2D.ts
import Clutter3 from "gi://Clutter";
import St2 from "gi://St";
var GestureRecognizer2D = class _GestureRecognizer2D {
  /**
   * A sufficient time with movements less than this amount of (logical) pixels may lead to a hold
   * pattern being recognized.
   */
  static PAUSE_TOLERANCE = 12;
  // delta_pixels
  /**
   * Swipe patterns with less than this distance will be merged with the patterns following them.
   * Should the only recognized pattern have less movement than this, [isTab] will return true.
   */
  static MIN_SWIPE_DISTANCE = 10;
  // delta_pixels
  /**
   * The number of milliseconds no (actually: less than [PAUSE_TOLERANCE]) movements needs to take
   * place for a hold pattern to be recognized.
   */
  static SIGNIFICANT_PAUSE = 500;
  // milliseconds
  scaleFactor = St2.ThemeContext.get_for_stage(global.stage).scaleFactor;
  recordedPatterns = [];
  lastEvent = null;
  lastAngle = -1;
  currentStrokeDx = 0;
  currentStrokeDy = 0;
  currentStrokeDt = 0;
  totalDx = 0;
  totalDy = 0;
  pauseTime = 0;
  pauseDx = 0;
  pauseDy = 0;
  _isDuringGesture = false;
  /**
   * Push an event to the recognizer for processing.
   */
  pushEvent(event) {
    if (event.type() == Clutter3.EventType.TOUCH_BEGIN || event.type() == Clutter3.EventType.BUTTON_PRESS || event.type() == Clutter3.EventType.PAD_BUTTON_PRESS) {
      this.resetAndStartGesture();
    }
    if (this._isDuringGesture) {
      this.processEvent(event);
      this.lastEvent = event;
    }
    if (event.type() == Clutter3.EventType.TOUCH_END || event.type() == Clutter3.EventType.TOUCH_CANCEL || event.type() == Clutter3.EventType.BUTTON_RELEASE || event.type() == Clutter3.EventType.PAD_BUTTON_RELEASE) {
      this._isDuringGesture = false;
    }
  }
  /**
   * Manually clear the recognizers state and notify it that a new gesture has started.
   *
   * This function only needs to be called in contexts where no sequence-start
   * events, such as touch down or button press are pushed to the recognizer. This
   * is normally not the case, but for example within a Clutter.Gesture the sequence
   * start events are not available.
   */
  resetAndStartGesture() {
    this.recordedPatterns = [];
    this.lastEvent = null;
    this.lastAngle = -1;
    this.currentStrokeDx = 0;
    this.currentStrokeDy = 0;
    this.currentStrokeDt = 0;
    this.totalDx = 0;
    this.totalDy = 0;
    this.pauseTime = 0;
    this.pauseDx = 0;
    this.pauseDy = 0;
    this._isDuringGesture = true;
  }
  /**
   * True if a gesture is currently being performed (i.e. a finger
   * is down or a pointer button clicked).
   */
  get isDuringGesture() {
    return this._isDuringGesture;
  }
  /**
   * Returns true if the last gesture was a single tap gesture.
   *
   * Should only be called after a gesture is complete.
   */
  wasTap(allowLongTap = false) {
    assert(!this.isDuringGesture, "You should call `isTap` only when a gesture is completed, not during a gesture.");
    return (
      // If no pattern has been recognized, this is a tap:
      this.recordedPatterns.length === 0 || this.recordedPatterns.length === 1 && this.recordedPatterns[0].type == "swipe" && this.recordedPatterns[0].swipeDistance < _GestureRecognizer2D.MIN_SWIPE_DISTANCE * this.scaleFactor || allowLongTap && this.wasLongTap()
    );
  }
  /**
   * Returns true if the last gesture was a long-tap gesture (default: longer than [SIGNIFICANT_PAUSE]).
   *
   * Should only be called after a gesture is complete.
   */
  wasLongTap(minDuration) {
    assert(!this.isDuringGesture, "You should call `isLongTap` only when a gesture is completed, not during a gesture.");
    return this.recordedPatterns.length === 1 && this.recordedPatterns[0].type === "hold" && this.recordedPatterns[0].duration > (minDuration ?? _GestureRecognizer2D.SIGNIFICANT_PAUSE);
  }
  /**
   * Get the patterns that where recognized during the gesture, in
   * chronological order.
   */
  getPatterns() {
    return [...this.recordedPatterns];
  }
  /**
   * The current total motion delta, i.e. the offset between the
   * current pointer/finger location and the starting point of the
   * gesture.
   */
  get totalMotionDelta() {
    return { x: this.totalDx, y: this.totalDy };
  }
  /**
   * The first pattern recorded so far.
   */
  get primaryPattern() {
    return this.recordedPatterns.at(0) ?? null;
  }
  /**
   * The pattern of the first stroke, i.e. the first `SwipePattern`. This
   * can already be retrieved for incomplete strokes, but it's properties
   * might change.
   *
   * If the primary move cannot (yet) be determined, `null` is returned.
   */
  get primaryMove() {
    let pattern = this.recordedPatterns.find((p) => p.type == "swipe");
    return pattern ? { ...pattern } : null;
  }
  /**
   * The pattern of the last stroke, i.e. the last `SwipePattern`. This
   * can already be retrieved for incomplete strokes, but it's properties
   * might change.
   *
   * If the primary move cannot (yet) be determined, `null` is returned.
   */
  get secondaryMove() {
    let pattern = this.recordedPatterns.findLast((p) => p.type == "swipe");
    return pattern ? { ...pattern } : null;
  }
  /**
   * Get whether the current event sequence is a touch event sequence
   * or a pointer sequence.
   */
  get isTouchGesture() {
    switch (this.lastEvent?.type()) {
      case Clutter3.EventType.TOUCH_BEGIN:
      case Clutter3.EventType.TOUCH_UPDATE:
      case Clutter3.EventType.TOUCH_CANCEL:
      case Clutter3.EventType.TOUCH_END:
        return true;
      default:
        return false;
    }
  }
  /**
   * Get whether the currently performed gesture is already certain to not be
   * a tap or long tap.
   */
  get isCertainlyMovement() {
    const d = Math.sqrt(this.totalMotionDelta.x ** 2 + this.totalMotionDelta.y ** 2);
    return d >= _GestureRecognizer2D.MIN_SWIPE_DISTANCE * this.scaleFactor;
  }
  /**
   * true, if the last pushed event was a sequence-end event, e.g. touch end
   * or button release.
   */
  get gestureHasJustFinished() {
    return this.lastEvent?.type() == Clutter3.EventType.TOUCH_END || this.lastEvent?.type() == Clutter3.EventType.TOUCH_CANCEL || this.lastEvent?.type() == Clutter3.EventType.BUTTON_RELEASE || this.lastEvent?.type() == Clutter3.EventType.PAD_BUTTON_RELEASE;
  }
  processEvent(event) {
    if (this.lastEvent !== null) {
      const dx = event.get_coords()[0] - this.lastEvent.get_coords()[0], dy = event.get_coords()[1] - this.lastEvent.get_coords()[1], dt = event.get_time() - this.lastEvent.get_time();
      const d = Math.sqrt(dx ** 2 + dy ** 2);
      if (d === 0 && dt === 0)
        return;
      if (this.checkForSignificantPause(dt, dx, dy, ...event.get_coords()))
        return;
      if (this.pauseTime === 0 && d > _GestureRecognizer2D.PAUSE_TOLERANCE * this.scaleFactor) {
        if (this.checkForSignificantAngleChange(dx, dy, dt))
          return;
      }
      this.currentStrokeDx += dx;
      this.currentStrokeDy += dy;
      this.currentStrokeDt += dt;
      this.totalDx += dx;
      this.totalDy += dy;
      if (this.currentStrokeDx == 0 && this.currentStrokeDy == 0)
        return;
      const distance = Math.sqrt(this.currentStrokeDx ** 2 + this.currentStrokeDy ** 2);
      const angle = this.angleBetween(this.currentStrokeDx, this.currentStrokeDy);
      const direction = this.directionForAngle(angle);
      this.pushPattern({
        type: "swipe",
        deltaX: this.currentStrokeDx,
        deltaY: this.currentStrokeDy,
        swipeDistance: distance,
        swipeAngle: angle,
        swipeSpeed: distance / this.currentStrokeDt,
        swipeDirection: direction,
        swipeAxis: this.axisForDirection(direction),
        totalTime: this.currentStrokeDt
      });
    }
  }
  checkForSignificantAngleChange(dx, dy, dt) {
    const lastAngle = this.lastAngle;
    const currentAngle = this.angleBetween(dx, dy);
    this.lastAngle = currentAngle;
    if (lastAngle !== -1) {
      const angleDiff = currentAngle - lastAngle;
      if (this.directionForAngle(currentAngle) != this.directionForAngle(lastAngle)) {
        this.currentStrokeDx = dx;
        this.currentStrokeDy = dy;
        this.currentStrokeDt = dt;
        return true;
      }
    }
    return false;
  }
  checkForSignificantPause(dt, dx, dy, x, y) {
    if (Math.sqrt(this.pauseDx ** 2 + this.pauseDy ** 2) >= _GestureRecognizer2D.PAUSE_TOLERANCE * this.scaleFactor) {
      this.pauseTime = this.pauseDx = this.pauseDy = 0;
    } else {
      this.pauseTime += dt;
      this.pauseDx += dx;
      this.pauseDy += dy;
    }
    if (this.pauseTime >= _GestureRecognizer2D.SIGNIFICANT_PAUSE) {
      this.currentStrokeDx = this.currentStrokeDy = this.currentStrokeDt = 0;
      this.pushPattern({
        type: "hold",
        x,
        y,
        duration: this.pauseTime
      });
      this.pauseTime = this.pauseDx = this.pauseDy = 0;
      return true;
    }
    return false;
  }
  pushPattern(pattern) {
    let ignorePattern = false;
    let lastPattern = this.recordedPatterns.at(-1);
    if (pattern.type == "hold") {
      while ((lastPattern = this.recordedPatterns.at(-1)) && lastPattern.type == "swipe" && lastPattern.swipeDistance < _GestureRecognizer2D.PAUSE_TOLERANCE) {
        pattern.duration += lastPattern.totalTime;
        this.recordedPatterns.pop();
      }
      if (lastPattern?.type == "hold") {
        lastPattern.duration += pattern.duration;
        ignorePattern = true;
      }
    } else if (pattern.type == "swipe") {
      while ((lastPattern = this.recordedPatterns.at(-1)) && lastPattern.type == "swipe" && lastPattern.swipeDistance < _GestureRecognizer2D.MIN_SWIPE_DISTANCE * this.scaleFactor) {
        pattern.deltaX += lastPattern.deltaX;
        pattern.deltaY += lastPattern.deltaY;
        pattern.totalTime += lastPattern.totalTime;
        pattern.swipeDistance += Math.sqrt(pattern.deltaX ** 2 + pattern.deltaY ** 2);
        pattern.swipeSpeed = pattern.swipeDistance / pattern.totalTime;
        pattern.swipeAngle = this.angleBetween(pattern.deltaX, pattern.deltaY);
        pattern.swipeDirection = this.directionForAngle(pattern.swipeAngle);
        this.recordedPatterns.pop();
      }
      if (lastPattern?.type == "swipe" && lastPattern.swipeDirection === pattern.swipeDirection) {
        lastPattern.deltaX = pattern.deltaX;
        lastPattern.deltaY = pattern.deltaY;
        lastPattern.totalTime = pattern.totalTime;
        lastPattern.swipeDistance = Math.sqrt(lastPattern.deltaX ** 2 + lastPattern.deltaY ** 2);
        lastPattern.swipeSpeed = lastPattern.swipeDistance / lastPattern.totalTime;
        lastPattern.swipeAngle = this.angleBetween(lastPattern.deltaX, lastPattern.deltaY);
        ignorePattern = true;
      }
    }
    if (!ignorePattern) {
      this.recordedPatterns.push(pattern);
    }
  }
  // up = 0, right = 90, down = 180, left = 270
  angleBetween(dx, dy) {
    return (Math.atan2(dy, dx) * 180 / Math.PI + 450) % 360;
  }
  directionForAngle(angle) {
    if (315 <= angle || angle <= 45) {
      return "up";
    } else if (45 <= angle && angle <= 135) {
      return "right";
    } else if (135 <= angle && angle <= 225) {
      return "down";
    } else {
      return "left";
    }
  }
  axisForDirection(direction) {
    if (direction === "up" || direction === "down") {
      return "vertical";
    }
    return "horizontal";
  }
  /**
   * Returns a human-readable representation of the recorded patterns
   */
  toString() {
    let s = [];
    for (let p of this.recordedPatterns) {
      switch (p.type) {
        case "hold":
          s.push(`hold ${(p.duration / 1e3).toFixed(2)}s`);
          break;
        case "swipe":
          s.push(`swipe ${p.swipeDirection} (${Math.round(p.swipeAngle)}\xB0, ${Math.round(p.swipeDistance)}px)`);
      }
    }
    return `<${this.constructor.name} (gesture ${this.isDuringGesture ? "ongoing" : "completed"}${this.wasLongTap() ? ", is-long-tap" : this.wasTap() ? ", is-tap" : ""}) patterns: [ ${s.join(" \u2022 ")} ]>`;
  }
};

// src/features/navigationBar/navigationBarGestureTracker.ts
import GObject3 from "gi://GObject";
import Clutter4 from "gi://Clutter";
import * as Main from "resource:///org/gnome/shell/ui/main.js";
import Shell from "gi://Shell";
var NavigationBarGestureTracker = class extends Clutter4.GestureAction {
  static {
    GObject3.registerClass({
      Properties: {},
      Signals: {
        "begin": { param_types: [GObject3.TYPE_UINT, GObject3.TYPE_DOUBLE, GObject3.TYPE_DOUBLE] },
        "update": { param_types: [GObject3.TYPE_UINT, GObject3.TYPE_DOUBLE, GObject3.TYPE_DOUBLE] },
        "end": { param_types: [GObject3.TYPE_STRING, GObject3.TYPE_DOUBLE] },
        "cancel": { param_types: [GObject3.TYPE_UINT] }
      }
    }, this);
  }
  _orientation = null;
  //@ts-ignore
  _init(allowedModes = Shell.ActionMode.ALL, nTouchPoints = 1, thresholdTriggerEdge = Clutter4.GestureTriggerEdge.AFTER) {
    super._init();
    this.set_n_touch_points(nTouchPoints);
    this.set_threshold_trigger_edge(thresholdTriggerEdge);
    this._allowedModes = allowedModes;
    this.recognizer = new GestureRecognizer2D();
  }
  get orientation() {
    return this._orientation;
  }
  set orientation(value) {
    this._orientation = value;
  }
  vfunc_gesture_prepare(actor) {
    if (!super.vfunc_gesture_prepare(actor))
      return false;
    if ((this._allowedModes & Main.actionMode) === 0)
      return false;
    let time = this.get_last_event(0).get_time();
    let [xPress, yPress] = this.get_press_coords(0);
    let [x, y] = this.get_motion_coords(0);
    const [xDelta, yDelta] = [x - xPress, y - yPress];
    if (this._orientation !== null) {
      const swipeOrientation = Math.abs(xDelta) > Math.abs(yDelta) ? Clutter4.Orientation.HORIZONTAL : Clutter4.Orientation.VERTICAL;
      if (swipeOrientation !== this._orientation)
        return false;
    }
    this.emit("begin", time, xPress, yPress);
    this.recognizer.resetAndStartGesture();
    this.recognizer.pushEvent(this.get_last_event(0));
    return true;
  }
  vfunc_gesture_progress(_actor) {
    let [x, y] = this.get_motion_coords(0);
    let [initialX, initialY] = this.get_press_coords(0);
    let time = this.get_last_event(0).get_time();
    this.emit("update", time, initialX - x, initialY - y);
    this.recognizer.pushEvent(this.get_last_event(0));
    return true;
  }
  vfunc_gesture_end(_actor) {
    this.recognizer.pushEvent(this.get_last_event(0));
    let lastPattern = this.recognizer.getPatterns().at(-1) || null;
    debugLog("Full gesture: ", this.recognizer.toString());
    debugLog("Last Pattern: ", lastPattern);
    if (lastPattern && lastPattern.type === "swipe") {
      this.emit("end", lastPattern.swipeDirection, lastPattern.swipeSpeed);
    } else {
      this.emit("end", null, null);
    }
  }
  vfunc_gesture_cancel(_actor) {
    let time = Clutter4.get_current_event_time();
    this.emit("cancel", time);
    this.recognizer.pushEvent(this.get_last_event(0));
  }
};

// src/utils/idleRunner.ts
import GLib2 from "gi://GLib";
var IdleRunner = class _IdleRunner {
  callback;
  idleId = null;
  _priority;
  lastRun = null;
  constructor(callback, priority = GLib2.PRIORITY_DEFAULT_IDLE) {
    this.callback = callback;
    this._priority = priority;
  }
  /**
   * An idle runner that is automatically stopped after the callback has been called once.
   *
   * Note that `start` still needs to be called to start the idle runner. If start is called multiple
   * times, the callback will run once per invocation.
   */
  static once(cb, priority = GLib2.PRIORITY_DEFAULT_IDLE) {
    return new _IdleRunner((stop) => {
      cb();
      stop();
    }, priority);
  }
  /**
   * Start the idle runner if it is not running already.
   */
  start() {
    if (this.idleId !== null)
      return;
    const iid = GLib2.idle_add(
      this._priority,
      () => {
        let now = Date.now();
        let dt = this.lastRun != null ? now - this.lastRun : null;
        this.lastRun = now;
        this.callback(this.stop.bind(this), dt);
        return this.idleId === iid ? GLib2.SOURCE_CONTINUE : GLib2.SOURCE_REMOVE;
      }
    );
    this.idleId = iid;
  }
  /**
   * Stop running the idle callback. Can be resumed using `start()`.
   */
  stop() {
    if (this.idleId !== null) {
      GLib2.source_remove(this.idleId);
      this.idleId = null;
      this.lastRun = null;
    }
  }
};

// src/utils/colors.ts
function calculateLuminance(r, g, b) {
  return 0.2126 * r / 255 + 0.7152 * g / 255 + 0.0722 * b / 255;
}
function calculateAverageColor(pixels, width, areasToInclude) {
  let totalR = 0, totalG = 0, totalB = 0, count = 0;
  for (let a of areasToInclude) {
    for (let x = a.x; x < a.x + a.width; x++) {
      for (let y = a.y; y < a.y + a.height; y++) {
        let index = (y * width + x) * 4;
        totalR += pixels[index] || 0;
        totalG += pixels[index + 1] || 0;
        totalB += pixels[index + 2] || 0;
        count++;
      }
    }
  }
  return [totalR, totalG, totalB].map((x) => Math.round(x / count));
}

// src/utils/ui/windowPositionTracker.ts
import Meta from "gi://Meta";
import * as Main2 from "resource:///org/gnome/shell/ui/main.js";
var WindowPositionTracker = class {
  _signalIds = /* @__PURE__ */ new Map();
  _updateDelay;
  callback;
  constructor(callback) {
    this.callback = callback;
    this._signalIds.set(Main2.overview, [
      Main2.overview.connect("showing", this._update.bind(this)),
      Main2.overview.connect("hiding", this._update.bind(this)),
      Main2.overview.connect("shown", this._update.bind(this)),
      Main2.overview.connect("hidden", this._update.bind(this))
    ]);
    this._signalIds.set(Main2.sessionMode, [
      Main2.sessionMode.connect("updated", this._update.bind(this))
    ]);
    for (const metaWindowActor of global.get_window_actors()) {
      this._onWindowActorAdded(metaWindowActor.get_parent(), metaWindowActor);
    }
    this._signalIds.set(global.windowGroup, [
      global.windowGroup.connect("child-added", this._onWindowActorAdded.bind(this)),
      global.windowGroup.connect("child-removed", this._onWindowActorRemoved.bind(this))
    ]);
    this._signalIds.set(global.windowManager, [
      global.windowManager.connect("switch-workspace", this._updateDelayed.bind(this))
    ]);
    this._update();
  }
  _onWindowActorAdded(container, metaWindowActor) {
    this._signalIds.set(metaWindowActor, [
      metaWindowActor.connect("notify::allocation", this._update.bind(this)),
      metaWindowActor.connect("notify::visible", this._update.bind(this))
    ]);
  }
  _onWindowActorRemoved(container, metaWindowActor) {
    for (const signalId of this._signalIds.get(metaWindowActor) ?? []) {
      metaWindowActor.disconnect(signalId);
    }
    this._signalIds.delete(metaWindowActor);
    this._update();
  }
  _update() {
    if (!Main2.layoutManager.primaryMonitor) {
      return;
    }
    const workspaceManager = global.workspaceManager;
    const activeWorkspace = workspaceManager.get_active_workspace();
    const windows = activeWorkspace.list_windows().filter((metaWindow) => {
      return metaWindow.is_on_primary_monitor() && metaWindow.showing_on_its_workspace() && !metaWindow.is_hidden() && metaWindow.get_window_type() !== Meta.WindowType.DESKTOP && !metaWindow.skipTaskbar;
    });
    this.callback(windows);
  }
  _updateDelayed() {
    this._updateDelay = Delay.ms(100).then(() => {
      this._update();
    });
  }
  destroy() {
    for (const [actor, signalIds] of this._signalIds) {
      for (const signalId of signalIds) {
        actor.disconnect(signalId);
      }
    }
    this._signalIds.clear();
    this._updateDelay?.cancel();
  }
};
;

// src/utils/signal.ts
var Signal = class {
  listeners = /* @__PURE__ */ new Map();
  emit(args) {
    for (let l of this.listeners.values()) {
      l(args);
    }
  }
  // Add a dummy signal name to keep this signature compatible with GObjects:
  connect(signal, handler) {
    console.assert(signal === "changed", "The only supported signal for now is `changed`");
    let id = Date.now() + Math.random();
    this.listeners.set(id, handler);
    return id;
  }
  disconnect(id) {
    this.listeners.delete(id);
  }
};

// src/features/navigationBar/widgets/baseNavigationBar.ts
import * as Main3 from "resource:///org/gnome/shell/ui/main.js";
var BaseNavigationBar = class {
  windowPositionTracker;
  _visible = false;
  _reserveSpace = true;
  actor;
  onVisibilityChanged = new Signal();
  onReserveSpaceChanged = new Signal();
  constructor({ reserveSpace }) {
    this._reserveSpace = reserveSpace;
    this.actor = this._buildActor();
  }
  get monitor() {
    return this._monitor;
  }
  get isVisible() {
    return this._visible;
  }
  get reserveSpace() {
    return this._reserveSpace;
  }
  show() {
    if (this.isVisible)
      return;
    this._addActor();
    this._visible = true;
    this.onVisibilityChanged.emit(true);
    this.reallocate();
    this._createWindowPositionTracker();
  }
  hide() {
    if (!this.isVisible)
      return;
    this._removeActor();
    this.windowPositionTracker?.destroy();
    this.windowPositionTracker = void 0;
    this._visible = false;
    this.onVisibilityChanged.emit(false);
  }
  setReserveSpace(reserveSpace) {
    if (reserveSpace != this._reserveSpace) {
      this._reserveSpace = reserveSpace;
      this._removeActor();
      this._addActor();
      this.onReserveSpaceChanged.emit(reserveSpace);
    }
  }
  reallocate() {
    this._monitor = Main3.layoutManager.primaryMonitor;
    this.onBeforeReallocate();
    this.actor.set_position(this.monitor.x, this.monitor.y + this.monitor.height - this.actor.height);
    this.actor.set_width(this.monitor.width);
  }
  _addActor() {
    Main3.layoutManager.addTopChrome(this.actor, {
      affectsStruts: this.reserveSpace,
      trackFullscreen: true,
      affectsInputRegion: true
    });
  }
  _removeActor() {
    Main3.layoutManager.removeChrome(this.actor);
  }
  onBeforeReallocate() {
  }
  _createWindowPositionTracker() {
    let lastIsWindowNear = false;
    this.windowPositionTracker = new WindowPositionTracker((windows) => {
      if (this.actor.realized) {
        const top = this.actor.get_transformed_position()[1];
        const isWindowNear = windows.some((metaWindow) => {
          const windowBottom = metaWindow.get_frame_rect().y + metaWindow.get_frame_rect().height;
          return windowBottom >= top;
        });
        if (isWindowNear !== lastIsWindowNear) {
          this.onIsWindowNearChanged(isWindowNear);
        }
        lastIsWindowNear = isWindowNear;
      }
    });
  }
  destroy() {
    this.actor.destroy();
    this.windowPositionTracker?.destroy();
  }
};

// src/features/navigationBar/widgets/gestureNavigationBar.ts
import St4 from "gi://St";
import Clutter6 from "gi://Clutter";
import Shell2 from "gi://Shell";
import * as Main4 from "resource:///org/gnome/shell/ui/main.js";
var LEFT_EDGE_OFFSET = 100;
var GestureNavigationBar = class extends BaseNavigationBar {
  styleClassUpdateInterval;
  _isWindowNear = false;
  constructor({ reserveSpace }) {
    super({ reserveSpace });
    this.styleClassUpdateInterval = new IntervalRunner(500, this.updateStyleClasses.bind(this));
    this.onVisibilityChanged.connect("changed", () => this._updateStyleClassIntervalActivity());
    this.onReserveSpaceChanged.connect("changed", (reserveSpace2) => {
      this._updateStyleClassIntervalActivity();
      void this.updateStyleClasses();
    });
  }
  _buildActor() {
    return new Widgets.Bin({
      name: "gnometouch-navbar",
      styleClass: "gnometouch-navbar gnometouch-navbar--transparent bottom-panel",
      reactive: true,
      trackHover: true,
      canFocus: true,
      layoutManager: new Clutter6.BinLayout(),
      onCreated: (widget) => this._setupGestureTrackerFor(widget),
      onRealize: () => this.styleClassUpdateInterval.scheduleOnce(),
      child: this.pill = new Widgets.Bin({
        // the navigation bars pill:
        name: "gnometouch-navbar__pill",
        styleClass: "gnometouch-navbar__pill",
        yAlign: Clutter6.ActorAlign.CENTER,
        xAlign: Clutter6.ActorAlign.CENTER
      })
    });
  }
  onIsWindowNearChanged(isWindowNear) {
    this._isWindowNear = isWindowNear;
    if (!this.reserveSpace) {
      let newInterval = Main4.overview.visible || !isWindowNear ? 3e3 : 500;
      if (newInterval != this.styleClassUpdateInterval.interval) {
        this.styleClassUpdateInterval.scheduleOnce(250);
      }
      this.styleClassUpdateInterval.setInterval(newInterval);
    } else {
      void this.updateStyleClasses();
    }
  }
  onBeforeReallocate() {
    const sf = St4.ThemeContext.get_for_stage(global.stage).scaleFactor;
    const height = 22 * sf;
    this.actor.set_height(height);
    this.pill.set_width(clamp(this.monitor.width * 0.25, 70 * sf, 330 * sf));
    this.pill.set_height(Math.floor(Math.min(height * 0.8, 5.5 * sf, height - 2)));
  }
  _setupGestureTrackerFor(actor) {
    const scaleFactor = St4.ThemeContext.get_for_stage(global.stage).scaleFactor;
    const wsController = Main4.wm._workspaceAnimation;
    const gesture = new NavigationBarGestureTracker();
    actor.add_action_full("navigation-bar-gesture", Clutter6.EventPhase.CAPTURE, gesture);
    gesture.orientation = null;
    let baseDistX = 900;
    let baseDistY = global.screenHeight;
    let initialWorkspaceProgress = 0;
    let targetWorkspaceProgress = 0;
    let currentWorkspaceProgress = 0;
    let initialOverviewProgress = 0;
    let targetOverviewProgress = 0;
    let currentOverviewProgress = 0;
    let overviewMaxSpeed = 5e-3;
    let workspaceMaxSpeed = 16e-4;
    const idleRunner = new IdleRunner((_, dt) => {
      dt ??= 0;
      if (Math.abs(targetOverviewProgress - currentWorkspaceProgress) > 5 * overviewMaxSpeed) {
        let d = targetOverviewProgress - currentOverviewProgress;
        currentOverviewProgress += Math.sign(d) * Math.min(Math.abs(d) ** 2, dt * overviewMaxSpeed);
        Main4.overview._gestureUpdate(gesture, currentOverviewProgress);
      }
      if (Math.abs(targetWorkspaceProgress - currentWorkspaceProgress) > 5 * workspaceMaxSpeed) {
        let d = targetWorkspaceProgress - currentWorkspaceProgress;
        currentWorkspaceProgress += Math.sign(d) * Math.min(Math.abs(d) ** 2, dt * workspaceMaxSpeed);
        wsController._switchWorkspaceUpdate({}, currentWorkspaceProgress);
      }
    });
    gesture.connect("begin", (_, time, xPress, yPress) => {
      wsController._switchWorkspaceBegin({
        confirmSwipe(baseDistance, points, progress, cancelProgress) {
          baseDistX = baseDistance;
          initialWorkspaceProgress = currentWorkspaceProgress = targetWorkspaceProgress = progress;
        }
      }, this.monitor.index);
      Main4.overview._gestureBegin({
        confirmSwipe(baseDistance, points, progress, cancelProgress) {
          baseDistY = baseDistance;
          initialOverviewProgress = currentOverviewProgress = targetOverviewProgress = Main4.overview._visible ? progress : 0;
        }
      });
      if (Main4.keyboard.visible)
        Main4.keyboard._keyboard ? Main4.keyboard._keyboard.close(true) : Main4.keyboard.close();
      idleRunner.start();
    });
    gesture.connect("update", (_, time, distX, distY) => {
      targetWorkspaceProgress = initialWorkspaceProgress + distX / baseDistX * 1.6;
      if (Main4.keyboard._keyboard && gesture.get_press_coords(0)[0] < LEFT_EDGE_OFFSET * scaleFactor) {
        Main4.keyboard._keyboard.gestureProgress(distY / baseDistY);
      } else {
        targetOverviewProgress = initialOverviewProgress + distY / (baseDistY * 0.2);
      }
    });
    gesture.connect("end", (_, direction, speed) => {
      idleRunner.stop();
      if (direction === "left" || direction === "right") {
        debugLog(`currenWorkspaceProgress=${targetWorkspaceProgress}, change: ${direction == "left" ? 0.5 : -0.5}`);
        wsController._switchWorkspaceEnd({}, 500, targetWorkspaceProgress + (direction == "left" ? 0.5 : -0.5));
      } else {
        wsController._switchWorkspaceEnd({}, 500, initialWorkspaceProgress);
      }
      if (Main4.keyboard._keyboard && gesture.get_press_coords(0)[0] < LEFT_EDGE_OFFSET * scaleFactor) {
        if (direction == "up") {
          Main4.keyboard._keyboard.gestureActivate(Main4.layoutManager.bottomIndex);
        }
      } else {
        if (direction === "up" || direction === null) {
          Main4.overview._gestureEnd({}, 300, clamp(Math.round(targetOverviewProgress), 1, 2));
        } else {
          Main4.overview._gestureEnd({}, 300, initialOverviewProgress);
        }
      }
    });
    gesture.connect("gesture-cancel", (_gesture) => {
      idleRunner.stop();
      wsController._switchWorkspaceEnd({}, 500, initialWorkspaceProgress);
      if (Main4.keyboard._keyboard && gesture.get_press_coords(0)[0] < LEFT_EDGE_OFFSET * scaleFactor) {
        Main4.keyboard._keyboard.gestureCancel();
      } else {
        Main4.overview._gestureEnd({}, 300, 0);
      }
    });
  }
  async updateStyleClasses() {
    if (this.reserveSpace && this._isWindowNear) {
      this.actor.remove_style_class_name("gnometouch-navbar--transparent");
      this.pill.remove_style_class_name("gnometouch-navbar__pill--dark");
    } else {
      this.actor.add_style_class_name("gnometouch-navbar--transparent");
      let brightness = await this.findBestPillBrightness();
      if (brightness == "dark") {
        this.pill.add_style_class_name("gnometouch-navbar__pill--dark");
      } else {
        this.pill.remove_style_class_name("gnometouch-navbar__pill--dark");
      }
    }
  }
  /**
   * Find the best pill brightness by analyzing what's on the screen behind the pill
   */
  async findBestPillBrightness() {
    try {
      const shooter = new Shell2.Screenshot();
      let rect = this.pill.get_transformed_extents();
      let colors = (await Promise.all([
        // We only use one pixel as doing this with multiple pixels appears to have very bad
        // performance (screen lags, visible e.g. when moving a window):
        shooter.pick_color(rect.get_x() + rect.get_width() * 0.5, rect.get_y() - 2)
        // shooter.pick_color(rect.get_x() + rect.get_width() * 0.4, rect.get_y() + rect.get_height() + 3),
        // @ts-ignore
      ])).map((c) => c[0]);
      let luminance = calculateLuminance(
        colors.reduce((a, b) => a + b.red, 0) / colors.length,
        colors.reduce((a, b) => a + b.green, 0) / colors.length,
        colors.reduce((a, b) => a + b.blue, 0) / colors.length
      );
      return luminance > 0.5 ? "dark" : "light";
    } catch (e) {
      log2("Exception during `findBestPillBrightness` (falling back to 'dark' brightness): ", e);
      return "dark";
    }
  }
  _updateStyleClassIntervalActivity() {
    this.styleClassUpdateInterval.setActive(this.isVisible && !this.reserveSpace);
  }
  destroy() {
    this.styleClassUpdateInterval.stop();
    super.destroy();
  }
};

// src/features/navigationBar/navigationBarUtils.ts
import * as Main5 from "resource:///org/gnome/shell/ui/main.js";
import Meta3 from "gi://Meta";
import Clutter7 from "gi://Clutter";
function navigateBack(props) {
  if (Main5.keyboard.visible) {
    Main5.keyboard._keyboard ? Main5.keyboard._keyboard.close(true) : Main5.keyboard.close();
    return true;
  }
  if (Main5.overview.dash.showAppsButton.checked) {
    Main5.overview.dash.showAppsButton.checked = false;
    return true;
  }
  if (Main5.overview.visible) {
    Main5.overview.hide();
    return true;
  }
  const focusWindow = global.display.focus_window;
  if (focusWindow?.is_fullscreen()) {
    focusWindow.unmake_fullscreen();
    return true;
  }
  if (focusWindow?.can_close()) {
    if (focusWindow.get_window_type() == Meta3.WindowType.DIALOG || focusWindow.get_window_type() == Meta3.WindowType.MODAL_DIALOG || focusWindow.get_window_type() == Meta3.WindowType.MENU || focusWindow.get_window_type() == Meta3.WindowType.DROPDOWN_MENU || focusWindow.get_window_type() == Meta3.WindowType.POPUP_MENU || focusWindow.get_window_type() == Meta3.WindowType.COMBO || focusWindow.get_window_type() == Meta3.WindowType.DND || focusWindow.get_transient_for() !== null) {
      focusWindow.delete(global.get_current_time());
      return true;
    }
    if (props.greedyMode && focusWindow.get_window_type() == Meta3.WindowType.NORMAL) {
      focusWindow.delete(global.get_current_time());
      return true;
    }
  }
  if (props.virtualKeyboardDevice) {
    props.virtualKeyboardDevice.notify_keyval(
      Clutter7.get_current_event_time() * 1e3,
      Clutter7.KEY_Back,
      Clutter7.KeyState.PRESSED
    );
    props.virtualKeyboardDevice.notify_keyval(
      Clutter7.get_current_event_time() * 1e3,
      Clutter7.KEY_Back,
      Clutter7.KeyState.RELEASED
    );
    return null;
  }
  return false;
}
function moveToWorkspace(direction) {
  const wm2 = global.workspaceManager;
  if (direction == "left" && wm2.get_active_workspace_index() == 0)
    return;
  if (direction == "right" && wm2.get_active_workspace_index() == wm2.get_n_workspaces() - 1)
    return;
  const ws = wm2.get_active_workspace().get_neighbor(
    direction == "left" ? Meta3.MotionDirection.LEFT : Meta3.MotionDirection.RIGHT
  );
  if (!ws.active) {
    ws.activate(global.get_current_time());
  }
}

// src/features/navigationBar/widgets/buttonsNavigationBar.ts
import St5 from "gi://St";
import * as Main6 from "resource:///org/gnome/shell/ui/main.js";
import Clutter8 from "gi://Clutter";
var ActorAlign = Clutter8.ActorAlign;
var ButtonsNavigationBar = class extends BaseNavigationBar {
  _virtualKeyboardDevice;
  constructor() {
    super({ reserveSpace: true });
    let seat = Clutter8.get_default_backend().get_default_seat();
    this._virtualKeyboardDevice = seat.create_virtual_device(Clutter8.InputDeviceType.KEYBOARD_DEVICE);
  }
  _buildActor() {
    return new Widgets.Row({
      name: "gnometouch-navbar",
      styleClass: "gnometouch-navbar bottom-panel",
      children: [
        // Left side:
        new Widgets.Row({
          xExpand: false,
          children: settings.navigationBar.buttonsLeft.get().map((b) => this._buildButton(b)),
          onCreated: (row) => {
            const id = settings.navigationBar.buttonsLeft.connect("changed", (newValue) => {
              row.destroy_all_children();
              for (let b of settings.navigationBar.buttonsLeft.get()) {
                row.add_child(this._buildButton(b));
              }
            });
            return () => settings.navigationBar.buttonsLeft.disconnect(id);
          }
        }),
        // Center:
        new Widgets.Row({
          xExpand: true,
          xAlign: ActorAlign.CENTER,
          children: settings.navigationBar.buttonsMiddle.get().map((b) => this._buildButton(b)),
          onCreated: (row) => {
            const id = settings.navigationBar.buttonsMiddle.connect("changed", (newValue) => {
              row.destroy_all_children();
              for (let b of settings.navigationBar.buttonsMiddle.get()) {
                row.add_child(this._buildButton(b));
              }
            });
            return () => settings.navigationBar.buttonsMiddle.disconnect(id);
          }
        }),
        // Right side:
        new Widgets.Row({
          xExpand: false,
          children: settings.navigationBar.buttonsRight.get().map((b) => this._buildButton(b)),
          onCreated: (row) => {
            const id = settings.navigationBar.buttonsRight.connect("changed", (newValue) => {
              row.destroy_all_children();
              for (let b of settings.navigationBar.buttonsRight.get()) {
                row.add_child(this._buildButton(b));
              }
            });
            return () => settings.navigationBar.buttonsRight.disconnect(id);
          }
        })
      ]
    });
  }
  onIsWindowNearChanged(isWindowNear) {
    if (isWindowNear && !Main6.overview.visible) {
      this.actor.remove_style_class_name("gnometouch-navbar--transparent");
    } else {
      this.actor.add_style_class_name("gnometouch-navbar--transparent");
    }
  }
  _buildButton(buttonType) {
    switch (buttonType) {
      case "keyboard":
        return new Widgets.Button({
          name: "gnometouch-navbar__osk-button",
          styleClass: "gnometouch-navbar__button",
          iconName: "input-keyboard-symbolic",
          onClicked: () => Main6.keyboard.open(this.monitor.index)
        });
      case "workspace-previous":
        return new Widgets.Button({
          name: "gnometouch-navbar__workspace-previous-button",
          styleClass: "gnometouch-navbar__button",
          iconName: "go-previous-symbolic",
          onClicked: () => moveToWorkspace("left")
        });
      case "workspace-next":
        return new Widgets.Button({
          name: "gnometouch-navbar__workspace-next-button",
          styleClass: "gnometouch-navbar__button",
          iconName: "go-next-symbolic",
          onClicked: () => moveToWorkspace("right")
        });
      case "overview":
        return new Widgets.Button({
          name: "gnometouch-navbar__overview-button",
          styleClass: "gnometouch-navbar__button",
          child: new Widgets.Icon({
            gicon: new AssetIcon("box-outline-symbolic")
          }),
          onClicked: () => Main6.overview.toggle()
        });
      case "apps":
        return new Widgets.Button({
          name: "gnometouch-navbar__apps-button",
          styleClass: "gnometouch-navbar__button",
          child: new Widgets.Icon({
            gicon: new AssetIcon("grid-large-symbolic")
          }),
          onClicked: () => Main6.overview.dash.showAppsButton.checked = !Main6.overview.dash.showAppsButton.checked
        });
      case "back":
        return new Widgets.Button({
          name: "gnometouch-navbar__back-button",
          styleClass: "gnometouch-navbar__button",
          child: new Widgets.Icon({
            gicon: new AssetIcon("arrow2-left-symbolic")
          }),
          onClicked: () => navigateBack({ virtualKeyboardDevice: this._virtualKeyboardDevice }),
          onLongPress: () => navigateBack({
            virtualKeyboardDevice: this._virtualKeyboardDevice,
            greedyMode: true
          })
        });
      case "spacer":
        return new Widgets.Bin({ width: 20 });
      default:
        log2(`Unknown button for ButtonNavigationBar: ${buttonType}`);
        assertExhaustive(buttonType);
        return new St5.Bin({});
    }
  }
};
function assertExhaustive(p) {
}

// src/features/navigationBar/navigationBarFeature.ts
import Clutter9 from "gi://Clutter";
import * as Main7 from "resource:///org/gnome/shell/ui/main.js";
var NavigationBarFeature = class extends ExtensionFeature {
  onVisibilityChanged = new Signal();
  removeOskActionPatch;
  constructor(pm) {
    super(pm);
    this.pm.connectTo(global.backend.get_monitor_manager(), "monitors-changed", () => {
      this.currentNavBar.reallocate();
    });
    this.pm.connectTo(settings.navigationBar.mode, "changed", (mode) => {
      this.setMode(mode);
    });
    this.pm.connectTo(settings.navigationBar.gesturesReserveSpace, "changed", (value) => {
      if (this._mode === "gestures") {
        this.currentNavBar.setReserveSpace(value);
      }
    });
    this.removeOskActionPatch = this.pm.patch(() => {
      let oskAction = global.stage.get_action("osk");
      if (oskAction)
        global.stage.remove_action(oskAction);
      return () => {
        if (oskAction)
          global.stage.add_action_full("osk", Clutter9.EventPhase.CAPTURE, oskAction);
      };
    });
    this.setMode(settings.navigationBar.mode.get());
  }
  setMode(mode) {
    if (mode === this._mode) {
      return;
    }
    this._mode = mode;
    const visible = this.isVisible;
    this.currentNavBar?.destroy();
    switch (mode) {
      case "gestures":
        this.currentNavBar = new GestureNavigationBar({ reserveSpace: settings.navigationBar.gesturesReserveSpace.get() });
        break;
      case "buttons":
        this.currentNavBar = new ButtonsNavigationBar();
        break;
      default:
        log2(`NavigationBarFeature.setMode() called with an unknown mode: ${mode}`);
        this._mode = "gestures";
        this.currentNavBar = new GestureNavigationBar({ reserveSpace: settings.navigationBar.gesturesReserveSpace.get() });
    }
    if (visible) {
      this.currentNavBar.show();
      this.updateGlobalStyleClasses();
    }
    this._mode == "gestures" ? this.removeOskActionPatch.enable() : this.removeOskActionPatch.disable();
  }
  get mode() {
    return this._mode;
  }
  show() {
    if (this.isVisible)
      return;
    this.currentNavBar.show();
    this.onVisibilityChanged.emit(true);
    this.updateGlobalStyleClasses();
  }
  hide() {
    if (!this.isVisible)
      return;
    this.currentNavBar.hide();
    this.onVisibilityChanged.emit(false);
    this.removeGlobalStyleClasses();
  }
  get isVisible() {
    return this.currentNavBar?.isVisible ?? false;
  }
  /**
   * Adds/updates style classes to Main.uiGroup to allow the CSS-side of this extension to style
   * different elements across the desktop in accordance with the current navigation bar mode and
   * visibility. This is for example used to move up the dash to make place for the navigation bar
   * below it.
   */
  updateGlobalStyleClasses() {
    this.removeGlobalStyleClasses();
    Main7.uiGroup.add_style_class_name(`gnometouch-navbar--${this.mode}`);
    Main7.uiGroup.add_style_class_name(`gnometouch-navbar--visible`);
  }
  /**
   * Remove any style class from Main.uiGroup that was added by [updateGlobalStyleClasses]
   */
  removeGlobalStyleClasses() {
    Main7.uiGroup.styleClass = Main7.uiGroup.styleClass.split(/\s+/).filter((c) => !c.startsWith("gnometouch-navbar--")).join(" ");
  }
  destroy() {
    this.removeGlobalStyleClasses();
    this.currentNavBar?.destroy();
    super.destroy();
  }
};

// src/features/osk/oskKeyPopupsFeature.ts
import * as Main8 from "resource:///org/gnome/shell/ui/main.js";
import * as Keyboard from "resource:///org/gnome/shell/ui/keyboard.js";
import * as BoxPointer from "resource:///org/gnome/shell/ui/boxpointer.js";
import St6 from "gi://St";
var OskKeyPopupsFeature = class extends ExtensionFeature {
  keyPrototype;
  boxPointers = /* @__PURE__ */ new Map();
  constructor(pm) {
    super(pm);
    const self = this;
    this.pm.appendToMethod(Keyboard.Keyboard.prototype, "open", function(..._) {
      if (!self.keyPrototype) {
        self.keyPrototype = self._extractKeyPrototype(this);
        if (!self.keyPrototype) {
          log2("Could not extract Key prototype, thus not patching OSK key popups.");
        } else {
          self._patchKeyMethods(self.keyPrototype);
        }
      }
    });
  }
  _patchKeyMethods(keyProto) {
    const self = this;
    this.pm.appendToMethod(keyProto, "_press", function(button, commitString) {
      if (!self.boxPointers.get(this) && commitString && commitString.trim().length > 0) {
        self.pm.patch(() => {
          const bp = self._buildBoxPointer(this, commitString);
          Main8.layoutManager.addTopChrome(bp);
          self.boxPointers.set(this, bp);
          bp.connect("destroy", () => self.boxPointers.delete(this));
          return () => bp.destroy();
        });
      }
      self.boxPointers.get(this)?.open(BoxPointer.PopupAnimation.FULL);
      Delay.ms(2e3).then(() => {
        self.boxPointers.get(this)?.close(BoxPointer.PopupAnimation.FULL);
      });
    });
    this.pm.appendToMethod(keyProto, "_release", function(button, commitString) {
      Delay.ms(settings.oskKeyPopups.duration.get()).then(() => {
        self.boxPointers.get(this)?.close(BoxPointer.PopupAnimation.FULL);
      });
    });
    this.pm.appendToMethod(keyProto, "_showSubkeys", function() {
      self.boxPointers.get(this)?.close();
    });
    this.pm.appendToMethod(keyProto, "_onDestroy", function() {
      self.boxPointers.get(this)?.destroy();
    });
  }
  _buildBoxPointer(key, commitString) {
    const bp = new BoxPointer.BoxPointer(St6.Side.BOTTOM, {
      styleClass: "key-container"
    });
    bp.add_style_class_name("keyboard-subkeys");
    bp.setPosition(key.keyButton, 0.5);
    if (key._icon && key.iconName) {
      bp.bin.set_child(new St6.Icon({
        styleClass: "keyboard-key",
        name: key.iconName,
        width: key.keyButton.allocation.get_width(),
        height: key.keyButton.allocation.get_height()
      }));
    } else {
      bp.bin.set_child(new St6.Button({
        styleClass: "keyboard-key",
        label: key.keyButton.get_label() || commitString,
        width: key.keyButton.allocation.get_width(),
        height: key.keyButton.allocation.get_height()
      }));
    }
    return bp;
  }
  _extractKeyPrototype(keyboard4) {
    let r = findActorBy(
      keyboard4._aspectContainer,
      (a) => a.constructor.name === "Key" && !!Object.getPrototypeOf(a)
    );
    return r !== null ? Object.getPrototypeOf(r) : null;
  }
};

// src/utils/ui/css.ts
import Cogl from "gi://Cogl";
function css(v) {
  let res = null;
  if (["string", "number"].indexOf(typeof v) !== -1) {
    res = v.toString();
  } else if (v instanceof Cogl.Color) {
    res = `rgba(${v.red}, ${v.green}, ${v.blue}, ${v.alpha / 255})`;
  } else if (Array.isArray(v)) {
    res = v.map((x) => css(x)).join(" ");
  } else if (typeof v === "object") {
    res = "";
    for (let prop in v) {
      if (Object.hasOwn(v, prop)) {
        const cssPropName = prop.replace(/[A-Z]+(?![a-z])|[A-Z]/g, ($, ofs) => (ofs ? "-" : "") + $.toLowerCase());
        res += cssPropName + ": " + css(v[prop]) + "; ";
      }
    }
  } else {
    log(`Warning: cannot convert \`${typeof v}\` to CSS: `, v);
    res = v.toString();
  }
  return !!res ? res.trim() : void 0;
}

// src/utils/monitorDBusUtils.ts
import Gio from "gi://Gio";
import GLib3 from "gi://GLib";
var Methods = Object.freeze({
  verify: 0,
  temporary: 1,
  persistent: 2
});
function callDbusMethod(method, handler, params = null) {
  if (handler !== null && handler !== void 0) {
    Gio.DBus.session.call(
      "org.gnome.Mutter.DisplayConfig",
      "/org/gnome/Mutter/DisplayConfig",
      "org.gnome.Mutter.DisplayConfig",
      method,
      params,
      null,
      Gio.DBusCallFlags.NONE,
      -1,
      null,
      handler
    );
  } else {
    Gio.DBus.session.call(
      "org.gnome.Mutter.DisplayConfig",
      "/org/gnome/Mutter/DisplayConfig",
      "org.gnome.Mutter.DisplayConfig",
      method,
      params,
      null,
      Gio.DBusCallFlags.NONE,
      -1,
      null
    );
  }
}
function setMonitorTransform(transform, targetMonitor) {
  DisplayConfigState.getCurrent().then((state) => {
    targetMonitor ??= state.builtinMonitor ?? state.monitors[0];
    const logicalMonitor = state.getLogicalMonitorFor(targetMonitor.connector);
    if (logicalMonitor) {
      logicalMonitor.transform = transform;
      callDbusMethod("ApplyMonitorsConfig", null, state.packToApply(Methods.temporary));
    }
  });
}
var LogicalMonitor = class {
  x;
  y;
  scale;
  transform;
  primary;
  monitors;
  properties;
  constructor(variant) {
    const unpacked = variant.unpack();
    this.x = unpacked[0].unpack();
    this.y = unpacked[1].unpack();
    this.scale = unpacked[2].unpack();
    this.transform = unpacked[3].unpack();
    this.primary = unpacked[4].unpack();
    this.monitors = unpacked[5].deep_unpack();
    this.properties = unpacked[6].unpack();
    for (const key in this.properties) {
      this.properties[key] = this.properties[key].unpack().unpack();
    }
  }
};
var Monitor2 = class {
  connector;
  currentModeId = null;
  isUnderscanning = false;
  isBuiltin = false;
  constructor(variant) {
    const unpacked = variant.unpack();
    this.connector = unpacked[0].unpack()[0].unpack();
    const modes = unpacked[1].unpack();
    for (const modeVariant of modes) {
      const mode = modeVariant.unpack();
      const id = mode[0].unpack();
      const modeProps = mode[6].unpack();
      if ("is-current" in modeProps) {
        const isCurrent = modeProps["is-current"].unpack().get_boolean();
        if (isCurrent) {
          this.currentModeId = id;
          break;
        }
      }
    }
    const props = unpacked[2].unpack();
    if ("is-underscanning" in props) {
      this.isUnderscanning = props["is-underscanning"].unpack().get_boolean();
    }
    if ("is-builtin" in props) {
      this.isBuiltin = props["is-builtin"].unpack().get_boolean();
    }
  }
};
var DisplayConfigState = class _DisplayConfigState {
  serial;
  monitors = [];
  logicalMonitors = [];
  properties;
  constructor(result) {
    const unpacked = result.unpack();
    this.serial = unpacked[0].unpack();
    const monitorVariants = unpacked[1].unpack();
    for (const monitorPacked of monitorVariants) {
      this.monitors.push(new Monitor2(monitorPacked));
    }
    const logicalMonitorVariants = unpacked[2].unpack();
    for (const logicalMonitorPacked of logicalMonitorVariants) {
      this.logicalMonitors.push(new LogicalMonitor(logicalMonitorPacked));
    }
    this.properties = unpacked[3].unpack();
    for (const key in this.properties) {
      this.properties[key] = this.properties[key].unpack().unpack();
    }
  }
  static async getCurrent() {
    return new Promise((resolve, reject) => {
      callDbusMethod("GetCurrentState", (conn, res) => {
        try {
          const reply = conn?.call_finish(res);
          const configState = new _DisplayConfigState(reply);
          resolve(configState);
        } catch (err) {
          reject(err);
        }
      });
    });
  }
  get builtinMonitor() {
    return this.monitors.find((monitor) => monitor.isBuiltin) ?? this.monitors[0];
  }
  getMonitor(connector) {
    return this.monitors.find((monitor) => monitor.connector === connector) || null;
  }
  getLogicalMonitorFor(connector) {
    return this.logicalMonitors.find(
      (logMonitor) => logMonitor.monitors.some((lmMonitor) => connector === lmMonitor[0])
    ) || null;
  }
  packToApply(method) {
    const packing = [this.serial, method, [], {}];
    const logicalMonitors = packing[2];
    const properties = packing[3];
    this.logicalMonitors.forEach((logicalMonitor) => {
      const lmonitorPack = [
        logicalMonitor.x,
        logicalMonitor.y,
        logicalMonitor.scale,
        logicalMonitor.transform,
        logicalMonitor.primary,
        []
      ];
      const monitors = lmonitorPack[5];
      for (const logMonitor of logicalMonitor.monitors) {
        const connector = logMonitor[0];
        const monitor = this.getMonitor(connector);
        if (monitor) {
          monitors.push([
            connector,
            monitor.currentModeId,
            {
              enable_underscanning: new GLib3.Variant("b", monitor.isUnderscanning)
            }
          ]);
        }
      }
      logicalMonitors.push(lmonitorPack);
    });
    if ("layout-mode" in this.properties) {
      properties["layout-mode"] = new GLib3.Variant("b", this.properties["layout-mode"]);
    }
    return new GLib3.Variant("(uua(iiduba(ssa{sv}))a{sv})", packing);
  }
};

// src/features/notifications/notificationGesturesFeature.ts
import St7 from "gi://St";
import Clutter11 from "gi://Clutter";
import * as Main9 from "resource:///org/gnome/shell/ui/main.js";
import {
  NotificationMessageGroup
} from "resource:///org/gnome/shell/ui/messageList.js";
import * as MessageTray from "resource:///org/gnome/shell/ui/messageTray.js";
var Ref2 = Widgets.Ref;
var NotificationGesturesFeature = class extends ExtensionFeature {
  unpatchOnTrayClose = [];
  calendarMessageList;
  constructor(pm) {
    super(pm);
    const self = this;
    this.calendarMessageList = findActorBy(global.stage, (a) => a.constructor.name == "CalendarMessageList");
    this.calendarMessageList?._messageView.messages.forEach((notificationGroup) => {
      for (let child of notificationGroup.get_children()) {
        if (child.get_first_child() != null) {
          self.patchNotification(child.get_first_child(), false);
        }
      }
    });
    this.pm.appendToMethod(
      NotificationMessageGroup.prototype,
      "_addNotification",
      function(notification) {
        self.patchNotification(this._notificationToMessage.get(notification), false);
      }
    );
    this.pm.appendToMethod(
      MessageTray.MessageTray.prototype,
      "_showNotification",
      function() {
        self.unpatchOnTrayClose.push(self.patchNotification(this._banner, true));
      }
    );
    this.pm.appendToMethod(MessageTray.MessageTray.prototype, "_hideNotification", function() {
      self.unpatchOnTrayClose.forEach((p) => p.disable());
      self.unpatchOnTrayClose = [];
    });
    this.pm.patchMethod(
      MessageTray.MessageTray.prototype,
      "_updateState",
      function(orig) {
        const originalValue = this._banner?.expanded;
        if (this._banner)
          this._banner.expanded = true;
        orig();
        if (this._banner)
          this._banner.expanded = originalValue;
      }
    );
  }
  patchNotification(message, isTray) {
    return this.pm.patch(() => {
      message.reactive = false;
      message.remove_style_pseudo_class("insensitive");
      const container = message.get_parent();
      container.reactive = true;
      container.trackHover = true;
      const notificationGroup = container.get_parent();
      const gestureHelper = new SwipeGesturesHelper({
        actor: container,
        scrollView: !isTray ? this.calendarMessageList?._scrollView : void 0,
        onHover: (isTouch) => {
          message.add_style_pseudo_class("hover");
          if (isTray && !isTouch && !message.expanded) {
            message.expand(true);
          }
        },
        onHoverEnd: () => message.remove_style_pseudo_class("hover"),
        onMoveHorizontally: (x) => {
          const actor = notificationGroup.expanded || isTray ? message : notificationGroup;
          actor.translationX = x;
        },
        onMoveVertically: (y) => {
          if (isTray) {
            message.translationY = Math.min(y, 0);
          }
        },
        onScrollScrollView: (deltaY) => this.scrollNotificationList(deltaY),
        onEaseBackPosition: () => {
          const actor = notificationGroup.expanded || isTray ? message : notificationGroup;
          gestureHelper.easeBackPositionOf(actor);
        },
        onActivate: () => (
          // @ts-ignore
          message.notification.activate()
        ),
        onExpand: () => {
          if (!message.expanded) {
            message.expand(true);
          }
        },
        onCollapse: () => {
          if (isTray) {
            message.ease({
              y: -message.height,
              rotationZ: 90,
              duration: 100,
              mode: Clutter11.AnimationMode.EASE_OUT,
              // @ts-ignore
              onComplete: () => Main9.messageTray._hideNotification(false)
            });
            return { easeBackPosition: false };
          } else if (message.expanded) {
            message.unexpand(true);
          }
        },
        onClose: (swipeDirection) => {
          const actor = notificationGroup.expanded || isTray ? message : notificationGroup;
          actor.ease({
            translationX: swipeDirection == "right" ? message.width : -message.width,
            opacity: 0,
            duration: 150,
            mode: Clutter11.AnimationMode.EASE_OUT,
            onComplete: () => message.emit("close")
          });
        }
      });
      let messageRef = new Ref2(message);
      let containerRef = new Ref2(container);
      const undo = () => {
        containerRef.apply((c) => {
          gestureHelper.destroy();
          c.reactive = false;
          c.trackHover = false;
        });
        messageRef.apply((m) => {
          m.translationX = 0;
          m.reactive = true;
        });
      };
      message.connect("destroy", () => undo());
      return undo;
    });
  }
  scrollNotificationList(delta) {
    const vadj = this.calendarMessageList?._scrollView?.get_vadjustment();
    if (vadj) {
      vadj.value -= delta;
    }
  }
};
var SwipeGesturesHelper = class {
  // Mid-gesture callbacks:
  onHover;
  onHoverEnd;
  onMoveHorizontally;
  onMoveVertically;
  onScrollScrollView;
  // Gesture finished callbacks:
  onActivate;
  onClose;
  onExpand;
  onCollapse;
  onEaseBackPosition;
  actor;
  scrollView;
  recognizer;
  _signalIds;
  /**
   * Whether the gesture currently being performed is a scroll gesture. This is set to `true` when the
   * [onScrollScrollView] callback has been called at least once during the current gesture.
   */
  isScrollGesture = false;
  constructor(props) {
    this.actor = props.actor;
    this.scrollView = props.scrollView;
    this.onHover = props.onHover;
    this.onHoverEnd = props.onHoverEnd;
    this.onMoveHorizontally = props.onMoveHorizontally;
    this.onMoveVertically = props.onMoveVertically;
    this.onScrollScrollView = props.onScrollScrollView;
    this.onActivate = props.onActivate;
    this.onClose = props.onClose;
    this.onExpand = props.onExpand;
    this.onCollapse = props.onCollapse;
    this.onEaseBackPosition = props.onEaseBackPosition || this._defaultOnEaseBackPosition;
    this.recognizer = new GestureRecognizer2D();
    this._signalIds = [
      this.actor.connect("touch-event", this._onEvent.bind(this)),
      this.actor.connect("button-press-event", this._onEvent.bind(this)),
      this.actor.connect("button-release-event", this._onEvent.bind(this)),
      this.actor.connect("notify::hover", this._updateHover.bind(this))
    ];
    this.actor.connect("destroy", () => this._signalIds = []);
  }
  _onEvent(_, e) {
    const prevDeltaY = this.recognizer.totalMotionDelta.y;
    this.recognizer.pushEvent(e);
    if (e.type() == Clutter11.EventType.TOUCH_BEGIN || e.type() == Clutter11.EventType.BUTTON_PRESS) {
      this._updateHover();
    }
    if (this.recognizer.isDuringGesture) {
      if (this.recognizer.primaryMove?.swipeAxis == "horizontal") {
        this.onMoveHorizontally?.(this.recognizer.totalMotionDelta.x);
      } else if (this.recognizer.primaryMove?.swipeAxis == "vertical") {
        const dy = this.recognizer.totalMotionDelta.y - prevDeltaY;
        if (!this.gestureStartedWithLongPress && this.canScrollScrollView(dy > 0 ? "up" : "down")) {
          this.onScrollScrollView?.(dy);
          if (this.recognizer.isCertainlyMovement) {
            this.isScrollGesture = true;
          }
        } else {
          this.onMoveVertically?.(this.recognizer.totalMotionDelta.y);
        }
      }
    }
    if (this.recognizer.gestureHasJustFinished) {
      this._onGestureFinished();
      this._updateHover();
      this.isScrollGesture = false;
    }
  }
  _updateHover() {
    if (this.recognizer.isDuringGesture || this.actor instanceof St7.Widget && this.actor.hover) {
      this.onHover?.(this.recognizer.isTouchGesture);
    } else {
      this.onHoverEnd?.();
    }
  }
  destroy() {
    for (let signalId of this._signalIds) {
      this.actor.disconnect(signalId);
    }
  }
  _onGestureFinished() {
    let defaultShouldEaseBack = false;
    let res = void 0;
    if (this.recognizer.wasTap() || !this.recognizer.isTouchGesture) {
      res = this.onActivate?.();
    } else if (this.recognizer.secondaryMove != null) {
      switch (this.recognizer.secondaryMove.swipeDirection) {
        case "up":
          if (this.isScrollGesture)
            break;
          res = this.onCollapse?.();
          defaultShouldEaseBack = true;
          break;
        case "down":
          if (this.isScrollGesture)
            break;
          res = this.onExpand?.();
          defaultShouldEaseBack = true;
          break;
        default:
          if (this.recognizer.secondaryMove.swipeDirection == "right" && this.recognizer.totalMotionDelta.x > 0 || this.recognizer.secondaryMove.swipeDirection == "left" && this.recognizer.totalMotionDelta.x < 0) {
            res = this.onClose?.(this.recognizer.secondaryMove.swipeDirection);
          } else {
            defaultShouldEaseBack = true;
          }
      }
    }
    if (res?.easeBackPosition || defaultShouldEaseBack && res?.easeBackPosition != false) {
      this.onEaseBackPosition?.();
    }
  }
  get gestureStartedWithLongPress() {
    return this.recognizer.primaryPattern?.type === "hold";
  }
  canScrollScrollView(direction = null) {
    if (!this.scrollView?.get_vscrollbar_visible())
      return false;
    const vadj = this.scrollView.get_vadjustment();
    switch (direction) {
      case "up":
        return !!vadj && vadj.value > 0;
      case "down":
        return !!vadj && vadj.value < vadj.upper - this.scrollView.contentBox.get_height();
    }
    return true;
  }
  _defaultOnEaseBackPosition() {
    this.easeBackPositionOf(this.actor);
  }
  easeBackPositionOf(actor) {
    actor.ease({
      translationX: 0,
      translationY: 0,
      duration: 200,
      mode: Clutter11.AnimationMode.EASE_OUT_BACK
    });
  }
};

// node_modules/@girs/gnome-shell/dist/ui/messageTray.js
var messageTray_exports = {};
__reExport(messageTray_exports, messageTray_star);
import * as messageTray_star from "resource:///org/gnome/shell/ui/messageTray.js";

// src/features/screenRotateUtils/floatingScreenRotateButtonFeature.ts
import Gio2 from "gi://Gio";
import St8 from "gi://St";
import Clutter12 from "gi://Clutter";
import Graphene from "gi://Graphene";
var Ref3 = Widgets.Ref;
var FloatingScreenRotateButtonFeature = class extends ExtensionFeature {
  touchscreenSettings = new Gio2.Settings({
    schema_id: "org.gnome.settings-daemon.peripherals.touchscreen"
  });
  floatingButton = new Ref3();
  constructor(pm) {
    super(pm);
    this.pm.connectTo(global.backend.get_monitor_manager(), "monitors-changed", (manager) => {
      this.removeFloatingRotateButton({ animate: false });
    });
    this.pm.patch(() => {
      const handlerId = Gio2.DBus.system.signal_subscribe(
        null,
        "org.freedesktop.DBus.Properties",
        "PropertiesChanged",
        "/net/hadess/SensorProxy",
        null,
        Gio2.DBusSignalFlags.NONE,
        (connection, sender_name, object_path, interface_name, signal_name, parameters) => {
          debugLog("SensorProxy PropertiesChanged changed: ", parameters?.deepUnpack());
          const orientation = parameters?.deepUnpack()?.at(1)?.AccelerometerOrientation?.deepUnpack();
          if (orientation) {
            this.onAccelerometerOrientationChanged(orientation).then();
          }
        }
      );
      return () => Gio2.DBus.system.signal_unsubscribe(handlerId);
    });
  }
  get isOrientationLockEnabled() {
    return this.touchscreenSettings.get_boolean("orientation-lock");
  }
  async onAccelerometerOrientationChanged(orientation) {
    this.removeFloatingRotateButton({ animate: true });
    if (this.isOrientationLockEnabled) {
      const targetTransform = {
        "normal": 0,
        "left-up": 1,
        "bottom-up": 2,
        "right-up": 3
      }[orientation];
      const { geometry, transform: currentTransform } = await this.getBuiltinMonitorGeometryAndTransform();
      if (currentTransform !== targetTransform) {
        this.showFloatingRotateButton(currentTransform, targetTransform, geometry);
      }
    }
  }
  showFloatingRotateButton(currentTransform, targetTransform, monitorGeometry) {
    let [aX, aY] = computeAlignment(currentTransform, targetTransform);
    const sf = St8.ThemeContext.get_for_stage(global.stage).scaleFactor;
    const buttonSize = 40 * sf;
    const margin = 40 * sf;
    let btn = this.pm.autoDestroy(new Widgets.Button({
      ref: this.floatingButton,
      styleClass: "gnometouch-floating-screen-rotation-button",
      iconName: "rotation-allowed-symbolic",
      width: 40 * sf,
      height: 40 * sf,
      x: monitorGeometry.x + clamp(monitorGeometry.width * aX, margin, monitorGeometry.width - buttonSize - margin),
      y: monitorGeometry.y + clamp(monitorGeometry.height * aY, margin, monitorGeometry.height - buttonSize - margin),
      onClicked: () => {
        btn?.destroy();
        setMonitorTransform(targetTransform);
      },
      onDestroy: () => btn = null,
      opacity: 128,
      scaleX: 0.5,
      scaleY: 0.5,
      pivotPoint: new Graphene.Point({ x: 0.5, y: 0.5 })
    }));
    global.stage.add_child(btn);
    btn.ease({
      opacity: 255,
      scaleX: 1,
      scaleY: 1,
      duration: 250,
      // ms
      mode: Clutter12.AnimationMode.EASE_OUT_QUAD
    });
    for (let i = 0; i < 3; i++) {
      Delay.ms(700 + 2e3 * i).then(() => {
        btn?.ease({
          rotationAngleZ: btn.rotationAngleZ - 90,
          duration: 550,
          // ms
          mode: Clutter12.AnimationMode.EASE_IN_OUT_QUAD
        });
      });
    }
    Delay.ms(7e3).then(() => {
      if (btn === this.floatingButton.current) {
        this.removeFloatingRotateButton({ animate: true });
      }
    });
  }
  removeFloatingRotateButton({ animate = false }) {
    if (animate) {
      const btn = this.floatingButton.current;
      btn?.ease({
        scaleX: 0.5,
        scaleY: 0.5,
        opacity: 128,
        duration: 250,
        // ms
        mode: Clutter12.AnimationMode.EASE_IN_QUAD,
        onComplete: () => btn?.destroy()
      });
    } else {
      this.floatingButton.current?.destroy();
    }
  }
  async getBuiltinMonitorGeometryAndTransform() {
    const state = await DisplayConfigState.getCurrent();
    const monitorConnector = (state.builtinMonitor ?? state.monitors[0]).connector;
    const monitorIndex = global.backend.get_monitor_manager().get_monitor_for_connector(monitorConnector);
    const geometry = global.display.get_monitor_geometry(monitorIndex);
    const transform = state.getLogicalMonitorFor(monitorConnector).transform;
    return { geometry, transform };
  }
};
function computeAlignment(currentTransform, targetTransform) {
  const [baseX, baseY] = {
    0: [1, 1],
    // Bottom-right
    1: [1, 0],
    // Bottom-left
    2: [0, 0],
    // Top-left
    3: [0, 1]
    // Top-right
  }[targetTransform] || [1, 1];
  return {
    0: [baseX, baseY],
    // Normal
    1: [1 - baseY, baseX],
    // 90°
    2: [1 - baseX, 1 - baseY],
    // 180°
    3: [baseY, 1 - baseX],
    // 270°
    4: [1 - baseX, baseY],
    // Flipped
    5: [baseY, baseX],
    // 90° Flipped
    6: [baseX, 1 - baseY],
    // 180° Flipped
    7: [1 - baseY, 1 - baseX]
    // 270° Flipped
  }[currentTransform] || [baseX, baseY];
}

// src/utils/ui/toast.ts
import Clutter13 from "gi://Clutter";
function showToast(text, actions) {
  const toast = new Widgets.Row({
    style: css({
      background: "rgba(32,32,32,0.8)",
      borderRadius: "50px",
      padding: "15px",
      fontSize: "90%",
      fontWeight: "normal",
      paddingLeft: "25px"
    }),
    reactive: true,
    trackHover: true,
    children: [
      new Widgets.Label({ text, style: css({ fontWeight: "bold" }), yAlign: Clutter13.ActorAlign.CENTER }),
      new Widgets.Bin({ width: actions.length > 0 ? 45 : 0 }),
      ...actions
    ]
  });
  toast.x = global.screenWidth / 2 - toast.width / 2;
  toast.y = global.screenHeight;
  global.stage.add_child(toast);
  toast.ease({
    y: global.screenHeight - toast.height - 100,
    duration: 300,
    mode: Clutter13.AnimationMode.EASE_OUT_QUAD
  });
  const ref = new Widgets.Ref(toast);
  actions.forEach((a) => a.connect("clicked", () => ref.current?.destroy()));
  Delay.ms(4e3, "resolve").then(() => ref.current?.ease({
    y: global.screenHeight,
    opacity: 0,
    duration: 150,
    mode: Clutter13.AnimationMode.EASE_IN_QUAD,
    onComplete: () => ref.current?.destroy()
  }));
}

// src/features/donations/donationsFeature.ts
import Gio3 from "gi://Gio";
import * as MessageTray3 from "resource:///org/gnome/shell/ui/messageTray.js";
import * as Main10 from "resource:///org/gnome/shell/ui/main.js";
var DonationsFeature = class _DonationsFeature extends ExtensionFeature {
  // Time to wait before showing a donation; this is to not show the donation immediately upon login because
  // the user at that point probably is busy, and we don't want to uselessly annoy them:
  static NOTIFICATION_DELAY = 20;
  // in minutes
  // Time between donation prompt notifications:
  static NOTIFICATION_INTERVAL = 90 * 24 * 60 * 60 * 1e3;
  // in ms; 90 days (~ quarter of a year)
  notificationSource;
  constructor(pm) {
    super(pm);
    Delay.ms(700).then((_) => this._initializeInstallationData()).then((data) => this._maybeScheduleNotification(data));
  }
  async _initializeInstallationData() {
    try {
      const data = await this._readInstallationData();
      debugLog("Installation data: ", data);
      this._validateInstallationData(data);
      return data;
    } catch (e) {
      const data = {
        installedAt: Date.now(),
        promptedForDonationAt: null
      };
      await this._writeInstallationData(data);
      return data;
    }
  }
  _maybeScheduleNotification(data) {
    if (data.dontAskAgain === true)
      return;
    const dt = data.promptedForDonationAt ?? data.installedAt;
    if (dt && Date.now() - dt > _DonationsFeature.NOTIFICATION_INTERVAL) {
      debugLog(`Scheduling notification in ${_DonationsFeature.NOTIFICATION_DELAY} minutes`);
      Delay.min(_DonationsFeature.NOTIFICATION_DELAY).then(() => this.showDonationNotification(data));
    }
  }
  /**
   * Show a panel notification asking the user to donate.
   */
  async showDonationNotification(data) {
    const n = randomChoice(NOTIFICATION_VARIANTS);
    const notification = new MessageTray3.Notification({
      source: this.getNotificationSource(),
      title: n.title,
      body: n.body,
      gicon: new AssetIcon("positive-feedback-symbolic"),
      urgency: MessageTray3.Urgency.NORMAL
    });
    notification.connect("activated", () => this.openDonationPage());
    notification.addAction("Learn more", () => this.openDonationPage());
    notification.addAction("Not now", async () => {
      showToast("No problem \u2013 you'll receive a notification in a few months again!", [
        new Widgets.Button({
          label: "Never ask again",
          styleClass: "button",
          onClicked: async () => await this._writeInstallationData({
            ...data ?? await this._readInstallationData(),
            dontAskAgain: true
          })
        }),
        new Widgets.Button({
          iconName: "window-close-symbolic",
          style: css({ height: "10px" })
        })
      ]);
    });
    this.notificationSource?.addNotification(notification);
    await this._writeInstallationData({
      ...data ?? await this._readInstallationData(),
      promptedForDonationAt: Date.now()
    });
  }
  openDonationPage() {
    if (!GnomeTouchExtension.instance)
      return;
    settings.initialPreferencesPage.set("donations");
    GnomeTouchExtension.instance.openPreferences();
  }
  async _readInstallationData() {
    return JSON.parse(settings.donations.installationData.get());
  }
  async _writeInstallationData(data) {
    try {
      settings.donations.installationData.set(JSON.stringify(data));
    } catch (e) {
      debugLog("Error while trying to write installation data: ", e instanceof Gio3.IOErrorEnum ? [e.code, e.message] : e);
    }
  }
  _validateInstallationData(data) {
    if (typeof data.installedAt !== "number") {
      throw new Error("Missing or invalid field in installation data: 'installedAt'");
    }
    if (data.promptedForDonationAt != null && !["number", "undefined"].includes(typeof data.promptedForDonationAt)) {
      throw new Error(`Invalid data type in installation data field 'promptedForDonation': ${typeof data.promptedForDonationAt}`);
    }
    if (data.dontAskAgain != null && !["boolean", "undefined"].includes(typeof data.dontAskAgain)) {
      throw new Error(`Invalid data type in installation data field 'dontAskAgain': ${typeof data.dontAskAgain}`);
    }
  }
  getNotificationSource() {
    if (!this.notificationSource) {
      this.notificationSource = new MessageTray3.Source({
        title: "GnomeTouch",
        // An icon for the source, used a fallback by notifications
        icon: new Gio3.ThemedIcon({ name: "dialog-information" }),
        iconName: "dialog-information",
        policy: new messageTray_exports.NotificationGenericPolicy()
      });
      this.pm.patch(() => {
        Main10.messageTray.add(this.notificationSource);
        return () => this.notificationSource?.destroy(messageTray_exports.NotificationDestroyedReason.SOURCE_CLOSED);
      });
      this.notificationSource.connect("destroy", (_source) => this.notificationSource = void 0);
    }
    return this.notificationSource ?? null;
  }
};
var NOTIFICATION_VARIANTS = [
  {
    "title": "Is GnomeTouch helpful for you? \u{1F31F}",
    "body": "Support its development by making a donation. Every contribution helps! \u{1F496}"
  },
  {
    "title": "Thank you for using GnomeTouch! \u2764\uFE0F",
    "body": "If you find it useful, consider supporting the project with a donation. Click to learn more."
  },
  {
    "title": "Help Keep GnomeTouch Going \u{1F91D}",
    "body": "Donations help cover development time and maintenance. Every little bit helps! \u2764\uFE0F"
  },
  {
    "title": "Consider Supporting GnomeTouch \u{1F91D}",
    "body": "We rely on your generosity to keep improving. Click here to donate. \u2764\uFE0F"
  },
  {
    "title": "Keep Us Coding! \u{1F4BB}",
    "body": "Your generosity powers innovation and independence. Make a donation today to support GnomeTouch! \u2764\uFE0F"
  },
  {
    "title": "Support Open Source \u2764\uFE0F",
    "body": "Your donations keep open-source projects like GnomeTouch alive. Help us grow! \u{1F31F}"
  },
  {
    "title": "Make a Difference \u{1F30D}",
    "body": "Your support fuels this project. Donate today to keep GnomeTouch going strong! \u{1F4AA}"
  },
  {
    "title": "Empower Open Platforms \u270A",
    "body": "GnomeTouch makes GNOME more useful on tablets \u2013 and helps it challenge corporate giants. Your donation strengthens the fight for open software! \u{1F30D}"
  },
  {
    "title": "Open Source Needs You! \u{1F6E0}\uFE0F",
    "body": "Big Tech monopolies dominate mobile OSes \u2014 but you and GnomeTouch can help making GNOME an independent alternative. Donate today!"
  },
  {
    "title": "We have big plans for GnomeTouch... ",
    "body": "... and you can help making it happen! Take a look at what's coming, leave us some new ideas or make us faster with a small donation! \u{1F604}"
  },
  {
    "title": "GnomeTouch has a long bucket list! \u{1FAA3}",
    "body": "Curios what else is coming? Have a look at the planned features and help us realize them with a small donation \u2764\uFE0F"
  }
];

// src/extension.ts
import Clutter14 from "gi://Clutter";
import { Extension } from "resource:///org/gnome/shell/extensions/extension.js";
import Gio4 from "gi://Gio";
var GnomeTouchExtension = class _GnomeTouchExtension extends Extension {
  static instance;
  pm;
  developmentTools;
  navigationBar;
  oskKeyPopups;
  floatingScreenRotateButton;
  virtualTouchpad;
  notificationGestures;
  donations;
  enable() {
    debugLog("*************************************************");
    debugLog(`          StartingGnome Touch v. ${this.metadata.version}          `);
    debugLog("*************************************************");
    debugLog();
    this.pm = new PatchManager("root");
    this.pm.patch(() => {
      const assets = Gio4.resource_load(this.dir.get_child(assetsGResourceFile).get_path());
      Gio4.resources_register(assets);
      return () => Gio4.resources_unregister(assets);
    }, "load-and-register-assets");
    this.pm.patch(() => {
      initSettings(this.getSettings());
      return () => uninitSettings();
    }, "init-settings");
    _GnomeTouchExtension.instance = this;
    this.defineFeatures();
    this.pm.connectTo(Clutter14.get_default_backend().get_default_seat(), "notify::touch-mode", () => this.syncUI());
    this.pm.connectTo(global.backend.get_monitor_manager(), "monitors-changed", () => this.syncUI());
    this.syncUI();
  }
  syncUI() {
    let touchMode = Clutter14.get_default_backend().get_default_seat().get_touch_mode();
    if (touchMode) {
      this.navigationBar?.show();
    } else {
      this.navigationBar?.hide();
    }
    this.virtualTouchpad?.setCanOpen(
      touchMode
      /*&&  global.display.get_n_monitors() > 1*/
    );
  }
  defineFeatures() {
    this.defineFeature(
      "navigation-bar",
      (pm) => new NavigationBarFeature(pm),
      (f) => this.navigationBar = f,
      settings.navigationBar.enabled
    );
    this.defineFeature(
      "osk-key-popups",
      (pm) => new OskKeyPopupsFeature(pm),
      (f) => this.oskKeyPopups = f,
      settings.oskKeyPopups.enabled
    );
    this.defineFeature(
      "floating-screen-rotate-button",
      (pm) => new FloatingScreenRotateButtonFeature(pm),
      (f) => this.floatingScreenRotateButton = f,
      settings.screenRotateUtils.floatingScreenRotateButtonEnabled
    );
    this.defineFeature(
      "notification-gestures",
      (pm) => new NotificationGesturesFeature(pm),
      (f) => this.notificationGestures = f,
      settings.notificationGestures.enabled
    );
    this.defineFeature(
      "donations",
      (pm) => new DonationsFeature(pm),
      (f) => this.donations = f
    );
  }
  /**
   * A utility method to define [ExtensionFeature]s that are optionally automatically enabled/disabled
   * depending on the given [setting] and are mapped to a class attribute using [assign].
   *
   * Note that the [assign] callback can (and will upon feature or extension disabling) be
   * called with `undefined' as its value; this is intended behavior and the callback should
   * unset the reference it assigned before in this case.
   *
   * All features are created in a patch and are therefore automatically disabled and set to
   * `undefined` when the extension is disabled.
   *
   * For example usages see [defineFeatures] above.
   */
  defineFeature(featureName, create, assign, setting) {
    let p = this.pm.registerPatch(() => {
      let feature = create(this.pm.fork(featureName));
      assign(feature);
      return () => {
        feature?.destroy();
        feature = void 0;
        assign(void 0);
      };
    }, `enable-feature(${featureName})`);
    if (setting) {
      if (setting.get())
        p.enable();
      this.pm.connectTo(setting, "changed", (value) => {
        if (value) {
          p.enable();
          this.syncUI();
        } else {
          p.disable();
        }
      });
    } else {
      p.enable();
    }
  }
  disable() {
    debugLog(`Cancelling ${Delay.getAllPendingDelays().length} pending delay(s)`);
    Delay.getAllPendingDelays().forEach((d) => d.cancel());
    this.pm?.destroy();
    this.pm = void 0;
    _GnomeTouchExtension.instance = void 0;
    debugLog("GnomeTouch extension successfully unloaded.");
  }
};
export {
  GnomeTouchExtension as default
};
