import {
  AssetIcon,
  Delay,
  assetsGResourceFile,
  donationPlatforms,
  feedbackPlatforms,
  initSettings,
  randomChoice,
  settings
} from "./chunk-TSRMGQWY.js";

// src/features/preferences/uiUtils.ts
import Adw from "gi://Adw";
import Gtk from "gi://Gtk";
import Gio from "gi://Gio";
var Align = Gtk.Align;
function buildPreferencesGroup(props) {
  const group = new Adw.PreferencesGroup({
    title: props.title,
    description: props.description ?? ""
  });
  props.children?.forEach((c) => group.add(c));
  props.onCreated?.(group);
  return group;
}
function buildSwitchRow(props) {
  const row = new Adw.SwitchRow({
    title: props.title,
    subtitle: props.subtitle
  });
  if ("setting" in props) {
    props.setting.bind(row, "active", props.invert ? Gio.SettingsBindFlags.INVERT_BOOLEAN : void 0);
  }
  if ("initialValue" in props) {
    row.active = props.invert ? !props.initialValue : props.initialValue;
  }
  if ("onChanged" in props) {
    row.connect("notify::active", () => props.onChanged?.(props.invert ? !row.active : row.active));
  }
  props.onCreated?.(row);
  return row;
}
function buildSpinRow(props) {
  const row = new Adw.SpinRow({
    title: props.title,
    subtitle: props.subtitle,
    adjustment: props.adjustment
  });
  if ("setting" in props) {
    props.setting.bind(row, "value");
  }
  if ("initialValue" in props) {
    row.set_value(props.initialValue);
  }
  if ("onChanged" in props) {
    row.connect("value-changed", () => props.onChanged(row.get_value()));
  }
  props.onCreated?.(row);
  return row;
}
function buildComboRow(props) {
  const row = new Adw.ComboRow({
    title: props.title,
    subtitle: props.subtitle,
    model: new Gtk.StringList({
      strings: props.items.map((i) => i.label)
    })
  });
  const initialValue = "initialValue" in props ? props.initialValue : props.setting.get();
  const onChanged = (v) => {
    if ("onChanged" in props)
      props.onChanged(v);
    if ("setting" in props)
      props.setting.set(v);
  };
  row.set_selected(props.items.findIndex((i) => i.value === initialValue));
  row.connect("notify::selected-item", onChanged);
  props.onCreated?.(row);
  return row;
}
function buildToggleButtonRow(props) {
  const row = new Adw.ActionRow({
    title: props.title,
    subtitle: props.subtitle
  });
  const toggleGroup = [];
  const buttonBox = new Gtk.Box({
    orientation: Gtk.Orientation.HORIZONTAL,
    cssClasses: ["linked"],
    valign: Align.CENTER
  });
  const initialValue = "initialValue" in props ? props.initialValue : props.setting.get();
  const onChanged = (v) => {
    if ("onChanged" in props)
      props.onChanged(v);
    if ("setting" in props)
      props.setting.set(v);
  };
  props.items.forEach((item) => {
    const button = new Gtk.ToggleButton({
      label: item.label
    });
    if (item.value === initialValue) {
      button.active = true;
    }
    button.connect("toggled", () => {
      if (button.active) {
        toggleGroup.forEach((b) => {
          if (b !== button) {
            b.active = false;
          }
        });
        onChanged(item.value);
      } else if (toggleGroup.every((b) => !b.active)) {
        button.active = true;
      }
    });
    toggleGroup.push(button);
    buttonBox.append(button);
  });
  row.add_suffix(buttonBox);
  row.set_activatable_widget(buttonBox);
  props.onCreated?.(row, toggleGroup);
  return row;
}

// src/utils/debounce.ts
function debounce(func, delay_ms) {
  let d = null;
  return (...args) => {
    d?.cancel();
    d = Delay.ms(delay_ms);
    d.then((_) => func(...args));
  };
}

// src/features/preferences/pages/navigationBarPage.ts
import Adw2 from "gi://Adw";
import GObject from "gi://GObject";
import Gtk2 from "gi://Gtk";
import Gdk from "gi://Gdk";
import Gio2 from "gi://Gio";
var PropagationPhase = Gtk2.PropagationPhase;
var NavigationBarPage = class extends Adw2.PreferencesPage {
  static {
    GObject.registerClass(this);
  }
  constructor() {
    super({
      name: "navigation-bar",
      title: "Navigation Bar",
      icon_name: "computer-apple-ipad-symbolic"
    });
    this.add(buildPreferencesGroup({
      title: "Navigation Bar",
      description: "Configure the behavior and appearance of the navigation bar",
      children: [
        buildSwitchRow({
          title: "Enable Navigation Bar",
          subtitle: "Toggle to enable or disable the navigation bar feature",
          setting: settings.navigationBar.enabled
        }),
        buildToggleButtonRow({
          title: "Navigation Bar Mode",
          subtitle: "Choose which kind of navigation experience you prefer",
          items: [
            { label: "Gestures", value: "gestures" },
            { label: "Buttons", value: "buttons" }
          ],
          setting: settings.navigationBar.mode
        })
      ]
    }));
    this.add(buildPreferencesGroup({
      title: "Gestures Navigation Bar",
      children: [
        buildSwitchRow({
          title: "Reserve Space",
          subtitle: "Keep space available for the navigation bar to avoid overlaying windows. If disabled, the navigation bar is shown on top of overlapping windows and adjusts its color dynamically.",
          setting: settings.navigationBar.gesturesReserveSpace
        })
      ],
      // Only show this group when mode is set to "gestures":
      onCreated: (group) => {
        const id = settings.navigationBar.mode.connect("changed", (mode) => group.visible = mode === "gestures");
        group.connect("destroy", () => settings.navigationBar.mode.disconnect(id));
        group.visible = settings.navigationBar.mode.get() === "gestures";
      }
    }));
    this.add(buildPreferencesGroup({
      title: "Buttons Navigation Bar",
      children: [
        this.buildButtonsRow()
      ],
      // Only show this group when mode is set to "buttons":
      onCreated: (group) => {
        const id = settings.navigationBar.mode.connect("changed", (mode) => group.visible = mode === "buttons");
        group.connect("destroy", () => settings.navigationBar.mode.disconnect(id));
        group.visible = settings.navigationBar.mode.get() === "buttons";
      }
    }));
  }
  buildButtonsRow() {
    const allButtons = [
      { id: "keyboard", icon: Gio2.ThemedIcon.new("input-keyboard-symbolic") },
      { id: "workspace-previous", icon: Gio2.ThemedIcon.new("go-previous-symbolic") },
      { id: "workspace-next", icon: Gio2.ThemedIcon.new("go-next-symbolic") },
      { id: "overview", icon: new AssetIcon("box-outline-symbolic") },
      { id: "apps", icon: new AssetIcon("grid-large-symbolic") },
      { id: "back", icon: new AssetIcon("arrow2-left-symbolic") },
      { id: "spacer", icon: Gio2.ThemedIcon.new("content-loading-symbolic") }
    ];
    const box = new Gtk2.Box({
      orientation: Gtk2.Orientation.VERTICAL,
      spacing: 10
    });
    const leftButtonsBox = new Gtk2.Box({
      orientation: Gtk2.Orientation.HORIZONTAL,
      spacing: 10,
      hexpand: true,
      halign: Gtk2.Align.START,
      cssClasses: ["navigation-bar-page__drop-target"]
    });
    createDropTarget(leftButtonsBox);
    const middleButtonsBox = new Gtk2.Box({
      orientation: Gtk2.Orientation.HORIZONTAL,
      spacing: 10,
      hexpand: true,
      halign: Gtk2.Align.CENTER,
      cssClasses: ["navigation-bar-page__drop-target"]
    });
    createDropTarget(middleButtonsBox);
    const rightButtonsBox = new Gtk2.Box({
      orientation: Gtk2.Orientation.HORIZONTAL,
      spacing: 10,
      hexpand: true,
      halign: Gtk2.Align.END,
      cssClasses: ["navigation-bar-page__drop-target"]
    });
    createDropTarget(rightButtonsBox);
    const unusedBox = new Gtk2.Box({
      orientation: Gtk2.Orientation.HORIZONTAL,
      spacing: 10,
      hexpand: true,
      cssClasses: ["navigation-bar-page__drop-target", "navigation-bar-page__drop-target--unused"]
    });
    createDropTarget(unusedBox);
    const navbarBox = new Gtk2.Box({
      orientation: Gtk2.Orientation.HORIZONTAL,
      spacing: 5
    });
    navbarBox.append(leftButtonsBox);
    navbarBox.append(middleButtonsBox);
    navbarBox.append(rightButtonsBox);
    box.append(new Gtk2.Label({
      label: "Navigation Bar Layout",
      halign: Gtk2.Align.START
    }));
    box.append(new Gtk2.Label({
      label: "Configure your navigation bar layout by dragging buttons into the left, middle or right box.",
      halign: Gtk2.Align.START,
      cssClasses: ["subtitle"]
    }));
    box.append(navbarBox);
    box.append(new Gtk2.Label({
      label: "Place buttons you don't want to use in the navigation bar in this box:",
      halign: Gtk2.Align.START,
      cssClasses: ["subtitle"]
    }));
    box.append(unusedBox);
    function createDragSource(parent, o) {
      const dragSource = new Gtk2.DragSource({
        actions: Gdk.DragAction.MOVE,
        content: Gdk.ContentProvider.new_for_value(o.id),
        propagationPhase: PropagationPhase.CAPTURE
      });
      const but = new Gtk2.Box({
        cssClasses: ["navigation-bar-page__drop-target__item"],
        tooltipText: o.id
      });
      but._optionId = o.id;
      but.append(new Gtk2.Image({ gicon: o.icon }));
      dragSource.connect("drag-end", (_source, drag, delete_data) => {
        if (delete_data && !(o.id === "spacer" && parent === unusedBox))
          parent.remove(but);
        return true;
      });
      but.add_controller(dragSource);
      but.add_controller(new Gtk2.DropTarget());
      return but;
    }
    function createDropTarget(parent) {
      const target = new Gtk2.DropTarget({
        actions: Gdk.DragAction.MOVE,
        formats: Gdk.ContentFormats.new_for_gtype(GObject.TYPE_STRING),
        preload: true,
        propagationPhase: PropagationPhase.CAPTURE
      });
      parent.add_controller(target);
      target.connect("accept", (_source, drop) => true);
      target.connect("drop", (_source, value, x, y) => {
        if (!(value === "spacer" && parent === unusedBox)) {
          let prevChild = null;
          for (let child = parent.get_first_child(); !!child; child = child?.get_next_sibling() ?? null) {
            if (x >= child.get_allocation().x + child.get_allocated_width() / 2) {
              prevChild = child;
            }
          }
          parent.insert_child_after(createDragSource(parent, allButtons.find((b) => b.id === value)), prevChild);
        }
        onChanged();
        return true;
      });
      return target;
    }
    for (let o of settings.navigationBar.buttonsLeft.get()) {
      leftButtonsBox.append(createDragSource(leftButtonsBox, allButtons.find((b) => b.id === o)));
    }
    for (let o of settings.navigationBar.buttonsMiddle.get()) {
      middleButtonsBox.append(createDragSource(middleButtonsBox, allButtons.find((b) => b.id === o)));
    }
    for (let o of settings.navigationBar.buttonsRight.get()) {
      rightButtonsBox.append(createDragSource(rightButtonsBox, allButtons.find((b) => b.id === o)));
    }
    const usedButtons = [
      ...settings.navigationBar.buttonsLeft.get(),
      ...settings.navigationBar.buttonsMiddle.get(),
      ...settings.navigationBar.buttonsRight.get()
    ];
    for (let o of allButtons.filter((b) => !usedButtons.includes(b.id) || b.id === "spacer")) {
      unusedBox.append(createDragSource(unusedBox, o));
    }
    const onChanged = debounce(() => {
      const leftButtons = [], middleButtons = [], rightButtons = [];
      for (let child = leftButtonsBox.get_first_child(); !!child; child = child?.get_next_sibling()) {
        leftButtons.push(child._optionId);
      }
      for (let child = middleButtonsBox.get_first_child(); !!child; child = child?.get_next_sibling()) {
        middleButtons.push(child._optionId);
      }
      for (let child = rightButtonsBox.get_first_child(); !!child; child = child?.get_next_sibling()) {
        rightButtons.push(child._optionId);
      }
      settings.navigationBar.buttonsLeft.set(leftButtons);
      settings.navigationBar.buttonsMiddle.set(middleButtons);
      settings.navigationBar.buttonsRight.set(rightButtons);
    }, 250);
    return new Adw2.PreferencesRow({
      title: "Navigation Bar Buttons",
      child: box,
      cssClasses: ["navigation-bar-page__buttons-bar-layout"]
    });
  }
};

// src/features/preferences/pages/oskKeyPopupPage.ts
import Adw3 from "gi://Adw";
import GObject2 from "gi://GObject";
import Gtk3 from "gi://Gtk";
var OskKeyPopupPage = class extends Adw3.PreferencesPage {
  static {
    GObject2.registerClass(this);
  }
  constructor() {
    super({
      name: "osk-key-popups",
      title: "OSK Key Popups",
      icon_name: "input-keyboard-symbolic"
    });
    this.add(buildPreferencesGroup({
      title: "OSK Key Popups",
      description: "Configure the popups appearing when pressing a button in the On-Screen-Keyboard (OSK).",
      children: [
        buildSwitchRow({
          title: "Enable OSK Key Popups",
          subtitle: "Toggle to enable or disable the OSK key popup feature",
          setting: settings.oskKeyPopups.enabled
        }),
        buildSpinRow({
          title: "Popup Duration",
          subtitle: "Set how long (in milliseconds) to show the OSK key popups.",
          setting: settings.oskKeyPopups.duration,
          adjustment: new Gtk3.Adjustment({
            lower: settings.oskKeyPopups.duration.min,
            upper: settings.oskKeyPopups.duration.max,
            step_increment: 1,
            page_increment: 10
          })
        })
      ]
    }));
  }
};

// src/features/preferences/pages/donationsPage.ts
import Adw4 from "gi://Adw";
import GObject3 from "gi://GObject";
import Gtk4 from "gi://Gtk";
import Gio3 from "gi://Gio";
import Pango from "gi://Pango";
var DonationsPage = class extends Adw4.PreferencesPage {
  static {
    GObject3.registerClass(this);
  }
  constructor() {
    super({
      name: "donations",
      title: "Support",
      icon_name: "emblem-favorite-symbolic"
    });
    this.add(buildPreferencesGroup({
      title: "",
      children: [
        this.buildInfoBox()
      ]
    }));
    this.add(buildPreferencesGroup({
      title: "Support by Donation",
      description: "Choose a platform below to make a donation \u2013 even a dollar a month helps!",
      children: [
        this.buildDonateOptions()
      ]
    }));
    this.add(buildPreferencesGroup({
      title: "Leave Feedback or Ideas",
      description: "Do you like Gnome Touch? Leave your feedback, rating or ideas!",
      children: this.buildFeedbackCards()
    }));
  }
  buildInfoBox() {
    const box = new Gtk4.Box({
      cssClasses: [
        "callout",
        "card"
        /*, 'callout--green'*/
      ],
      orientation: Gtk4.Orientation.VERTICAL
    });
    box.append(new Gtk4.Label({
      cssClasses: ["title-2"],
      halign: Gtk4.Align.START,
      label: randomChoice([
        "The Mission",
        "Why should I donate?",
        "What we believe in"
      ])
    }));
    box.append(new Gtk4.Label({
      wrap: true,
      naturalWrapMode: Gtk4.NaturalWrapMode.WORD,
      hexpand: true,
      label: "Mobile platforms are dominated by big tech oligarchs, and Gnome itself is tough to use on touch devices in everyday life. Gnome Touch helps improve Gnome\u2019s usability on touchscreen devices and makes it a viable, free alternative: tablet users are no longer forced to sacrifice user experience for freedom, control and privacy.\n\nBy donating, you\u2019re supporting a project that helps users regain control over their own devices and gives them another option than profit-driven ecosystems. Every contribution \u2013 be it code, monetary, suggestions or feedback \u2013 helps keep open software such as Gnome Touch competitive and accessible.\n\nThank you for making a difference! \u2764\uFE0F"
    }));
    return box;
  }
  buildDonateOptions() {
    const box = new Gtk4.FlowBox({
      orientation: Gtk4.Orientation.HORIZONTAL,
      column_spacing: 10,
      row_spacing: 10,
      selectionMode: Gtk4.SelectionMode.NONE,
      homogeneous: true
    });
    for (let p of donationPlatforms) {
      const card = new Gtk4.Box({
        cssClasses: ["card", "padding-normal", "donations-page__donation-option"],
        orientation: Gtk4.Orientation.VERTICAL
      });
      card.append(new Gtk4.Label({ label: p.name, cssClasses: ["donations-page__donation-option__title"] }));
      if (p.recommended) {
        card.add_css_class("donations-page__donation-option--recommended");
        card.append(new Gtk4.Label({
          label: "\u2605 recommended",
          halign: Gtk4.Align.CENTER,
          cssClasses: ["donations-page__donation-option__badge"]
        }));
      }
      card.append(new Gtk4.Label({
        label: p.description,
        cssClasses: ["body"],
        wrapMode: Pango.WrapMode.WORD_CHAR,
        wrap: true,
        justify: Gtk4.Justification.CENTER,
        maxWidthChars: 10,
        vexpand: true
      }));
      const btn = new Gtk4.Button({ label: `Use ${p.name}`, halign: Gtk4.Align.CENTER });
      btn.connect("clicked", () => Gio3.AppInfo.launch_default_for_uri(p.url, null));
      card.append(btn);
      box.append(card);
    }
    return box;
  }
  buildFeedbackCards() {
    const res = [];
    for (let r of feedbackPlatforms) {
      const card = new Gtk4.Box({
        cssClasses: ["card", "padding-normal"],
        marginBottom: 10,
        orientation: Gtk4.Orientation.HORIZONTAL
      });
      card.append(new Gtk4.Label({ label: r.title }));
      card.append(new Gtk4.Separator({ hexpand: true, cssClasses: ["spacer"] }));
      const btn = new Gtk4.Button({ label: r.buttonLabel });
      btn.connect("clicked", () => Gio3.AppInfo.launch_default_for_uri(r.url, null));
      card.append(btn);
      res.push(card);
    }
    return res;
  }
};

// src/features/preferences/pages/miscPage.ts
import Adw5 from "gi://Adw";
import GObject4 from "gi://GObject";
var MiscPage = class extends Adw5.PreferencesPage {
  static {
    GObject4.registerClass(this);
  }
  constructor() {
    super({
      name: "misc",
      title: "Other",
      icon_name: "preferences-other-symbolic"
    });
    this.add(buildPreferencesGroup({
      title: "Notification Gestures",
      description: "Notification swipe gestures allow you to expand, collapse or swipe away notifications using your touch screen.",
      children: [
        buildSwitchRow({
          title: "Enable Notification Gestures",
          subtitle: "Toggle to enable or disable the notification gestures feature",
          setting: settings.notificationGestures.enabled
        })
      ]
    }));
    this.add(buildPreferencesGroup({
      title: "Floating Screen Rotate Button",
      description: "When auto-rotate is off, rotating your device triggers a temporary floating button in the corner of the screen, offering a quick way to adjust orientation.",
      children: [
        buildSwitchRow({
          title: "Enable Floating Screen Rotate Button",
          subtitle: "Toggle to enable or disable the floating screen rotate button feature",
          setting: settings.screenRotateUtils.floatingScreenRotateButtonEnabled
        })
      ]
    }));
  }
};

// src/features/preferences/prefs.ts
import { ExtensionPreferences } from "resource:///org/gnome/Shell/Extensions/js/extensions/prefs.js";
import Gtk5 from "gi://Gtk";
import Gdk2 from "gi://Gdk";
import Gio4 from "gi://Gio";
var GnomeTouchPreferences = class extends ExtensionPreferences {
  async fillPreferencesWindow(window) {
    initSettings(this.getSettings());
    const assets = Gio4.resource_load(this.dir.get_child(assetsGResourceFile).get_path());
    Gio4.resources_register(assets);
    this.loadCss();
    Gtk5.Settings.get_default()?.connect("notify::gtk-application-prefer-dark-theme", () => this.loadCss());
    const pages = [
      new NavigationBarPage(),
      new OskKeyPopupPage(),
      new MiscPage(),
      new DonationsPage()
    ];
    pages.forEach((p) => window.add(p));
    const initialPage = settings.initialPreferencesPage.get();
    if (pages.some((p) => p.name == initialPage)) {
      window.visiblePageName = initialPage;
    }
    settings.initialPreferencesPage.set("default");
    window.searchEnabled = true;
  }
  loadCss() {
    const display = Gdk2.Display.get_default();
    const settings2 = Gtk5.Settings.get_for_display(display);
    const cssProvider = new Gtk5.CssProvider();
    const cssFile = this.dir.get_child(
      settings2.gtk_application_prefer_dark_theme ? "prefs-dark.css" : "prefs-light.css"
    );
    cssProvider.load_from_file(cssFile);
    Gtk5.StyleContext.add_provider_for_display(
      display,
      cssProvider,
      Gtk5.STYLE_PROVIDER_PRIORITY_APPLICATION
    );
  }
};

// src/prefs.ts
var prefs_default = GnomeTouchPreferences;
export {
  prefs_default as default
};
