/**
 * The code in this file is heavily based on Marcin Jahn's Gnome extension to hide audio devices from panel.
 *
 * The original code can be found at https://github.com/marcinjahn/gnome-quicksettings-audio-devices-hider-extension/tree/main
 * Original code is licensed under the MIT license (https://github.com/marcinjahn/gnome-quicksettings-audio-devices-hider-extension/blob/main/LICENSE)
 */
import Gvc from 'gi://Gvc';
import * as Volume from 'resource:///org/gnome/shell/ui/status/volume.js';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import { DeviceType } from "./deviceSettings.js";
import { delay, range } from "./utils.js";
export var Action;
(function (Action) {
    Action["ADDED"] = "ADDED";
    Action["REMOVED"] = "REMOVED";
})(Action || (Action = {}));
export class MixerSource {
    async getMixer() {
        const mixer = Volume.getMixerControl();
        await waitForMixerToBeReady(mixer);
        await delay(200);
        return new Mixer(mixer, () => { });
    }
}
async function waitForMixerToBeReady(mixer) {
    while (mixer.get_state() === Gvc.MixerControlState.CONNECTING) {
        await delay(200);
    }
    const state = mixer.get_state();
    if (state === Gvc.MixerControlState.FAILED) {
        throw new Error("MixerControl is in a failed state");
    }
    else if (state === Gvc.MixerControlState.CLOSED) {
        throw new Error("MixerControl is in a closed state");
    }
}
export class Mixer {
    control;
    disposal;
    constructor(control, disposal) {
        this.control = control;
        this.disposal = disposal;
    }
    getAllDevices(type) {
        const quickSettings = Main.panel.statusArea.quickSettings;
        const devices = type === DeviceType.OUTPUT
            ? quickSettings._volumeOutput._output._deviceItems
            : quickSettings._volumeInput._input._deviceItems;
        const ids = Array.from(devices, ([id]) => id);
        return this.getAudioDevicesFromIds(ids, type);
    }
    /**
     * Generate a name similar to https://gitlab.gnome.org/GNOME/gnome-shell/-/blob/main/js/ui/status/volume.js#L132
     *
     * @param device Gvc lookup value
     * @private
     */
    constructDeviceName(device) {
        return device.origin ? `${device.description} - ${device.origin}` : device.description;
    }
    getAudioDevicesFromIds(ids, type) {
        return this.getUIDevicesFromIds(ids, type).map(device => {
            return { id: device.get_id(), name: this.constructDeviceName(device) };
        });
    }
    getUIDevicesFromIds(ids, type) {
        return ids.map((id) => {
            const lookup = type === DeviceType.OUTPUT
                ? this.control.lookup_output_id(id)
                : this.control.lookup_input_id(id);
            return lookup;
        });
    }
    getDefaultOutput() {
        const stream = this.control.get_default_sink();
        return this.constructDeviceName(this.control.lookup_device_from_stream(stream));
    }
    getDefaultInput() {
        const stream = this.control.get_default_source();
        return this.constructDeviceName(this.control.lookup_device_from_stream(stream));
    }
    /**
     * Set output device. First, try by id. If id not found, try finding it with name.
     *
     * @param id device id
     * @param name display name
     * @returns true if device changed, false if no device found with this name
     */
    setOutput(id, name) {
        let device = this.control.lookup_output_id(id);
        if (!device) {
            const deviceByName = this.getDeviceFromName(name, DeviceType.OUTPUT);
            if (deviceByName) {
                device = deviceByName;
            }
        }
        if (device) {
            this.control.change_output(device);
            return true;
        }
        else {
            return false;
        }
    }
    /**
     * Set input device. First, try by id. If id not found, try finding it with name.
     *
     * @param id device id
     * @param name display name
     * @returns true if device changed, false if no device found with this name
     */
    setInput(id, name) {
        let device = this.control.lookup_input_id(id);
        if (!device) {
            const deviceByName = this.getDeviceFromName(name, DeviceType.INPUT);
            if (deviceByName) {
                device = deviceByName;
            }
        }
        if (device) {
            this.control.change_input(device);
            return true;
        }
        else {
            return false;
        }
    }
    /**
     * Uses a Dummy Device "trick" from
     * https://github.com/kgshank/gse-sound-output-device-chooser/blob/master/sound-output-device-chooser@kgshank.net/base.js#LL299C20-L299C20
     * @param name display name
     * @param type device type
     * @returns mixer stream
     */
    getDeviceFromName(name, type) {
        const dummyDevice = new Gvc.MixerUIDevice();
        const devices = this.getUIDevicesFromIds(range(dummyDevice.get_id()), type);
        console.info(devices);
        return devices.find(d => d !== null && this.constructDeviceName(d) === name);
    }
    subscribeToDeviceChanges(callback) {
        const addOutputId = this.control.connect("output-added", (_, deviceId) => callback({ deviceId, type: DeviceType.OUTPUT, action: Action.ADDED }));
        const removeOutputId = this.control.connect("output-removed", (_, deviceId) => callback({ deviceId, type: DeviceType.OUTPUT, action: Action.REMOVED }));
        const addInputId = this.control.connect("input-added", (_, deviceId) => callback({ deviceId, type: DeviceType.INPUT, action: Action.ADDED }));
        const removeInputId = this.control.connect("input-removed", (_, deviceId) => callback({ deviceId, type: DeviceType.INPUT, action: Action.REMOVED }));
        return { ids: [addOutputId, removeOutputId, addInputId, removeInputId] };
    }
    unsubscribe(subscription) {
        subscription.ids.forEach((id) => {
            this.control.disconnect(id);
        });
    }
    dispose() {
        this.disposal();
    }
}
