import GObject from "gi://GObject";
import { ManagedConnects } from "./managed-connects.js";
import { getMonitorIdentifier } from "./monitors.js";
class Profile {
    #monitors;
    name;
    fontScaleFactor;
    dashToDockIconSize;
    #monitorsKeysSet;
    constructor({ monitors, name, fontScaleFactor, dashToDockIconSize }) {
        this.#monitors = monitors;
        this.name = name;
        this.fontScaleFactor = fontScaleFactor;
        this.dashToDockIconSize = dashToDockIconSize;
        const profileMonitorIdentifiers = getMonitorIdentifier(this.#monitors);
        this.#monitorsKeysSet = new Set(getMonitorIdentifier(this.#monitors));
    }
    getMonitorsCount() {
        return this.#monitors.length;
    }
    matchesCurrentMonitors(currentMonitorIdentifiers) {
        if (currentMonitorIdentifiers.length !== this.#monitorsKeysSet.size) {
            return false;
        }
        return currentMonitorIdentifiers.every((item) => this.#monitorsKeysSet.has(item));
    }
    toJSON() {
        return {
            name: this.name,
            fontScaleFactor: this.fontScaleFactor,
            dashToDockIconSize: this.dashToDockIconSize,
            monitors: this.#monitors,
        };
    }
}
class ProfilesManagerImpl extends GObject.Object {
    #settings;
    #monitorsConfig;
    #profiles = [];
    #activeProfile;
    #logger;
    #connections = new ManagedConnects();
    constructor(settings, monitorsConfig, logger) {
        super();
        this.#settings = settings;
        this.#monitorsConfig = monitorsConfig;
        this.#logger = logger;
        const onUpdate = this.#onUpdate.bind(this);
        this.#connections.connect(this.#settings, "changed", onUpdate);
        this.#connections.connect(this.#monitorsConfig, "updated", onUpdate);
        this.#onUpdate();
    }
    disconnectAll() {
        this.#connections.disconnectAll();
        this.#monitorsConfig.disconnectAll();
    }
    getActiveProfile() {
        if (!this.#activeProfile) {
            const currentMonitorIdentifiers = getMonitorIdentifier(this.#monitorsConfig.monitors);
            this.#activeProfile = this.#profiles.find((profile) => profile.matchesCurrentMonitors(currentMonitorIdentifiers));
        }
        return this.#activeProfile;
    }
    getProfiles() {
        return this.#profiles;
    }
    appendProfile(profile) {
        this.#profiles.push(new Profile(profile));
        this.#syncProfiles();
    }
    updateProfile(idx, profileUpd) {
        Object.assign(this.#profiles[idx], profileUpd);
        this.#syncProfiles();
    }
    removeProfile(idx) {
        this.#profiles.splice(idx, 1);
        this.#syncProfiles();
    }
    #onUpdate(updateProfilesList = true) {
        this.#activeProfile = null;
        updateProfilesList && this.#fetchProfiles();
    }
    #fetchProfiles() {
        try {
            this.#profiles = (JSON.parse(this.#settings.get_string("profiles"))?.profiles ?? []).map(
            // biome-ignore lint/suspicious/noExplicitAny: no schema parsing for now
            (rawProfile) => new Profile(rawProfile));
            this.emit("updated");
        }
        catch (error) {
            this.#logger.error("Invalid profile configuration", String(error));
        }
    }
    #syncProfiles() {
        this.#settings.set_string("profiles", JSON.stringify({ profiles: this.#profiles }));
        // no need to emit update: it will be done by updating settings already
    }
}
const ProfilesManagerRegistered = GObject.registerClass({
    Signals: {
        updated: {},
    },
}, ProfilesManagerImpl);
export function getProfileManager(settings, monitorsConfig, logger) {
    return new ProfilesManagerRegistered(settings, monitorsConfig, logger);
}
