// Window Pinner Extension for GNOME Shell
// Maintains and restores window positions after screen unlock or session restart.
// Author: David Harker
// License: MIT

import St from 'gi://St'; // (May be unused, but kept for possible future UI)
import { Extension } from 'resource:///org/gnome/shell/extensions/extension.js';
import GLib from 'gi://GLib';
import Gio from 'gi://Gio';
import { CHECK_INTERVAL_MIN, CHECK_INTERVAL_MAX, DELAY_AFTER_UNLOCK_MIN, DELAY_AFTER_UNLOCK_MAX } from './constants.js';
import * as ExtensionUtils from 'resource:///org/gnome/shell/misc/extensionUtils.js';

// --- Utility: Lazy-load Main UI object for compatibility with GNOME Shell ---
let _Main = null;
async function getMain() {
    if (_Main) return _Main;
    if (typeof global !== 'undefined' && global.Main) {
        _Main = global.Main;
        return _Main;
    }
    // Fallback for test/mocking environments
    _Main = (await import('resource:///org/gnome/shell/ui/main.js')).default || (await import('resource:///org/gnome/shell/ui/main.js'));
    return _Main;
}

/**
 * Main extension class. Handles window state tracking and restoration.
 */
export default class WindowPinnerExtension extends Extension {
    constructor(meta) {
        super(meta);
        // Allow dependency injection for testing
        this._global = (meta && meta._global) || global;
        this._GLib = (meta && meta._GLib) || GLib;
        this._Main = (meta && meta._Main) || null;
        // Autoload schema from extension's own schemas directory
        if (meta && meta._Settings) {
            this._settings = new meta._Settings();
        } else {
            try {
                const schemaDir = (meta && meta.dir)
                    ? meta.dir.get_child('schemas').get_path()
                    : null;
                const schemaSource = Gio.SettingsSchemaSource.new_from_directory(
                    schemaDir,
                    Gio.SettingsSchemaSource.get_default(),
                    false
                );
                const schemaObj = schemaSource.lookup('org.gnome.shell.extensions.window-pinner', true);
                if (!schemaObj) {
                    log('[window-pinner] Schema not found in schemas directory!');
                    throw new Error('Schema org.gnome.shell.extensions.window-pinner could not be found in schemas directory');
                }
                this._settings = new Gio.Settings({ settings_schema: schemaObj });
            } catch (e) {
                log(`[window-pinner] Falling back to schema_id. Error: ${e}`);
                // Fallback for test/mocking environments or if meta.dir is not available
                this._settings = new Gio.Settings({ schema_id: 'org.gnome.shell.extensions.window-pinner' });
            }
        }
    }

    /**
     * Called when the extension is enabled.
     * Loads previous window state if available, otherwise starts periodic tracking.
     */
    async enable() {
        if (!this._Main) this._Main = await getMain();
        this._windowInfoLog = [];
        this._cachedState = null;
        const stateFile = this._getStateFilePath();
        let [exists, contents] = this._GLib.file_test(stateFile, this._GLib.FileTest.EXISTS)
            ? this._GLib.file_get_contents(stateFile)
            : [false, null];
        if (exists && contents) {
            try {
                const json = JSON.parse(imports.byteArray.toString(contents));
                this._cachedState = this.previousWindowLayout = json;
                this._windowInfoLog.push(json);
                this._waitForMonitorMatch();
                return;
            } catch (e) {
                logError(new Error('Failed to parse window state file'));
                logError(e);
            }
        }
        this._startMainTimer();
    }

    /**
     * Starts the main periodic timer for window state tracking.
     */
    _startMainTimer() {
        let interval = this._clampSetting(this._settings.get_int('check-interval'));
        log(`[window-pinner] Watching for window changes every ${interval} seconds`);
        this._timerId = this._GLib.timeout_add_seconds(
            this._GLib.PRIORITY_DEFAULT,
            interval,
            this._mainTimerHandler.bind(this)
        );
    }

    /**
     * Timer handler: records current window info.
     */
    _mainTimerHandler() {
        this._recordWindowInfo();
        return true; // Continue timer
    }

    /**
     * Waits for the monitor configuration to match the previous session before restoring windows.
     */
    _waitForMonitorMatch() {
        const lastInfo = this.previousWindowLayout;
        if (!lastInfo) {
            this._startMainTimer();
            return;
        }
        const expectedMonitors = lastInfo.numScreens;
        this._monitorPollId = this._GLib.timeout_add_seconds(this._GLib.PRIORITY_DEFAULT, 2, () => {
            const currentMonitors = this._global.display.get_n_monitors();
            if (currentMonitors === expectedMonitors) {
                this._GLib.source_remove(this._monitorPollId);
                this._monitorPollId = null;
                const unlockDelay = this._clampSetting(this._settings.get_int('delay-after-unlock'), 'delay-after-unlock');
                this._GLib.timeout_add_seconds(this._GLib.PRIORITY_DEFAULT, unlockDelay, () => {
                    this._restoreWindowLayout();
                    this._startMainTimer();
                    return false;
                });
                return false;
            }
            return true;
        });
    }

    /**
     * Restores window positions and sizes from the previous session.
     */
    _restoreWindowLayout() {
        if (!this.previousWindowLayout || !this.previousWindowLayout.windows) return;
        const currentWindows = this._global.get_window_actors()
            .map(actor => actor.meta_window)
            .filter(Boolean);

        // Map PIDs to windows for matching
        const pidMap = {};
        for (const win of currentWindows) {
            const pid = win.get_pid();
            pidMap[pid] = win;
        }

        let movedCount = 0;
        for (const savedWin of this.previousWindowLayout.windows) {
            const win = pidMap[savedWin.pid];
            if (win) {
                win.move_resize_frame(false, savedWin.x, savedWin.y, savedWin.width, savedWin.height);
                movedCount++;
            }
        }
        log(`[window-pinner] Moved ${movedCount} windows to original locations`);
    }

    /**
     * Checks if two window info snapshots are materially different.
     */
    _windowInfoMateriallyDiffers(a, b) {
        if (!a || !b) return true;
        if (a.numScreens !== b.numScreens) return true;
        if (!Array.isArray(a.windows) || !Array.isArray(b.windows)) return true;
        if (a.windows.length !== b.windows.length) return true;
        for (let i = 0; i < a.windows.length; i++) {
            const wa = a.windows[i], wb = b.windows[i];
            if (!wa || !wb) return true;
            if (wa.pid !== wb.pid) return true;
            if (wa.x !== wb.x || wa.y !== wb.y || wa.width !== wb.width || wa.height !== wb.height) return true;
        }
        return false;
    }

    /**
     * Records the current window info (positions, sizes, command lines).
     * Saves to disk if the state has changed.
     */
    async _recordWindowInfo() {
        const info = {
            timestamp: Date.now(),
            numScreens: this._global.display.get_n_monitors(),
            windows: []
        };
        const windowActors = this._global.get_window_actors();
        for (const actor of windowActors) {
            const metaWindow = actor.meta_window;
            if (!metaWindow) continue;
            const rect = metaWindow.get_frame_rect();
            const pid = metaWindow.get_pid();
            const cmdline = await this._getCmdline(pid);
            info.windows.push({
                x: rect.x,
                y: rect.y,
                width: rect.width,
                height: rect.height,
                cmdline,
                pid // <--- Add PID to recorded info
            });
        }
        this._windowInfoLog.push(info);
        if (this._windowInfoLog.length > 2)
            this._windowInfoLog = this._windowInfoLog.slice(-2);

        if (this._windowInfoMateriallyDiffers(info, this._cachedState)) {
            const stateFile = this._getStateFilePath();
            try {
                this._GLib.file_set_contents(stateFile, JSON.stringify(info));
                this._cachedState = info;
            } catch (_) {
                logError(new Error('[window-pinner] Failed to write window state file'));
            }
        }
    }

    /**
     * Reads the command line for a process by PID.
     */
    async _getCmdline(pid) {
        try {
            let [ok, contents] = this._GLib.file_get_contents(`/proc/${pid}/cmdline`);
            if (ok)
                return imports.byteArray.toString(contents).replace(/\0/g, ' ');
        } catch (_) {}
        return '';
    }

    /**
     * Called when the extension is disabled. Cleans up timers and saves state.
     */
    disable() {
        // No panel button to destroy
        if (this._timerId) {
            this._GLib.source_remove(this._timerId);
            this._timerId = null;
        }
        if (this._monitorPollId) {
            this._GLib.source_remove(this._monitorPollId);
            this._monitorPollId = null;
        }
        const stateFile = this._getStateFilePath();
        try {
            this._GLib.file_set_contents(stateFile, JSON.stringify(this._windowInfoLog[this._windowInfoLog.length - 1] || {}));
        } catch (_) {
            // Ignore write errors
        }
    }

    /**
     * Returns the path to the state file used for saving window info.
     */
    _getStateFilePath() {
        return `${this._GLib.get_user_runtime_dir()}/window-pinner-state.json`;
    }

    /**
     * Clamps a setting value to the allowed range.
     * @param {number} val - The value to clamp.
     * @param {string} type - The setting type ('check-interval' or 'delay-after-unlock').
     */
    _clampSetting(val, type = 'check-interval') {
        if (type === 'delay-after-unlock') {
            return Math.max(DELAY_AFTER_UNLOCK_MIN, Math.min(val, DELAY_AFTER_UNLOCK_MAX));
        }
        return Math.max(CHECK_INTERVAL_MIN, Math.min(val, CHECK_INTERVAL_MAX));
    }
}