/* extension.js
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

import GObject from "gi://GObject";
import St from "gi://St";

import { Extension } from "resource:///org/gnome/shell/extensions/extension.js";
import * as PanelMenu from "resource:///org/gnome/shell/ui/panelMenu.js";
import * as PopupMenu from "resource:///org/gnome/shell/ui/popupMenu.js";
import * as Main from "resource:///org/gnome/shell/ui/main.js";

const Indicator = GObject.registerClass(
  class Indicator extends PanelMenu.Button {
    _init() {
      super._init(0.0, "Strings Indicator");
      this._icon = new St.Icon({
        icon_name: "edit-copy-symbolic",
        style_class: "system-status-icon",
      });
      this.add_child(this._icon);

      // Keep references to dynamic menu items
      this.dynamicMenuItems = [];
      this._updateTimer = null;

      // For random strings
      this._words =
        "lorem ipsum dolor sit amet consectetur adipiscing elit sed do eiusmod tempor incididunt ut labore et dolore magna aliqua velit egestas dui id ornare arcu odio sem nulla risus ultricies tristique aliquet enim tortor at nibh pulvinar proin gravida hendrerit lectus a vulputate cursus euismod quis viverra cras lobortis scelerisque fermentum faucibus in dictumst vestibulum rhoncus est pellentesque blandit ultrices mi tempus pharetra diam nisl suscipit";

      // Connect to menu open event to refresh dynamic items
      this.menu.connect("open-state-changed", (_, isOpen) => {
        if (isOpen) {
          this._updateDynamicMenuItems();
          this._startPeriodicUpdates();
          if (this.dateSection) {
            this.dateSection.menu.open(true);
          }
        } else {
          this._stopPeriodicUpdates();
        }
      });
      this.menu.actor.add_style_class_name("strings-menu");

      // ===== DATE SECTION =====
      this.dateSection = new PopupMenu.PopupSubMenuMenuItem("📅 Date & Time");
      this.menu.addMenuItem(this.dateSection);
      this.dateSection.menu.actor.add_style_class_name("strings-menu");

      // Date full
      this.dateFullItem = this._createDynamicMenuItem(
        () => this._getFullDateString(),
        3000,
      );
      this.dateSection.menu.addMenuItem(this.dateFullItem);

      // Epoch
      this.epochItem = this._createDynamicMenuItem(
        () => this._getEpochString(),
        123,
      );
      this.dateSection.menu.addMenuItem(this.epochItem);

      // Date ISO 8601
      this.dateIsoItem = this._createDynamicMenuItem(
        () => this._getIsoDateString(),
        123,
      );
      this.dateSection.menu.addMenuItem(this.dateIsoItem);

      // Date YYYY-MM-DD
      this.dateYmdItem = this._createDynamicMenuItem(
        () => this._getYmdDateString(),
        3000,
      );
      this.dateSection.menu.addMenuItem(this.dateYmdItem);

      // ===== RANDOM SECTION =====
      this.randomSection = new PopupMenu.PopupSubMenuMenuItem(
        "🎲 Hashes & IDs",
      );
      this.menu.addMenuItem(this.randomSection);
      this.randomSection.menu.actor.add_style_class_name("strings-menu");

      // UUID
      this.uuidItem = this._createDynamicMenuItem(
        () => this._generateUuid(),
        3000,
      );
      this.randomSection.menu.addMenuItem(this.uuidItem);

      // Hash
      this.hashItem = this._createDynamicMenuItem(
        () => this._generateHash(),
        3000,
      );
      this.randomSection.menu.addMenuItem(this.hashItem);

      // Passphrase
      this.passPhraseItem = this._createDynamicMenuItem(
        () => this._generatePassPhrase(),
        3000,
      );
      this.randomSection.menu.addMenuItem(this.passPhraseItem);

      // ===== COLOR SECTION =====
      this.colorSection = new PopupMenu.PopupSubMenuMenuItem("🌈 Random Color");
      this.menu.addMenuItem(this.colorSection);
      this.colorSection.menu.actor.add_style_class_name("strings-menu");

      // CSS Color Name
      this.colorNameItem = this._createDynamicMenuItem(
        () => this._generateCssColorName(),
        3000,
      );
      this.colorSection.menu.addMenuItem(this.colorNameItem);

      // HEX Color
      this.colorHexItem = this._createDynamicMenuItem(
        () => this._generateColorHex(),
        3000,
      );
      this.colorSection.menu.addMenuItem(this.colorHexItem);

      // RGB Color
      this.colorRgbItem = this._createDynamicMenuItem(
        () => this._generateRgbColor(),
        3000,
      );
      this.colorSection.menu.addMenuItem(this.colorRgbItem);

      // HSL Color
      this.colorHslItem = this._createDynamicMenuItem(
        () => this._generateHslColor(),
        3000,
      );
      this.colorSection.menu.addMenuItem(this.colorHslItem);

      // LCH Color
      this.colorLchItem = this._createDynamicMenuItem(
        () => this._generateLchColor(),
        3000,
      );
      this.colorSection.menu.addMenuItem(this.colorLchItem);

      // ===== LOREM IPSUM SECTION =====
      this.loremSection = new PopupMenu.PopupSubMenuMenuItem("📑 Lorem Ipsum");
      this.menu.addMenuItem(this.loremSection);

      // Add CSS Class
      this.loremSection.menu.actor.add_style_class_name("strings-menu");

      // Title
      this.titleItem = this._createMenuItem("Title", () => {
        this.copyString(this._generateTitle());
      });
      this.loremSection.menu.addMenuItem(this.titleItem);

      // Sentence
      this.sentenceItem = this._createMenuItem("Sentence", () => {
        this.copyString(this._generateSentence());
      });
      this.loremSection.menu.addMenuItem(this.sentenceItem);

      // Paragraph
      this.paragraphItem = this._createMenuItem("Paragraph", () => {
        this.copyString(this._generateParagraph());
      });
      this.loremSection.menu.addMenuItem(this.paragraphItem);

      // Initial update of dynamic items
      this._updateDynamicMenuItems();
    }

    // Helper to create static menu items
    _createMenuItem(label, handler) {
      const item = new PopupMenu.PopupMenuItem(label);
      item.connect("activate", handler);
      item.label.add_style_class_name("menu-item-label");
      return item;
    }

    // Helper to create dynamic menu items
    _createDynamicMenuItem(labelFn, interval) {
      const item = new PopupMenu.PopupMenuItem("");
      const dynamicItem = {
        item,
        labelFn,
        currentValue: null,
        interval,
        timerId: null,
      };
      this.dynamicMenuItems.push(dynamicItem);

      item.connect("activate", () => {
        if (dynamicItem.currentValue) {
          this.copyString(dynamicItem.currentValue);
        }
      });

      item.label.add_style_class_name("menu-item-label");
      return item;
    }

    // Update a single dynamic item
    _updateSingleDynamicItem(dynamicItem) {
      dynamicItem.currentValue = dynamicItem.labelFn();
      dynamicItem.item.label.text = dynamicItem.currentValue;
    }

    // Update all dynamic items
    _updateDynamicMenuItems() {
      this.dynamicMenuItems.forEach((dynamicItem) => {
        this._updateSingleDynamicItem(dynamicItem);
      });
    }

    // Start individual timers for each dynamic item
    _startPeriodicUpdates() {
      this.dynamicMenuItems.forEach((dynamicItem) => {
        if (dynamicItem.interval > 0) {
          dynamicItem.timerId = setInterval(() => {
            this._updateSingleDynamicItem(dynamicItem);
          }, dynamicItem.interval);
        }
      });
    }

    // Stop all individual timers
    _stopPeriodicUpdates() {
      this.dynamicMenuItems.forEach((dynamicItem) => {
        if (dynamicItem.timerId) {
          clearInterval(dynamicItem.timerId);
          dynamicItem.timerId = null;
        }
      });
    }

    // Date formatters
    _getFullDateString() {
      return new Date().toLocaleString("en-US", {
        weekday: "long",
        day: "numeric",
        month: "long",
        year: "numeric",
      });
    }

    _getYmdDateString() {
      const now = new Date();
      const offsetDate = new Date(
        now.getTime() - now.getTimezoneOffset() * 60000,
      );
      return offsetDate.toISOString().split("T")[0];
    }

    _getIsoDateString() {
      return new Date().toISOString();
    }

    _getEpochString() {
      return Date.now().toString();
    }

    _generateTitle() {
      const words = this._words.split(" ");
      words.sort(() => Math.random() - 0.5);
      const capitalizedWords = words.map((word) => {
        if (word.length === 0) return "";
        return word.charAt(0).toUpperCase() + word.slice(1).toLowerCase();
      });
      const randomWordsCount = Math.floor(Math.random() * 3 + 2);
      return capitalizedWords.slice(0, randomWordsCount).join(" ");
    }

    _generateSentence() {
      const words = this._words.split(" ");
      words.sort(() => Math.random() - 0.5);
      words[0] = words[0].charAt(0).toUpperCase() + words[0].slice(1);
      const randomWordsCount = Math.floor(Math.random() * 4 + 8);
      return words.slice(0, randomWordsCount).join(" ") + ".";
    }

    _generateParagraph() {
      const sentenceCount = 3 + Math.floor(Math.random() * 3);
      let paragraph = "";
      for (let i = 0; i < sentenceCount; i++) {
        paragraph += this._generateSentence() + " ";
      }
      return paragraph.trim();
    }

    _generateUuid() {
      const uuid = "xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx".replace(
        /[xy]/g,
        (c) => {
          const r = (Math.random() * 16) | 0;
          return (c === "x" ? r : (r & 0x3) | 0x8).toString(16);
        },
      );
      return uuid;
    }

    _generateHash() {
      const chars =
        "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";
      const hash = Array.from({ length: 16 }, () =>
        chars.charAt((Math.random() * chars.length) | 0),
      ).join("");
      return hash;
    }

    _generatePassPhrase() {
      const words =
        "Aardvark Abacus Abbey Acorn Adagio Aerial Agenda Alchemy Almanac Alpaca Amber Anagram Anchor Anvil Apex Apogee Arbor Argon Armada Aroma Arsenal Asteroid Atlas Aurora Avalanche Aztec Bamboo Banjo Baroque Basilisk Bastion Beacon Belfry Beryl Bivouac Blizzard Bonanza Bonsai Boomerang Bouquet Bravado Bronco Bungalow Cabana Cameo Canyon Caravan Cascade Celestial Ceramic Chalet Chasm Citadel Cobalt Comet Coral Cortex Crescendo Crimson Crystal Delta Denim Desert Dewdrop Diadem Diesel Diorama Dolomite Dragon Dynamo Eclipse Elysium Emerald Enigma Epitome Equinox Ethereal Exodus Falcon Fandango Fjord Flotilla Folio Forge Fresco Galaxy Gazebo Geode Ginseng Glacier Gossamer Granary Grotto Halcyon Harbor Haven Helix Hinterland Horizon Hydra Hypnosis Igloo Inferno Iridium Isotope Ivory Jade Jamboree Jasmine Javelin Jubilee Jungle Kaleido Karate Karma Kestrel Kiosk Krypton Lagoon Lariat Legacy Lemon Lilac Limbo Lotus Lunar Macaw Maelstrom Magnet Majesty Mango Mantra Marble Meadow Meridian Mirage Monsoon Mosaic Mystic Nadir Nebula Nectar Nimbus Nirvana Nocturne Nomad Nova Obsidian Ocelot Octagon Odyssey Onyx Oracle Orbit Orchid Origami Overture Ozone Pagoda Paladin Panacea Paragon Paradox Pavilion Pendulum Penumbra Peregrine Phantom Phoenix Pinnacle Pirate Plasma Plume Polaris Prairie Prism Proton Pueblo Pyramid Quagmire Quantum Quasar Quicksilver Quiver Radial Radiance Raven Rebel Renaissance Reptile Resonance Rhapsody Riddle Ripple Rococo Rogue Rubicon Rune Sable Saffron Saga Salamander Sapphire Satire Savanna Scarab Scepter Sequoia Serenade Shogun Sierra Silhouette Sirocco Solstice Sonata Sphinx Stardust Stratosphere Sultan Summit Sundial Symphony Talisman Tarot Tempest Terra Thunder Titan Topaz Torrent Tranquility Triton Tundra Typhoon Utopia Vagabond Valhalla Vanguard Vapor Vortex Voyage Warlock Zenith Zephyr Zodiac";
      const availableWords = words.split(" ");
      const selected = [];

      // Select 3 distinct words
      for (let i = 0; i < 3; i++) {
        const randomIndex = Math.floor(Math.random() * availableWords.length);
        selected.push(availableWords.splice(randomIndex, 1)[0]);
      }

      // Generate random digit (0-9)
      const randomDigit = Math.floor(Math.random() * 10);

      // Choose which word gets the digit
      const digitPosition = Math.floor(Math.random() * 3);

      // Apply digit to selected position
      const randomWords = selected.map((word, index) =>
        index === digitPosition ? word + randomDigit : word,
      );

      // Add the # between the words
      return randomWords.join("-");
    }

    _generateColorHex() {
      const letters = "0123456789abcdef";
      const color =
        "#" +
        Array.from({ length: 6 }, () =>
          letters.charAt((Math.random() * 16) | 0),
        ).join("");
      return color;
    }

    _generateCssColorName() {
      const colorNames =
        "AliceBlue AntiqueWhite Aqua Aquamarine Azure Beige Bisque Black BlanchedAlmond Blue BlueViolet Brown BurlyWood CadetBlue Chartreuse Chocolate Coral CornflowerBlue Cornsilk Crimson Cyan DarkBlue DarkCyan DarkGoldenRod DarkGray DarkGrey DarkGreen DarkKhaki DarkMagenta DarkOliveGreen Darkorange DarkOrchid DarkRed DarkSalmon DarkSeaGreen DarkSlateBlue DarkSlateGray DarkSlateGrey DarkTurquoise DarkViolet DeepPink DeepSkyBlue DimGray DimGrey DodgerBlue FireBrick FloralWhite ForestGreen Fuchsia Gainsboro GhostWhite Gold GoldenRod Gray Grey Green GreenYellow HoneyDew HotPink IndianRed Indigo Ivory Khaki Lavender LavenderBlush LawnGreen LemonChiffon LightBlue LightCoral LightCyan LightGoldenRodYellow LightGray LightGrey LightGreen LightPink LightSalmon LightSeaGreen LightSkyBlue LightSlateGray LightSlateGrey LightSteelBlue LightYellow Lime LimeGreen Linen Magenta Maroon MediumAquaMarine MediumBlue MediumOrchid MediumPurple MediumSeaGreen MediumSlateBlue MediumSpringGreen MediumTurquoise MediumVioletRed MidnightBlue MintCream MistyRose Moccasin NavajoWhite Navy OldLace Olive OliveDrab Orange OrangeRed Orchid PaleGoldenRod PaleGreen PaleTurquoise PaleVioletRed PapayaWhip PeachPuff Peru Pink Plum PowderBlue Purple Red RosyBrown RoyalBlue SaddleBrown Salmon SandyBrown SeaGreen SeaShell Sienna Silver SkyBlue SlateBlue SlateGray SlateGrey Snow SpringGreen SteelBlue Tan Teal Thistle Tomato Turquoise Violet Wheat White WhiteSmoke Yellow YellowGreen";
      const colors = colorNames.split(" ");
      return colors[Math.floor(Math.random() * colors.length)];
    }

    _generateRgbColor() {
      const r = Math.floor(Math.random() * 256);
      const g = Math.floor(Math.random() * 256);
      const b = Math.floor(Math.random() * 256);
      return `rgb(${r}, ${g}, ${b})`;
    }

    _generateHslColor() {
      const h = Math.floor(Math.random() * 360);
      const s = Math.floor(Math.random() * 101);
      const l = Math.floor(Math.random() * 101);
      return `hsl(${h}, ${s}%, ${l}%)`;
    }

    _generateLchColor() {
      const l = Math.floor(Math.random() * 101);
      const c = Math.floor(Math.random() * 101);
      const h = Math.floor(Math.random() * 360);
      return `lch(${l}% ${c} ${h})`;
    }

    destroy() {
      this._stopPeriodicUpdates();
      super.destroy();
    }

    copyString(text) {
      St.Clipboard.get_default().set_text(St.ClipboardType.CLIPBOARD, text);

      const originalIcon = this._icon.icon_name;
      this._icon.icon_name = "check-plain-symbolic";

      setTimeout(() => {
        this._icon.icon_name = originalIcon;
      }, 1500);
    }
  },
);

export default class StringsExtension extends Extension {
  enable() {
    this._indicator = new Indicator();
    Main.panel.addToStatusArea(this.uuid, this._indicator);
  }

  disable() {
    this._indicator.destroy();
    this._indicator = null;
  }
}
