import GLib from 'gi://GLib';
import Gio from 'gi://Gio';
export default class FileHelpers {
    static getDefaultConfigPath(presetDir = false) {
        const homeDir = GLib.get_home_dir();
        const relPath = [homeDir, '.config', 'input-remapper-2'];
        return GLib.build_filenamev(presetDir ? [...relPath, 'presets'] : relPath);
    }
    getConfigDirectories(configPath) {
        const presetsPath = GLib.build_filenamev([configPath, 'presets']);
        try {
            const configDir = Gio.File.new_for_path(presetsPath);
            const directories = [];
            const enumerator = configDir.enumerate_children('standard::name,standard::type', Gio.FileQueryInfoFlags.NONE, null);
            let fileInfo;
            while ((fileInfo = enumerator.next_file(null)) !== null) {
                if (fileInfo.get_file_type() === Gio.FileType.DIRECTORY) {
                    const fileName = fileInfo.get_name();
                    console.log(`Found Input Remapper device directory: ${fileName}`);
                    directories.push(GLib.build_filenamev([presetsPath, fileName]));
                }
            }
            return directories;
        }
        catch (error) {
            logError(error);
            return [];
        }
    }
    getConfigFiles(configPath) {
        const configDirectories = this.getConfigDirectories(configPath);
        const result = {};
        for (const dirPath of configDirectories) {
            const dir = Gio.File.new_for_path(dirPath);
            const jsonFiles = [];
            try {
                const enumerator = dir.enumerate_children('standard::name,standard::type', Gio.FileQueryInfoFlags.NONE, null);
                let fileInfo;
                while ((fileInfo = enumerator.next_file(null)) !== null) {
                    if (fileInfo.get_file_type() === Gio.FileType.REGULAR) {
                        const fileName = fileInfo.get_name();
                        if (fileName.endsWith('.json')) {
                            jsonFiles.push(GLib.build_filenamev([dirPath, fileName]));
                        }
                    }
                }
                if (jsonFiles.length > 0) {
                    result[dirPath] = jsonFiles;
                }
            }
            catch (error) {
                logError(error);
            }
        }
        return result;
    }
    static openDirectory(dirPath) {
        try {
            const launcher = new Gio.SubprocessLauncher({
                flags: Gio.SubprocessFlags.NONE
            });
            // Use xdg-open which will use the default file manager
            launcher.spawnv(['xdg-open', dirPath]);
            return true;
        }
        catch (error) {
            logError(error);
            return false;
        }
    }
    static runAfter(seconds, callback) {
        const timeout = GLib.timeout_add(GLib.PRIORITY_DEFAULT, seconds * 1000, () => {
            callback();
            if (activeTimeouts.includes(timeout)) {
                activeTimeouts = activeTimeouts.filter(t => t != timeout);
            }
            return GLib.SOURCE_REMOVE;
        }, null);
        activeTimeouts.push(timeout);
        return timeout;
    }
}
export let activeTimeouts = [];
export const runAfter = FileHelpers.runAfter;
export function openInputRemapperUi() {
    try {
        const launcher = new Gio.SubprocessLauncher({
            flags: Gio.SubprocessFlags.NONE
        });
        // Use xdg-open which will use the default file manager
        launcher.spawnv(['input-remapper-gtk']);
        return true;
    }
    catch (error) {
        logError(error);
        return false;
    }
}
