// GNOME 46–48 — Eversolo Now Playing (UI + Last.fm + GSettings + toggle miniature)

import St from 'gi://St';
import Gio from 'gi://Gio';
import GLib from 'gi://GLib';
import GObject from 'gi://GObject';
import Soup from 'gi://Soup?version=3.0';

import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import * as PopupMenu from 'resource:///org/gnome/shell/ui/popupMenu.js';

const ByteArray = imports.byteArray;

// === CONFIG / CONSTS =========================================================
const SCHEMA_ID = 'org.gnome.shell.extensions.eversolo-nowplaying';

const REFRESH_SECS = 5;
const COVER_SIZE = 24;
const MENU_COVER_SIZE = 230;

const STATE_PATH = '/ZidooMusicControl/v2/getState';
const CMD_PREV   = '/ZidooMusicControl/v2/playLast';
const CMD_TOGGLE = '/ZidooMusicControl/v2/playOrPause';
const CMD_NEXT   = '/ZidooMusicControl/v2/playNext';

const LASTFM_URL = 'https://ws.audioscrobbler.com/2.0/';

const CACHE_DIR       = GLib.build_filenamev([GLib.get_user_cache_dir(), 'eversolo-nowplaying']);
const COVER_FILE_MENU = GLib.build_filenamev([CACHE_DIR, 'cover_menu.jpg']);
const LFM_CACHE_FILE  = GLib.build_filenamev([CACHE_DIR, 'lastfm-cache.json']);

// === SETTINGS (compat) ====================================
function _settingsFromLocalSchema(schemaId) {
  // extension.js -> dir de la extensión
  const extFile = Gio.File.new_for_uri(import.meta.url);
  const extDir  = extFile.get_parent();                 // ~/.local/.../eversolo-nowplaying@.../
  const schemas = extDir.get_child('schemas');          // .../schemas

  const source = Gio.SettingsSchemaSource.new_from_directory(
    schemas.get_path(),
    Gio.SettingsSchemaSource.get_default(),
    false
  );
  const schema = source.lookup(schemaId, true);
  if (!schema)
    throw new Error(`Schema ${schemaId} no encontrado en ${schemas.get_path()}`);
  return new Gio.Settings({ settings_schema: schema });
}

let SETTINGS = null;
let _settingsChangedId = 0;

function getBase()       { return (SETTINGS?.get_string('eversolo-base') || '').trim(); }
function getLastFmKey()  { return (SETTINGS?.get_string('lastfm-apikey') || '').trim(); }
function getPanelCover() { return !!(SETTINGS?.get_boolean('panel-cover')); }

// === UTILS ===================================================================
function ensureCacheDir() {
  try {
    const f = Gio.File.new_for_path(CACHE_DIR);
    if (!f.query_exists(null)) f.make_directory_with_parents(null);
  } catch (e) { log(`[Eversolo] couldn't create cache: ${e}`); }
}

let _lfmCache = {};
function loadLfmCache() {
  try {
    const f = Gio.File.new_for_path(LFM_CACHE_FILE);
    if (!f.query_exists(null)) return;
    const [ok, bytes] = f.load_contents(null);
    if (ok) _lfmCache = JSON.parse(ByteArray.toString(bytes));
  } catch { _lfmCache = {}; }
}
function saveLfmCache() {
  try {
    Gio.File.new_for_path(LFM_CACHE_FILE)
      .replace_contents(JSON.stringify(_lfmCache), null, false, Gio.FileCreateFlags.REPLACE_DESTINATION, null);
  } catch {}
}

function httpSession() { const s = new Soup.Session(); s.timeout = REFRESH_SECS; return s; }
async function httpGetJSON(session, url) {
  return new Promise((resolve) => {
    const msg = Soup.Message.new('GET', url);
    session.send_and_read_async(msg, GLib.PRIORITY_DEFAULT, null, (sess, res) => {
      try {
        const bytes = sess.send_and_read_finish(res);
        const data  = ByteArray.toString(bytes.get_data());
        try { resolve(JSON.parse(data)); } catch { resolve(null); }
      } catch { resolve(null); }
    });
  });
}
async function httpGetToFile(session, url, destPath) {
  return new Promise((resolve) => {
    const msg = Soup.Message.new('GET', url);
    session.send_and_read_async(msg, GLib.PRIORITY_DEFAULT, null, (sess, res) => {
      try {
        const bytes = sess.send_and_read_finish(res);
        const data = bytes.get_data();
        if (!data?.length) return resolve(false);
        Gio.File.new_for_path(destPath)
          .replace_contents(data, null, false, Gio.FileCreateFlags.REPLACE_DESTINATION, null);
        resolve(true);
      } catch (e) { log(`[Eversolo] Error saving cover: ${e}`); resolve(false); }
    });
  });
}

function formatInfo(o) {
  const parts = [];
  if (o.quality) parts.push(String(o.quality));
  if (o.bits) parts.push(String(o.bits).replace(/-bit$/, ''));
  if (o.samplerate) parts.push(String(o.samplerate));
  return parts.join(' – ');
}

function parseState(json) {
  if (!json) return null;
  const roonActive = !!(json?.everSoloPlayInfo?.everSoloPlayAudioInfo?.songName);
  if (roonActive) {
    const i = json.everSoloPlayInfo.everSoloPlayAudioInfo;
    const o = json.everSoloPlayOutputInfo || {};
    return {
      title: i.songName || 'Unkwown',
      artist: i.artistName || '',
      album: i.albumName || '',
      cover_url: i.albumUrl || '',
      quality: o.outPutDecodec || 'PCM',
      bits: i.audioBitsPerSample > 0 ? `${i.audioBitsPerSample}` : '',
      samplerate: i.audioSampleRate > 0 ? `${Math.floor(i.audioSampleRate / 1000)} kHz` : '',
    };
  }
  const p = json.playingMusic || {};
  return {
    title: p.title || 'Unkwown',
    artist: p.artist || '',
    album: p.album || '',
    cover_url: p.albumArt || '',
    quality: p.audioQuality || '',
    bits: p.bits || '',
    samplerate: p.sampleRate || '',
  };
}

function setTextureFromPath(bin, path, size) {
  try {
    const child = bin.get_child && bin.get_child();
    if (child) child.destroy();
    const uri = Gio.File.new_for_path(path).get_uri();
    const bust = `${uri}?t=${Date.now()}`;
    bin.set_style(
      `background-image: url("${bust}");` +
      `background-size: cover; background-position: center;` +
      `width: ${size}px; height: ${size}px; border-radius: ${Math.round(size*0.15)}px;`
    );
  } catch (e) { log(`[Eversolo] Error showing cover: ${e}`); }
}

function actorAlive(a) { return a && !a.destroyed && a.get_stage?.(); }
function _key(artist, track) { const n = s => (s || '').toLowerCase().trim(); return `${n(artist)}::${n(track)}`; }

async function getLastFmCover(session, artist, track) {
  const key = getLastFmKey();
  if (!key || !artist || !track) return null;
  const ck = _key(artist, track);
  if (_lfmCache[ck] !== undefined) return _lfmCache[ck];
  const params = new URLSearchParams({ method: 'track.getInfo', api_key: key, artist, track, format: 'json' });
  const url = `${LASTFM_URL}?${params.toString()}`;
  const json = await httpGetJSON(session, url);
  const imgs = json?.track?.album?.image || [];
  const found = imgs.length ? imgs[imgs.length - 1]['#text'] : null;
  _lfmCache[ck] = found; saveLfmCache();
  return found;
}

// === INDICATOR ===============================================================
const Indicator = GObject.registerClass(
class Indicator extends PanelMenu.Button {
  _init() {
    super._init(0.0, 'Eversolo Now Playing');
    this.name = 'eversolo-nowplaying';
    this._alive = true;
    this._lastCoverUrl = '';

    ensureCacheDir();

    this.add_style_class_name('eversolo-button');
    this._panelIcon = null;
    this._coverBin  = null;
    this._rebuildPanelActor();

    // Tooltip
    this._tooltip = new St.Label({ text: 'Eversolo', style_class: 'eversolo-tooltip' });
    this._tooltip.hide();
    Main.layoutManager.addTopChrome(this._tooltip);
    this._enterId = this.connect('enter-event', () => this._showTooltip());
    this._leaveId = this.connect('leave-event', () => this._hideTooltip());

    // Info en menú
    this._title  = new St.Label({ text: '—', style_class: 'eversolo-title' });
    this._artist = new St.Label({ text: '',  style_class: 'eversolo-artist' });
    this._info   = new St.Label({ text: '',  style_class: 'eversolo-info' });
    this._menuCover = new St.Bin({ width: MENU_COVER_SIZE, height: MENU_COVER_SIZE, style_class: 'eversolo-cover-menu' });

    const headerBox = new St.BoxLayout({ vertical: true, style_class: 'eversolo-header' });
    headerBox.add_child(this._title); headerBox.add_child(this._artist); headerBox.add_child(this._info);

    const topRow = new St.BoxLayout({ x_expand: true });
    topRow.add_child(this._menuCover); topRow.add_child(new St.Widget({ width: 12 })); topRow.add_child(headerBox);

    const topItem = new PopupMenu.PopupBaseMenuItem({ reactive: false, can_focus: false });
    topItem.add_child(topRow); this.menu.addMenuItem(topItem);
    this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());

    const ctrls = new St.BoxLayout({ style_class: 'eversolo-ctrls' });
    ctrls.add_child(this._makeCtrlButton('media-skip-backward-symbolic', () => this._sendCmd(CMD_PREV)));
    ctrls.add_child(this._makeCtrlButton('media-playback-start-symbolic', () => this._sendCmd(CMD_TOGGLE)));
    ctrls.add_child(this._makeCtrlButton('media-skip-forward-symbolic', () => this._sendCmd(CMD_NEXT)));
    const ctrlItem = new PopupMenu.PopupBaseMenuItem({ reactive: false, can_focus: false });
    ctrlItem.add_child(ctrls); this.menu.addMenuItem(ctrlItem);

    this._session = httpSession();

    this._timeout = GLib.timeout_add_seconds(GLib.PRIORITY_DEFAULT, REFRESH_SECS, () => {
      if (!this._alive) return GLib.SOURCE_REMOVE;
      this._refresh().catch(e => log(`[Eversolo] refresh error: ${e}`));
      return GLib.SOURCE_CONTINUE;
    });

    this._refresh().catch(e => log(`[Eversolo] initial refresh error: ${e}`));
  }

  _rebuildPanelActor() {
    if (this._panelIcon && !this._panelIcon.destroyed) this._panelIcon.destroy();
    if (this._coverBin  && !this._coverBin.destroyed)  this._coverBin.destroy();

    if (getPanelCover()) {
      this._coverBin = new St.Bin({ style_class: 'eversolo-cover-bin', width: COVER_SIZE, height: COVER_SIZE });
      this.add_child(this._coverBin);
    } else {
      this._panelIcon = new St.Icon({ icon_name: 'media-playback-start-symbolic', icon_size: 16 });
      this.add_child(this._panelIcon);
    }
  }

  _applySettings() {
    this._rebuildPanelActor();
  }

  _makeCtrlButton(iconName, cb) {
    const btn = new St.Button({ style_class: 'eversolo-btn' });
    btn.set_child(new St.Icon({ icon_name: iconName, icon_size: 18 }));
    btn.connect('clicked', () => { if (this._alive) cb(); });
    return btn;
  }

  _positionTooltip() {
    if (!actorAlive(this) || !actorAlive(this._tooltip)) return;
    const [x, y] = this.get_transformed_position();
    const [w, h] = this.get_transformed_size();
    this._tooltip.set_position(Math.round(x), Math.round(y + h + 6));
  }
  _showTooltip() { if (actorAlive(this._tooltip)) { this._positionTooltip(); this._tooltip.show(); } }
  _hideTooltip() { if (actorAlive(this._tooltip)) this._tooltip.hide(); }

  async _sendCmd(path) {
    const base = getBase();
    if (!base) return;
    const msg = Soup.Message.new('GET', base + path);
    this._session.send_and_read_async(msg, GLib.PRIORITY_DEFAULT, null, () => {});
  }

  async _refresh() {
    if (!this._alive) return;
    try {
      const base = getBase();
      if (!base) { this._title.text = 'Eversolo'; this._artist.text = 'Configure IP in Preferences'; return; }

      const json = await httpGetJSON(this._session, base + STATE_PATH);
      if (!this._alive) return;
      const state = parseState(json);

      const safeSet = (label, text) => { if (actorAlive(label)) label.text = text; };

      if (!state) {
        safeSet(this._title, 'Eversolo'); safeSet(this._artist, '— no signal —'); safeSet(this._info, ''); return;
      }

      safeSet(this._title, state.title || '—');
      safeSet(this._artist, state.artist || '');
      safeSet(this._info, formatInfo(state));
      if (actorAlive(this._tooltip)) this._tooltip.text = `${state.title || ''} — ${state.artist || ''}`.trim();

      let coverUrl = state.cover_url && /^https?:\/\//.test(state.cover_url) ? state.cover_url : null;
      if (!coverUrl) coverUrl = await getLastFmCover(this._session, state.artist, state.title);
      if (!this._alive) return;

      if (coverUrl && coverUrl !== this._lastCoverUrl) {
        const ok = await httpGetToFile(this._session, coverUrl, COVER_FILE_MENU);
        if (ok && this._alive) {
          this._lastCoverUrl = coverUrl;
          setTextureFromPath(this._menuCover, COVER_FILE_MENU, MENU_COVER_SIZE);
          if (getPanelCover() && this._coverBin && !this._coverBin.destroyed)
            setTextureFromPath(this._coverBin, COVER_FILE_MENU, COVER_SIZE);
        }
      }

      if (!getPanelCover() && this._panelIcon && !this._panelIcon.destroyed)
        this._panelIcon.icon_name = 'media-playback-start-symbolic';

    } catch (e) { log(`[Eversolo] _refresh exception: ${e}`); }
  }

  destroy() {
    this._alive = false;
    if (this._timeout) { GLib.source_remove(this._timeout); this._timeout = 0; }
    if (this._enterId) { this.disconnect(this._enterId); this._enterId = 0; }
    if (this._leaveId) { this.disconnect(this._leaveId); this._leaveId = 0; }
    if (this._tooltip) { this._tooltip.destroy(); this._tooltip = null; }
    super.destroy();
  }
});

let _indicator = null;

// === ENTRYPOINT ==============================================================
export default class Extension {
  enable() {
    SETTINGS = _settingsFromLocalSchema(SCHEMA_ID);
    _settingsChangedId = SETTINGS.connect('changed', () => {
      if (_indicator) { _indicator._applySettings(); _indicator._refresh?.(); }
    });

    ensureCacheDir();
    loadLfmCache();

    _indicator = new Indicator();
    Main.panel.addToStatusArea('eversolo-nowplaying', _indicator, 0, 'right');
  }
  disable() {
    if (_settingsChangedId && SETTINGS) { SETTINGS.disconnect(_settingsChangedId); _settingsChangedId = 0; }
    if (_indicator) { _indicator.destroy(); _indicator = null; }
    SETTINGS = null;
  }
}

