import St from 'gi://St';
import Clutter from 'gi://Clutter';
import { IntervalRunner } from '../../../utils/intervalRunner.js';
import { clamp } from '../../../utils/utils.js';
import Shell from 'gi://Shell';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import { IdleRunner } from '../../../utils/idleRunner.js';
import { log } from '../../../utils/logging.js';
import { calculateLuminance } from '../../../utils/colors.js';
import BaseNavigationBar from './baseNavigationBar.js';
import { Bin } from '../../../utils/ui/widgets.js';
import OverviewAndWorkspaceGestureController from '../../../utils/overviewAndWorkspaceGestureController.js';
import { GestureRecognizer, GestureRecognizerEvent } from '../../../utils/ui/gestureRecognizer.js';
import { Delay } from '../../../utils/delay.js';
import GObject from 'gi://GObject';
import Mtk from 'gi://Mtk';

// Area reserved on the left side of the navbar in which a swipe up opens the OSK
// Note: This is in logical pixels, not physical pixels
const LEFT_EDGE_OFFSET = 100;
class GestureNavigationBar extends BaseNavigationBar {
    styleClassUpdateInterval;
    _isWindowNear = false;
    gestureManager;
    constructor(props) {
        super({ reserveSpace: props.reserveSpace });
        this.setInvisibleMode(props.invisibleMode);
        this.styleClassUpdateInterval = new IntervalRunner(500, this.updateStyleClasses.bind(this));
        this.gestureManager = new NavigationBarGestureManager({
            edgeThreshold: this.computeHeight(),
        });
        this.actor.connect('notify::mapped', () => this.gestureManager.setEnabled(this.actor.mapped));
        this.connect('notify::visible', _ => this._updateStyleClassIntervalActivity());
        this.connect('notify::reserve-space', _ => {
            this._updateStyleClassIntervalActivity();
            void this.updateStyleClasses();
        });
    }
    _buildActor() {
        return new _EventPassthroughActor({
            name: 'touchup-navbar',
            styleClass: 'touchup-navbar touchup-navbar--transparent bottom-panel',
            reactive: true,
            trackHover: true,
            canFocus: true,
            layoutManager: new Clutter.BinLayout(),
            onRealize: () => this.styleClassUpdateInterval.scheduleOnce(),
            child: this.pill = new Bin({
                name: 'touchup-navbar__pill',
                styleClass: 'touchup-navbar__pill',
                yAlign: Clutter.ActorAlign.CENTER,
                xAlign: Clutter.ActorAlign.CENTER,
            }),
        });
    }
    onIsWindowNearChanged(isWindowNear) {
        this._isWindowNear = isWindowNear;
        if (!this.reserveSpace) {
            let newInterval = Main.overview.visible || !isWindowNear ? 3000 : 500;
            if (newInterval != this.styleClassUpdateInterval.interval) {
                // if a window is moved onto/away from the navigation bar or overview is toggled, schedule update soonish:
                this.styleClassUpdateInterval.scheduleOnce(250);
            }
            this.styleClassUpdateInterval.setInterval(newInterval);
        }
        else {
            void this.updateStyleClasses();
        }
    }
    computeHeight() {
        const sf = St.ThemeContext.get_for_stage(global.stage).scaleFactor;
        return 22 * sf;
    }
    computePillSize() {
        const sf = St.ThemeContext.get_for_stage(global.stage).scaleFactor;
        return {
            width: clamp(this.monitor.width * 0.25, 70 * sf, 330 * sf),
            height: Math.floor(Math.min(this.computeHeight() * 0.8, 6 * sf, this.computeHeight() - 2)),
        };
    }
    onBeforeReallocate() {
        this.actor.set_height(this.computeHeight());
        this.pill.set_width(this.computePillSize().width);
        this.pill.set_height(this.computePillSize().height);
        this.gestureManager.setEdgeThreshold(this.computeHeight());
    }
    setMonitor(monitorIndex) {
        super.setMonitor(monitorIndex);
        this.gestureManager.setMonitor(monitorIndex);
    }
    async updateStyleClasses() {
        if (this.reserveSpace && this._isWindowNear) {
            // Make navbar opaque (black or white, based on shell theme brightness):
            this.actor.remove_style_class_name('touchup-navbar--transparent');
            this.pill.remove_style_class_name('touchup-navbar__pill--dark');
        }
        else {
            // Make navbar transparent:
            this.actor.add_style_class_name('touchup-navbar--transparent');
            // Adjust pill brightness:
            let brightness = await this.findBestPillBrightness();
            if (brightness == 'dark') {
                this.pill.add_style_class_name('touchup-navbar__pill--dark');
            }
            else {
                this.pill.remove_style_class_name('touchup-navbar__pill--dark');
            }
        }
    }
    /**
     * Find the best pill brightness by analyzing what's on the screen behind the pill
     */
    async findBestPillBrightness() {
        try {
            // FIXME: This relies on the color of a single pixel right now, see below for several other attempts
            //  that all have problems due to GJS/introspection limitations
            const shooter = new Shell.Screenshot();
            // @ts-ignore (typescript doesn't understand Gio._promisify(...) - see top of file)
            // const [content]: [Clutter.TextureContent] = await shooter.screenshot_stage_to_content();
            // const wholeScreenTexture = content.get_texture();
            // An area surrounding the pill to use for brightness analysis:
            // const area = {
            //     x: this.pill.x - 20 * this.scaleFactor,
            //     y: this.y,
            //     w: this.pill.width + 40 * this.scaleFactor,
            //     h: this.height,
            // };
            // const verticalPadding = (area.h - this.pill.height) / 2;
            // High-level attempt (works but has memory leak - at least since Gnome Shell 46, maybe before too):
            // const stream = Gio.MemoryOutputStream.new_resizable();
            // // @ts-ignore (ts doesn't understand Gio._promisify())
            // // noinspection JSVoidFunctionReturnValueUsed
            // const pixbuf: GdkPixbuf.Pixbuf = await Shell.Screenshot.composite_to_stream(  // takes around 4-14ms, most of the time 7ms
            //     wholeScreenTexture, area.x, area.y, area.w, area.h,
            //     this.scaleFactor, null, 0, 0, 1, stream
            // );
            // stream.close(null);
            // //  -- memory leak is above this line --
            // const avgColor = calculateAverageColor(pixbuf.get_pixels(), pixbuf.width, [
            //    {x: 0, y: 0, width: pixbuf.width, height: verticalPadding},  // above pill
            //     {x: 0, y: verticalPadding + this.pill.height, width: pixbuf.width, height: verticalPadding}  // below pill
            // ]);
            // const luminance = calculateLuminance(...avgColor);
            // // Save pxibuf as png image to tempdir to inspect:
            // // pixbuf.savev(`/tmp/pxibuf-1-${avgColor}-${luminance}.png`, 'png', null, null);
            // Low-level api attempt (not working; missing introspection annotations for `Cogl.SubTexture.get_data`):
            // try {
            //     const ctx = Clutter.get_default_backend().get_cogl_context();
            //     const subtex = Cogl.SubTexture.new(ctx, wholeScreenTexture, area.x, area.y, area.w, area.h);
            //     //const surface = new Cairo.ImageSurface(Cairo.Format.ARGB32, subtex.get_width(), subtex.get_height());
            //
            //     if (subtex) {
            //         //const size = subtex.get_data(PixelFormat.ARGB_8888, 0, null);
            //         //const buf = new Uint8Array(size);
            //         let [buf, size] = subtex.get_data(PixelFormat.ARGB_8888, 0);
            //
            //         debugLog("Buf length: ", buf.length, " - max: ", Math.max(...buf.values()));
            //     } else {
            //         debugLog("Subtex is null");
            //     }
            // } catch (e) {
            //     debugLog("Error in updatePillBrightness: ", e);
            // }
            // Mid-level attempt (not working; missing introspection annotations for `Cogl.Framebuffer.read_pixels`):
            // const ctx = Clutter.get_default_backend().get_cogl_context();
            // const subtex = Cogl.SubTexture.new(ctx, wholeScreenTexture, area.x, area.y, area.w, area.h);
            // debugLog("subtex: ", subtex);
            // if (subtex) {
            //     /*(global.stage as Clutter.Stage).paint_to_buffer(
            //         new Mtk.Rectangle({x: area.x, y: area.y, width: area.w, height: area.h}),
            //         1,
            //         buf,
            //         0,
            //         PixelFormat.ARGB_8888,
            //         PaintFlag.NO_CURSORS,
            //     );*/
            //     /*
            //     const tex = Cogl.Texture2D.new_with_size(ctx, area.w, area.h);
            //     const fb = Cogl.Offscreen.new_with_texture(tex);
            //     global.stage.paint_to_framebuffer(
            //         fb,
            //         new Mtk.Rectangle({x: area.x, y: area.y, width: area.w, height: area.h}),
            //         1,
            //         PaintFlag.NO_CURSORS,
            //     );
            //     const buffer: Uint8Array = fb.read_pixels(0, 0, area.w, area.h, PixelFormat.ARGB_8888);
            //     */
            // }
            // Individual pixel attempt:
            let rect = this.pill.get_transformed_extents();
            // @ts-ignore
            let colors = (await Promise.all([
                // We only use one pixel as doing this with multiple pixels appears to have very bad
                // performance (screen lags, visible e.g. when moving a window):
                shooter.pick_color(rect.get_x() + rect.get_width() * 0.5, rect.get_y() - 2),
                // shooter.pick_color(rect.get_x() + rect.get_width() * 0.4, rect.get_y() + rect.get_height() + 3),
                // @ts-ignore
            ])).map(c => c[0]);
            // Calculate the luminance of the average RGB values:
            let luminance = calculateLuminance(colors.reduce((a, b) => a + b.red, 0) / colors.length, colors.reduce((a, b) => a + b.green, 0) / colors.length, colors.reduce((a, b) => a + b.blue, 0) / colors.length);
            return luminance > 0.5 ? 'dark' : 'light';
        }
        catch (e) {
            log("Exception during `findBestPillBrightness` (falling back to 'dark' brightness): ", e);
            return 'dark';
        }
    }
    _updateStyleClassIntervalActivity() {
        this.styleClassUpdateInterval.setActive(this.isVisible && !this.reserveSpace);
    }
    setInvisibleMode(invisible) {
        // We use opacity here instead of the actors `visible` property since [LayoutManager.addTopChrome] uses the
        // `visible` property itself which would interfere with this.
        this.actor.opacity = invisible ? 0 : 255;
    }
    destroy() {
        this.styleClassUpdateInterval.stop();
        this.gestureManager.destroy();
        super.destroy();
    }
}
class NavigationBarGestureManager {
    static _overviewMaxSpeed = 0.005;
    static _workspaceMaxSpeed = 0.0016;
    action;
    _controller;
    _recognizer;
    _idleRunner;
    _targetWorkspaceProgress = 0;
    _targetOverviewProgress = null;
    /**
     * This virtual input device is used to emulate touch events in click-through-navbar scenarios.
     */
    _virtualTouchscreenDevice;
    _scaleFactor;
    _hasStarted = false;
    _isKeyboardGesture = false;
    constructor(props) {
        this._scaleFactor = St.ThemeContext.get_for_stage(global.stage).scaleFactor;
        // The controller used to actually perform the navigation gestures:
        this._controller = new OverviewAndWorkspaceGestureController({
            monitorIndex: props.monitor?.index ?? Main.layoutManager.primaryIndex,
        });
        // Use an [IdleRunner] to make the gestures asynchronously follow the users' finger:
        this._idleRunner = new IdleRunner((_, dt) => this._onIdleRun(dt ?? undefined));
        // Our [GestureRecognizer] to interpret the gestures:
        this._recognizer = new GestureRecognizer({
            onGestureProgress: state => this._onGestureProgress(state),
            onGestureCompleted: state => this._onGestureCompleted(state),
        });
        // Action that listens to appropriate events on the stage:
        this.action = new GestureNavigationBarAction({
            edgeThreshold: props.edgeThreshold,
        });
        this.action.connect('progress', (_, evt) => {
            this._recognizer.push(GestureRecognizerEvent.fromClutterEvent(evt));
        });
        this.action.connect('end', (_, evt) => {
            this._recognizer.push(GestureRecognizerEvent.fromClutterEvent(evt));
        });
        global.stage.add_action_full('touchup-navigation-bar', Clutter.EventPhase.CAPTURE, this.action);
        // To emit virtual events:
        this._virtualTouchscreenDevice = Clutter.get_default_backend().get_default_seat().create_virtual_device(Clutter.InputDeviceType.TOUCHSCREEN_DEVICE);
    }
    setMonitor(monitorIndex) {
        this._controller.monitorIndex = monitorIndex;
    }
    setEdgeThreshold(edgeThreshold) {
        this.action.setEdgeThreshold(edgeThreshold);
    }
    setEnabled(enabled) {
        this.action.enabled = enabled;
    }
    destroy() {
        global.stage.remove_action(this.action);
    }
    _onGestureProgress(state) {
        if (state.hasMovement) {
            if (!this._hasStarted) {
                this._startGestures(state);
            }
            if (this._isKeyboardGesture) {
                Main.keyboard._keyboard.gestureProgress(-state.totalMotionDelta.y);
            }
            else {
                this._targetWorkspaceProgress = this._controller.initialWorkspaceProgress
                    - (state.totalMotionDelta.x / this._controller.baseDistX) * 1.6;
                this._targetOverviewProgress = this._controller.initialOverviewProgress
                    + (-state.totalMotionDelta.y / (this._controller.baseDistY * 0.2));
            }
        }
    }
    _startGestures(state) {
        this._hasStarted = true;
        this._isKeyboardGesture = false;
        if (Main.keyboard.visible) {
            // Close the keyboard if it's visible:
            Main.keyboard._keyboard
                ? Main.keyboard._keyboard.close(true) // immediate = true
                : Main.keyboard.close();
        }
        else if (Main.keyboard._keyboard
            && state.pressCoordinates.x < LEFT_EDGE_OFFSET * this._scaleFactor
            && state.firstMotionDirection?.axis === 'vertical') {
            this._isKeyboardGesture = true;
        }
        if (!this._isKeyboardGesture) {
            // Start navigation gestures:
            this._controller.gestureBegin();
            this._targetOverviewProgress = this._controller.initialOverviewProgress;
            this._targetWorkspaceProgress = this._controller.initialWorkspaceProgress;
            this._idleRunner.start();
        }
    }
    _onGestureCompleted(state) {
        this._idleRunner.stop();
        this._hasStarted = false;
        const direction = state.lastMotionDirection?.direction ?? null;
        if (state.isTap) {
            this._controller.gestureEnd({ direction: null });
            this._virtualTouchscreenDevice.notify_touch_down(state.events[0].timeUS, 0, state.pressCoordinates.x, state.pressCoordinates.y);
            Delay.ms(45).then(() => {
                this._virtualTouchscreenDevice.notify_touch_up(state.events.at(-1).timeUS, 0);
            });
        }
        else if (this._isKeyboardGesture) {
            if (direction === 'up') {
                Main.keyboard._keyboard?.gestureActivate();
            }
            else {
                Main.keyboard._keyboard?.gestureCancel();
            }
        }
        else {
            this._controller.gestureEnd({ direction });
        }
        this._targetOverviewProgress = null;
        this._targetWorkspaceProgress = null;
    }
    _onIdleRun(dt = 0) {
        let overviewProg = this._controller.currentOverviewProgress;
        let workspaceProg = this._controller.currentWorkspaceProgress;
        if (this._targetOverviewProgress !== null
            && Math.abs(this._targetOverviewProgress - overviewProg) > 5 * NavigationBarGestureManager._overviewMaxSpeed) {
            let d = this._targetOverviewProgress - overviewProg;
            overviewProg += Math.sign(d) * Math.min(Math.abs(d) ** 2, dt * NavigationBarGestureManager._overviewMaxSpeed);
        }
        if (this._targetWorkspaceProgress !== null
            && Math.abs(this._targetWorkspaceProgress - workspaceProg) > 5 * NavigationBarGestureManager._workspaceMaxSpeed) {
            let d = this._targetWorkspaceProgress - workspaceProg;
            workspaceProg += Math.sign(d) * Math.min(Math.abs(d) ** 2, dt * NavigationBarGestureManager._workspaceMaxSpeed);
        }
        this._controller.gestureUpdate({
            overviewProgress: overviewProg - this._controller.initialOverviewProgress,
            workspaceProgress: workspaceProg - this._controller.initialWorkspaceProgress,
        });
    }
}
/**
 * An actor that is invisible to events, i.e. passes them through to any actors below.
 */
class _EventPassthroughActor extends Bin {
    static {
        GObject.registerClass(this);
    }
    vfunc_pick(pick_context) {
        // By not making any call to this.pick_box(...) here, we make this actor pass through all events to
        // any actor potentially below it. Therefore, this actor is only a visuals and does not react to
        // events.
        return;
    }
}
/**
 * To handle navigation bar events, we register a [GestureNavigationBarAction] on the stage. This class
 * emit gesture progress and end events when events occur in the lower edge of the screen, i.e. in the
 * lowest area the height of which is defined by [edgeThreshold]. The signals emitted just propagate the
 * raw [Clutter.Event] instances to their listeners, such that they can then analyze them using a
 * [GestureRecognizer] which contains all the logic we need to interprete gestures.
 *
 * Therefore, the task of this class is not recognizing any gestures but instead merely filtering out
 * the events that we want to act on and integrate as an outermost layer with the shell.
 */
class GestureNavigationBarAction extends Clutter.GestureAction {
    static {
        GObject.registerClass({
            Signals: {
                'end': { param_types: [Clutter.Event.$gtype] },
                'progress': { param_types: [Clutter.Event.$gtype] },
                'begin': { param_types: [Clutter.Event.$gtype] }
            },
        }, this);
    }
    constructor(props) {
        // Note: This constructor is only to make typescript happy. DON'T put any logic in here – use `_init` below.
        // @ts-ignore
        super(props);
    }
    // @ts-ignore
    _init(props) {
        super._init();
        this._allowedModes = props.allowedModes ?? Shell.ActionMode.ALL;
        this._edgeThreshold = props.edgeThreshold;
        this.set_n_touch_points(1);
        this.set_threshold_trigger_edge(Clutter.GestureTriggerEdge.AFTER);
    }
    setEdgeThreshold(edgeThreshold) {
        this._edgeThreshold = edgeThreshold;
    }
    _getMonitorRect(x, y) {
        const rect = new Mtk.Rectangle({ x: x - 1, y: y - 1, width: 1, height: 1 });
        let monitorIndex = global.display.get_monitor_index_for_rect(rect);
        return global.display.get_monitor_geometry(monitorIndex);
    }
    vfunc_gesture_prepare(actor) {
        if (this.get_n_current_points() === 0)
            return false;
        if (!(this._allowedModes & Main.actionMode))
            return false;
        let [x, y] = this.get_press_coords(0);
        let monitorRect = this._getMonitorRect(x, y);
        const res = y > monitorRect.y + monitorRect.height - this._edgeThreshold;
        if (res) {
            this.emit('begin', this.get_last_event(0));
        }
        return res;
    }
    vfunc_gesture_progress(actor) {
        this.emit('progress', this.get_last_event(0));
        return true;
    }
    vfunc_gesture_end(actor) {
        this.emit('end', this.get_last_event(0));
    }
}
// Note: these are potentially needed for some of the approaches in `updatePillBrightness`, should
// they work one day:
//
//Gio._promisify(Shell.Screenshot.prototype, 'screenshot_stage_to_content');
//Gio._promisify(Shell.Screenshot.prototype, 'pick_color');
//
// if (typeof Cairo.format_stride_for_width === 'undefined') {
//     // Polyfill since the GJS bindings of Cairo are missing `format_stride_width`
//     Cairo.format_stride_for_width = (w: number, bpp: number = 32) => {  // bpp for Cairo.Format.ARGB32 (see https://github.com/ImageMagick/cairo/blob/main/src/cairo-image-surface.c#L741)
//         // Translated from original C-Code (https://github.com/ImageMagick/cairo/blob/main/src/cairoint.h#L1570):
//         //
//         // #define CAIRO_STRIDE_ALIGNMENT (sizeof (uint32_t))
//         // #define CAIRO_STRIDE_FOR_WIDTH_BPP(w,bpp) \
//         //    ((((bpp)*(w)+7)/8 + CAIRO_STRIDE_ALIGNMENT-1) & -CAIRO_STRIDE_ALIGNMENT)
//
//         const CAIRO_STRIDE_ALIGNMENT = Uint32Array.BYTES_PER_ELEMENT || 4  // sizeof(uint32_t) is 4 bytes in most systems
//         return (((bpp * w + 7) / 8 + CAIRO_STRIDE_ALIGNMENT - 1) & -CAIRO_STRIDE_ALIGNMENT);
//     }
// }
//

export { GestureNavigationBar as default };
