import Gio from "gi://Gio";

export function forSettingsKeyChange(logger, settings, key, callback) {
    return {
        id: `${settings.schemaId}.${key}::changed`,
        async connect() {
            const handlerId = settings.connect(`changed::${key}`, async (settings, key) => {
                const value = settings.get_value(key).recursiveUnpack();
                logger.info(`changed ${settings.schema_id}.${key}=${value}`);
                await callback(value);
            });
            logger.debug(`connected to changes of ${settings.schemaId}.${key} with handler ID ${handlerId}`);
            // Need to always access a key after connecting for changes
            await callback(settings.get_value(key).recursiveUnpack());
            return {
                async disconnect() {
                    settings.disconnect(handlerId);
                    logger.debug(`disconnected from changes to ${settings.schemaId}.${key} with handler ID ${handlerId}`);
                },
            };
        },
    };
}

export function forProxySignal(logger, proxy, signal, callback) {
    return {
        id: `${proxy.get_object_path()}/g-signal::${signal}`,
        async connect() {
            const handlerId = proxy.connect(`g-signal::${signal}`, async (source, _senderName, signalName, parameters) => {
                logger.info(`received signal ${signalName}=${parameters.print(true)} from ${source.get_interface_name()}`);
                const params = parameters.recursiveUnpack();
                await callback(...params);
            });
            logger.debug(`connected to signal '${signal}' of ${proxy.get_interface_name()} with handler ID ${handlerId}`);
            return {
                async disconnect() {
                    proxy.disconnect(handlerId);
                    logger.debug(`disconnected from signal '${signal}' of ${proxy.get_interface_name()} with handler ID ${handlerId}`);
                },
            };
        },
    };
}

export function forProxyCall(logger, proxy, subscribe, unsubscribe, args) {
    const interfaceName = proxy.get_interface_name();
    return {
        id: `${interfaceName}.${subscribe}}()`,
        async connect() {
            const handlerId = await proxy.call(subscribe, args, Gio.DBusCallFlags.NONE, -1, null);
            const handlerIdStr = handlerId.print(false);
            logger.debug(`${interfaceName}.${subscribe}(${args.print(false)}) -> ${handlerIdStr}`);
            return {
                async disconnect() {
                    await proxy.call(unsubscribe, handlerId, Gio.DBusCallFlags.NONE, -1, null);
                    logger.debug(`${interfaceName}.${unsubscribe}(${handlerIdStr})`);
                },
            };
        },
    };
}
