// GNOME 46–48 — Eversolo Now Playing (UI + Last.fm + GSettings + provider/model + panel toggle)

import St from 'gi://St';
import Gio from 'gi://Gio';
import GLib from 'gi://GLib';
import GObject from 'gi://GObject';
import Soup from 'gi://Soup';
import Clutter from 'gi://Clutter';

import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import * as PopupMenu from 'resource:///org/gnome/shell/ui/popupMenu.js';
import { Extension } from 'resource:///org/gnome/shell/extensions/extension.js';

// === CONSTS ==================================================================
const REFRESH_SECS = 5;
const COVER_SIZE = 24;
const MENU_COVER_SIZE = 230;

const STATE_PATH = '/ZidooMusicControl/v2/getState';
const CMD_PREV   = '/ZidooMusicControl/v2/playLast';
const CMD_TOGGLE = '/ZidooMusicControl/v2/playOrPause';
const CMD_NEXT   = '/ZidooMusicControl/v2/playNext';

const LASTFM_URL = 'https://ws.audioscrobbler.com/2.0/';
const CACHE_DIR       = GLib.build_filenamev([GLib.get_user_cache_dir(), 'eversolo-nowplaying']);
const COVER_FILE_MENU = GLib.build_filenamev([CACHE_DIR, 'cover_menu.jpg']);

// === UTILS ===================================================================
function ensureCacheDir() {
  try {
    const f = Gio.File.new_for_path(CACHE_DIR);
    if (!f.query_exists(null)) f.make_directory_with_parents(null);
  } catch (e) { log(`[Eversolo] couldn't create cache: ${e}`); }
}

function httpSession() {
  const s = new Soup.Session();
  s.timeout = REFRESH_SECS;
  return s;
}

async function httpGetJSON(session, url) {
  return new Promise((resolve) => {
    const msg = Soup.Message.new('GET', url);
    session.send_and_read_async(msg, GLib.PRIORITY_DEFAULT, null, (sess, res) => {
      try {
        const bytes = sess.send_and_read_finish(res);
        const data  = new TextDecoder('utf-8').decode(bytes.get_data());
        try { resolve(JSON.parse(data)); } catch { resolve(null); }
      } catch { resolve(null); }
    });
  });
}

async function httpGetToFile(session, url, destPath) {
  return new Promise((resolve) => {
    const msg = Soup.Message.new('GET', url);
    session.send_and_read_async(msg, GLib.PRIORITY_DEFAULT, null, (sess, res) => {
      try {
        const bytes = sess.send_and_read_finish(res);
        const data = bytes.get_data();
        if (!data?.length) return resolve(false);
        Gio.File.new_for_path(destPath)
          .replace_contents(data, null, false, Gio.FileCreateFlags.REPLACE_DESTINATION, null);
        resolve(true);
      } catch (e) { log(`[Eversolo] Error saving cover: ${e}`); resolve(false); }
    });
  });
}

function formatInfo(o) {
  const parts = [];
  if (o.mqa) parts.push('MQA');
  // Evita mostrar "PCM" si ya indicamos MQA
  if (o.quality && !(o.mqa && String(o.quality).toUpperCase() === 'PCM'))
    parts.push(String(o.quality));
  if (o.bits) parts.push(String(o.bits).replace(/-bit$/, ''));
  if (o.samplerate) parts.push(String(o.samplerate));
  return parts.join(' – ');
}



function prettyProvider(id, fallback) {
  if (!id && fallback) id = String(fallback).toLowerCase();
  if (!id) return 'Unknown';
  const map = {
    tidal: 'Tidal',
    qobuz: 'Qobuz',
    spotify: 'Spotify',
    apple: 'Apple Music',
    deezer: 'Deezer',
    local: 'Local',
    airplay: 'AirPlay',
    dlna: 'DLNA',
  };
  id = id.replace(/\.[a-z]+$/i, '');
  return map[id] || id.charAt(0).toUpperCase() + id.slice(1);
}

// playing = (state === 4) — fuente de verdad simple y robusta
function isPlayingFrom(json) {
  return json?.state === 4;
}



function isMqa(json) {
  const out = json?.everSoloPlayInfo?.everSoloPlayOutputInfo;
  if (!out && json?.mqaMode === 1) return true;
  if (!out) return false;

  const byCodec = String(out.outPutDecodec || '').toUpperCase() === 'MQA';
  const byType  = (out.outPutMqaType ?? 0) > 0;
  return byCodec || byType || json?.mqaMode === 1;
}

function parseState(json) {
  if (!json) return null;

  const deviceModel   = json?.deviceInfo?.model || json?.deviceInfo?.deviceName || '';
  const playStatus    = json?.everSoloPlayInfo?.playStatus ?? 0;
  const providerId    = json?.playingMusic?.streamId || (json?.everSoloPlayInfo?.playTypeSubtitle || '').toLowerCase();
  const providerName  = prettyProvider(providerId, providerId);
  const streamQuality = json?.playingMusic?.streamQuality || '';
  const playing       = isPlayingFrom(json);
  const mqa           = isMqa(json);

  const roonActive = !!(json?.everSoloPlayInfo?.everSoloPlayAudioInfo?.songName);
  if (roonActive) {
    const i = json.everSoloPlayInfo.everSoloPlayAudioInfo;
    const o = json.everSoloPlayOutputInfo || {};
    return {
      title: i.songName || 'Unknown',
      artist: i.artistName || '',
      album: i.albumName || '',
      cover_url: i.albumUrl || '',
      quality: o.outPutDecodec || 'PCM',
      bits: i.audioBitsPerSample > 0 ? `${i.audioBitsPerSample}` : '',
      samplerate: i.audioSampleRate > 0 ? `${Math.floor(i.audioSampleRate / 1000)} kHz` : '',
      deviceModel, playStatus, provider: providerName, streamQuality, playing, mqa,
    };
  }

  const p = json.playingMusic || {};
  return {
    title: p.title || 'Unknown',
    artist: p.artist || '',
    album: p.album || '',
    cover_url: p.albumArt || '',
    quality: p.audioQuality || '',
    bits: p.bits || '',
    samplerate: p.sampleRate || '',
    deviceModel, playStatus, provider: providerName, streamQuality, playing, mqa,
  };
}


function setTextureFromPath(bin, path, size) {
  try {
    const child = bin.get_child && bin.get_child();
    if (child) child.destroy();
    const uri = Gio.File.new_for_path(path).get_uri();
    const bust = `${uri}?t=${Date.now()}`;
    bin.set_style(
      `background-image: url("${bust}");` +
      `background-size: cover; background-position: center;` +
      `width: ${size}px; height: ${size}px; border-radius: ${Math.round(size*0.15)}px;`
    );
  } catch (e) { log(`[Eversolo] Error showing cover: ${e}`); }
}

function actorAlive(a) { return a && !a.destroyed && a.get_stage?.(); }
const norm = s => (s || '').toLowerCase().trim();

// === INDICATOR ===============================================================
const Indicator = GObject.registerClass(
class Indicator extends PanelMenu.Button {
  _init(settings, lfmCacheRef) {
    super._init(0.0, 'Eversolo Now Playing');
    this._prevPos = null;   // para detectar si avanza y saber si está reproduciendo
    this.name = 'eversolo-nowplaying';
    this._alive = true;
    this._lastCoverUrl = '';
    this._settings = settings;
    this._lfmCache = lfmCacheRef;
    this._session = httpSession();

    ensureCacheDir();

    this.add_style_class_name('eversolo-button');
    this._panelIcon = null;
    this._coverBin  = null;
    this._rebuildPanelActor();

    // Clic medio en el botón del panel = Play/Pause sin abrir menú
    this._clickId = this.connect('button-press-event', (_actor, event) => {
      if (event.get_button && event.get_button() === 2) {
        this._sendCmd(CMD_TOGGLE);
        // refresco rápido para que el icono cambie
        GLib.timeout_add(GLib.PRIORITY_DEFAULT, 400, () => {
          this._refresh().catch(e => log(`[Eversolo] post-click refresh: ${e}`));
          return GLib.SOURCE_REMOVE;
        });
        return Clutter.EVENT_STOP;
      }
      return Clutter.EVENT_PROPAGATE;
    });

    // Tooltip
    this._tooltip = new St.Label({ text: 'Eversolo', style_class: 'eversolo-tooltip' });
    this._tooltip.hide();
    Main.layoutManager.addTopChrome(this._tooltip);
    this._enterId = this.connect('enter-event', () => this._showTooltip());
    this._leaveId = this.connect('leave-event', () => this._hideTooltip());

    // Info en menú
    this._title  = new St.Label({ text: '—', style_class: 'eversolo-title' });
    this._artist = new St.Label({ text: '',  style_class: 'eversolo-artist' });
    this._info   = new St.Label({ text: '',  style_class: 'eversolo-info' });
    this._extra  = new St.Label({ text: '',  style_class: 'eversolo-extra' }); // modelo • proveedor
    this._menuCover = new St.Bin({ width: MENU_COVER_SIZE, height: MENU_COVER_SIZE, style_class: 'eversolo-cover-menu' });

    const headerBox = new St.BoxLayout({ vertical: true, style_class: 'eversolo-header' });
    headerBox.add_child(this._title);
    headerBox.add_child(this._artist);
    headerBox.add_child(this._info);
    headerBox.add_child(this._extra);

    const topRow = new St.BoxLayout({ x_expand: true });
    topRow.add_child(this._menuCover);
    topRow.add_child(new St.Widget({ width: 12 }));
    topRow.add_child(headerBox);

    const topItem = new PopupMenu.PopupBaseMenuItem({ reactive: false, can_focus: false });
    topItem.add_child(topRow);
    this.menu.addMenuItem(topItem);

    this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());

    const ctrls = new St.BoxLayout({ style_class: 'eversolo-ctrls' });
    this._btnPrev = this._makeCtrlButton('media-skip-backward-symbolic', () => this._sendCmd(CMD_PREV));
    const toggleCtl = this._makeCtrlButton('media-playback-start-symbolic', () => this._sendCmd(CMD_TOGGLE));
    this._btnToggle = toggleCtl.btn;
    this._btnToggleIcon = toggleCtl.icon;
    this._btnNext = this._makeCtrlButton('media-skip-forward-symbolic', () => this._sendCmd(CMD_NEXT));

    ctrls.add_child(this._btnPrev.btn || this._btnPrev);
    ctrls.add_child(this._btnToggle);
    ctrls.add_child(this._btnNext.btn || this._btnNext);

    const ctrlItem = new PopupMenu.PopupBaseMenuItem({ reactive: false, can_focus: false });
    ctrlItem.add_child(ctrls);
    this.menu.addMenuItem(ctrlItem);

    this._timeout = GLib.timeout_add_seconds(GLib.PRIORITY_DEFAULT, REFRESH_SECS, () => {
      if (!this._alive) return GLib.SOURCE_REMOVE;
      this._refresh().catch(e => log(`[Eversolo] refresh error: ${e}`));
      return GLib.SOURCE_CONTINUE;
    });

    this._refresh().catch(e => log(`[Eversolo] initial refresh error: ${e}`));
  }

  _getBase()       { return (this._settings?.get_string('eversolo-base') || '').trim(); }
  _getPanelCover() { return !!(this._settings?.get_boolean('panel-cover')); }
  _getLfmKey()     { return (this._settings?.get_string('lastfm-apikey') || '').trim(); }

  _rebuildPanelActor() {
    if (this._panelIcon && !this._panelIcon.destroyed) this._panelIcon.destroy();
    if (this._coverBin  && !this._coverBin.destroyed)  this._coverBin.destroy();

    if (this._getPanelCover()) {
      this._coverBin = new St.Bin({ style_class: 'eversolo-cover-bin', width: COVER_SIZE, height: COVER_SIZE });
      this.add_child(this._coverBin);
    } else {
      this._panelIcon = new St.Icon({ icon_name: 'media-playback-start-symbolic', icon_size: 16 });
      this.add_child(this._panelIcon);
    }
  }

  _applySettings() { this._rebuildPanelActor(); }

  _makeCtrlButton(iconName, cb) {
    const btn = new St.Button({ style_class: 'eversolo-btn' });
    const icon = new St.Icon({ icon_name: iconName, icon_size: 18 });
    btn.set_child(icon);
    btn.connect('clicked', () => {
      if (!this._alive) return;
      cb();
      // refresco rápido para reflejar Play/Pause/Skip
      GLib.timeout_add(GLib.PRIORITY_DEFAULT, 400, () => {
        this._refresh().catch(e => log(`[Eversolo] post-click refresh: ${e}`));
        return GLib.SOURCE_REMOVE;
      });
    });
    return { btn, icon };
  }

  _positionTooltip() {
    if (!actorAlive(this) || !actorAlive(this._tooltip)) return;
    const [x, y] = this.get_transformed_position();
    const [w, h] = this.get_transformed_size();
    this._tooltip.set_position(Math.round(x), Math.round(y + h + 6));
  }
  _showTooltip() { if (actorAlive(this._tooltip)) { this._positionTooltip(); this._tooltip.show(); } }
  _hideTooltip() { if (actorAlive(this._tooltip)) this._tooltip.hide(); }

  async _sendCmd(path) {
    const base = this._getBase();
    if (!base) return;
    const msg = Soup.Message.new('GET', base + path);
    this._session.send_and_read_async(msg, GLib.PRIORITY_DEFAULT, null, () => {});
  }

  async _getLastFmCover(artist, track) {
    const key = this._getLfmKey();
    if (!key || !artist || !track) return null;
    const ck = `${norm(artist)}::${norm(track)}`;
    if (this._lfmCache[ck] !== undefined) return this._lfmCache[ck];

    const params = new URLSearchParams({ method: 'track.getInfo', api_key: key, artist, track, format: 'json' });
    const url = `${LASTFM_URL}?${params.toString()}`;
    const json = await httpGetJSON(this._session, url);
    const imgs = json?.track?.album?.image || [];
    const found = imgs.length ? imgs[imgs.length - 1]['#text'] : null;
    this._lfmCache[ck] = found;
    return found;
  }

  async _refresh() {
    if (!this._alive) return;
    try {
      const base = this._getBase();
      if (!base) {
        if (actorAlive(this._title))  this._title.text  = 'Eversolo';
        if (actorAlive(this._artist)) this._artist.text = 'Configure IP in Preferences';
        if (actorAlive(this._info))   this._info.text   = '';
        if (actorAlive(this._extra))  this._extra.text  = '';
        return;
      }

      const json = await httpGetJSON(this._session, base + STATE_PATH);
      if (!this._alive) return;
      const state = parseState(json);
      // --- corrección robusta del estado de reproducción ---
      let playing = false;

      // heurística base (por si no hay posición)
      if (json?.state === 4) playing = true;

      // si tenemos posición, manda la realidad:
      const pos = json?.everSoloPlayInfo?.currentPosition ?? json?.position ?? null;
      if (pos !== null) {
        if (this._prevPos !== null) {
          const delta = pos - this._prevPos;
          // si la posición sube claramente, está reproduciendo; si no sube, está en pausa
          if (delta > 200) playing = true;       // >200 ms entre refrescos
          else if (delta <= 0) playing = false;  // no avanza o retrocede
        }
        this._prevPos = pos;
      }

      // inyecta el valor correcto en el estado usado por la UI
      state.playing = playing;

      const safeSet = (label, text) => { if (actorAlive(label)) label.text = text; };

      if (!state) {
        safeSet(this._title, 'Eversolo');
        safeSet(this._artist, '— no signal —');
        safeSet(this._info, '');
        safeSet(this._extra, '');
        if (this._btnToggleIcon) this._btnToggleIcon.icon_name = 'media-playback-start-symbolic';
        if (!this._getPanelCover() && this._panelIcon && !this._panelIcon.destroyed)
          this._panelIcon.icon_name = 'media-playback-start-symbolic';
        return;
      }

      // Texto principal
      safeSet(this._title, state.title || '—');
      safeSet(this._artist, state.artist || '');
      safeSet(this._info, formatInfo(state));
      safeSet(this._extra,
        `${state.deviceModel || ''}${state.deviceModel && state.provider ? ' • ' : ''}${state.provider || ''}` +
        `${state.streamQuality ? ' • ' + state.streamQuality : ''}`
      );

      // Tooltip enriquecido
      if (actorAlive(this._tooltip)) {
        const statusTxt = state.playing ? '▶ Playing' : '⏸ Paused';
        const provTxt   = state.provider ? ` • ${state.provider}${state.streamQuality ? ' (' + state.streamQuality + ')' : ''}` : '';
        const modelTxt  = state.deviceModel ? ` • ${state.deviceModel}` : '';
        this._tooltip.text = `${state.title || ''} — ${state.artist || ''}\n${statusTxt}${provTxt}${modelTxt}`;
      }

      // Icono Play/Pause (botón de menú + panel sin miniatura)
      const toggleIcon = state.playing ? 'media-playback-pause-symbolic' : 'media-playback-start-symbolic';
      if (this._btnToggleIcon) this._btnToggleIcon.icon_name = toggleIcon;
      if (!this._getPanelCover() && this._panelIcon && !this._panelIcon.destroyed)
        this._panelIcon.icon_name = toggleIcon;

      // Carátula: preferimos la del estado; si no, Last.fm
      let coverUrl = state.cover_url && /^https?:\/\//.test(state.cover_url) ? state.cover_url : null;
      if (!coverUrl) coverUrl = await this._getLastFmCover(state.artist, state.title);
      if (!this._alive) return;

      if (coverUrl && coverUrl !== this._lastCoverUrl) {
        const ok = await httpGetToFile(this._session, coverUrl, COVER_FILE_MENU);
        if (ok && this._alive) {
          this._lastCoverUrl = coverUrl;
          setTextureFromPath(this._menuCover, COVER_FILE_MENU, MENU_COVER_SIZE);
          if (this._getPanelCover() && this._coverBin && !this._coverBin.destroyed)
            setTextureFromPath(this._coverBin, COVER_FILE_MENU, COVER_SIZE);
        }
      }

    } catch (e) {
      log(`[Eversolo] _refresh exception: ${e}`);
    }
  }

  destroy() {
    this._alive = false;
    if (this._timeout) { GLib.source_remove(this._timeout); this._timeout = 0; }
    if (this._enterId) { this.disconnect(this._enterId); this._enterId = 0; }
    if (this._leaveId) { this.disconnect(this._leaveId); this._leaveId = 0; }
    if (this._clickId) { this.disconnect(this._clickId); this._clickId = 0; }
    if (this._tooltip) { this._tooltip.destroy(); this._tooltip = null; }
    try { this._session?.abort(); } catch {}
    super.destroy();
  }
});

// === ENTRYPOINT ==============================================================
export default class EversoloExtension extends Extension {
  enable() {
    this._settings = this.getSettings();
    this._lfmCache = {};

    ensureCacheDir();

    this._indicator = new Indicator(this._settings, this._lfmCache);
    this._settingsChangedId = this._settings.connect('changed', () => {
      this._indicator?._applySettings();
      this._indicator?._refresh?.();
    });

    Main.panel.addToStatusArea('eversolo-nowplaying', this._indicator, 0, 'right');
  }

  disable() {
    if (this._settingsChangedId) {
      this._settings.disconnect(this._settingsChangedId);
      this._settingsChangedId = 0;
    }
    if (this._indicator) {
      try { this._indicator._session?.abort(); } catch {}
      this._indicator.destroy();
      this._indicator = null;
    }
    this._settings = null;
    this._lfmCache = null;
  }
}

