// GNOME 46–48 — Eversolo Now Playing (UI + Last.fm + GSettings + toggle miniature)

import St from 'gi://St';
import Gio from 'gi://Gio';
import GLib from 'gi://GLib';
import GObject from 'gi://GObject';
import Soup from 'gi://Soup';
import Pango from 'gi://Pango';


import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import * as PopupMenu from 'resource:///org/gnome/shell/ui/popupMenu.js';
import { Extension } from 'resource:///org/gnome/shell/extensions/extension.js';

// === CONSTS ==================================================================
const REFRESH_SECS = 5;
const COVER_SIZE = 24;
const MENU_COVER_SIZE = 230;

const STATE_PATH = '/ZidooMusicControl/v2/getState';
const CMD_PREV   = '/ZidooMusicControl/v2/playLast';
const CMD_TOGGLE = '/ZidooMusicControl/v2/playOrPause';
const CMD_NEXT   = '/ZidooMusicControl/v2/playNext';

const LASTFM_URL = 'https://ws.audioscrobbler.com/2.0/';
const CACHE_DIR       = GLib.build_filenamev([GLib.get_user_cache_dir(), 'eversolo-nowplaying']);
const COVER_FILE_MENU = GLib.build_filenamev([CACHE_DIR, 'cover_menu.jpg']);

// === UTILS ===================================================================
function _isPlaying(json) {
  const ps = json?.everSoloPlayInfo?.playStatus;
  if (ps === 2) return true;
  if (ps === 3 || ps === 1) return false;
  return json?.state === 4;
}

function ensureCacheDir() {
  try {
    const f = Gio.File.new_for_path(CACHE_DIR);
    if (!f.query_exists(null)) f.make_directory_with_parents(null);
  } catch (e) { log(`[Eversolo] couldn't create cache: ${e}`); }
}

function httpSession() {
  const s = new Soup.Session();
  s.timeout = REFRESH_SECS;
  return s;
}

async function httpGetJSON(session, url) {
  return new Promise((resolve) => {
    const msg = Soup.Message.new('GET', url);
    session.send_and_read_async(msg, GLib.PRIORITY_DEFAULT, null, (sess, res) => {
      try {
        const bytes = sess.send_and_read_finish(res);
        const data  = new TextDecoder('utf-8').decode(bytes.get_data());
        try { resolve(JSON.parse(data)); } catch { resolve(null); }
      } catch { resolve(null); }
    });
  });
}

async function httpGetToFile(session, url, destPath) {
  return new Promise((resolve) => {
    const msg = Soup.Message.new('GET', url);
    session.send_and_read_async(msg, GLib.PRIORITY_DEFAULT, null, (sess, res) => {
      try {
        const bytes = sess.send_and_read_finish(res);
        const data = bytes.get_data();
        if (!data?.length) return resolve(false);
        Gio.File.new_for_path(destPath)
          .replace_contents(data, null, false, Gio.FileCreateFlags.REPLACE_DESTINATION, null);
        resolve(true);
      } catch (e) { log(`[Eversolo] Error saving cover: ${e}`); resolve(false); }
    });
  });
}

function formatInfo(o) {
  const parts = [];
  if (o.quality) parts.push(String(o.quality));
  if (o.bits) parts.push(String(o.bits).replace(/-bit$/, ''));
  if (o.samplerate) parts.push(String(o.samplerate));
  return parts.join(' – ');
}

function parseState(json) {
  if (!json) return null;
  const roonActive = !!(json?.everSoloPlayInfo?.everSoloPlayAudioInfo?.songName);
  if (roonActive) {
    const i = json.everSoloPlayInfo.everSoloPlayAudioInfo;
    const o = json.everSoloPlayOutputInfo || {};
    return {
      title: i.songName || 'Unknown',
      artist: i.artistName || '',
      album: i.albumName || '',
      cover_url: i.albumUrl || '',
      quality: o.outPutDecodec || 'PCM',
      bits: i.audioBitsPerSample > 0 ? `${i.audioBitsPerSample}` : '',
      samplerate: i.audioSampleRate > 0 ? `${Math.floor(i.audioSampleRate / 1000)} kHz` : '',
    };
  }
  const p = json.playingMusic || {};
  return {
    title: p.title || 'Unknown',
    artist: p.artist || '',
    album: p.album || '',
    cover_url: p.albumArt || '',
    quality: p.audioQuality || '',
    bits: p.bits || '',
    samplerate: p.sampleRate || '',
  };
}

function setTextureFromPath(bin, path, size) {
  try {
    const child = bin.get_child && bin.get_child();
    if (child) child.destroy();
    const uri = Gio.File.new_for_path(path).get_uri();
    const bust = `${uri}?t=${Date.now()}`;
    bin.set_style(
      `background-image: url("${bust}");` +
      `background-size: cover; background-position: center;` +
      `width: ${size}px; height: ${size}px; border-radius: ${Math.round(size*0.15)}px;`
    );
  } catch (e) { log(`[Eversolo] Error showing cover: ${e}`); }
}

function actorAlive(a) { return a && !a.destroyed && a.get_stage?.(); }
const norm = s => (s || '').toLowerCase().trim();

// === INDICATOR ===============================================================
const Indicator = GObject.registerClass(
class Indicator extends PanelMenu.Button {
  _init(settings, lfmCacheRef) {
    super._init(0.0, 'Eversolo Now Playing');

    this.set_x_expand(false);
    this.set_y_expand(false);
    this.set_style(`min-width: ${COVER_SIZE + 8}px; max-width: ${COVER_SIZE + 8}px; padding: 0 4px;`);

    this.name = 'eversolo-nowplaying';
    this._alive = true;
    this._lastCoverUrl = '';
    this._settings = settings;
    this._lfmCache = lfmCacheRef;
    this._session = httpSession();

    ensureCacheDir();

    this.add_style_class_name('eversolo-button');

    this._panelBox = new St.Bin({
      width:  COVER_SIZE + 8,
      height: Math.max(COVER_SIZE, 24),
      x_expand: false,
      y_expand: false,
    });
    this.add_child(this._panelBox);

    this._panelIcon = null;
    this._coverBin  = null;
    this._rebuildPanelActor();

    // Tooltip (max width + wrap)
    this._tooltip = new St.Label({ text: 'Eversolo', style_class: 'eversolo-tooltip' });
    const TOOLTIP_MAX_W = 480;
    this._tooltip.set_style(`max-width: ${TOOLTIP_MAX_W}px; padding: 6px 8px;`);
    {
        const tct = this._tooltip.get_clutter_text?.();
        if (tct) {
          if (typeof tct.set_single_line_mode === 'function')
            tct.set_single_line_mode(false);
          else if (typeof tct.set_single_line === 'function')
            tct.set_single_line(false);

          if (typeof tct.set_line_wrap === 'function')
            tct.set_line_wrap(true);
          if (typeof tct.set_line_wrap_mode === 'function')
            tct.set_line_wrap_mode(Pango.WrapMode.WORD_CHAR);
          if (typeof tct.set_ellipsize === 'function')
            tct.set_ellipsize(Pango.EllipsizeMode.NONE);
        }

    }
    this._tooltip.hide();
    Main.layoutManager.addTopChrome(this._tooltip);

    this._enterId = this.connect('enter-event', () => this._showTooltip());
    this._leaveId = this.connect('leave-event', () => this._hideTooltip());

    // Info en menú
    this._title  = new St.Label({ text: '—', style_class: 'eversolo-title' });
    this._artist = new St.Label({ text: '',  style_class: 'eversolo-artist' });
    this._info   = new St.Label({ text: '',  style_class: 'eversolo-info' });
    this._extra  = new St.Label({ text: '',  style_class: 'eversolo-extra' }); // opcional: modelo • proveedor
    const MENU_TEXT_MAX_W = 420; // ajustable
    for (const lbl of [this._title, this._artist, this._info]) {
      lbl.set_style(`max-width: ${MENU_TEXT_MAX_W}px;`);

      const ct = lbl.get_clutter_text?.();
      if (ct) {
        if (typeof ct.set_single_line_mode === 'function')
          ct.set_single_line_mode(false);
        else if (typeof ct.set_single_line === 'function')
          ct.set_single_line(false);

        if (typeof ct.set_line_wrap === 'function')
          ct.set_line_wrap(true);
        if (typeof ct.set_line_wrap_mode === 'function')
          ct.set_line_wrap_mode(Pango.WrapMode.WORD_CHAR);
        if (typeof ct.set_ellipsize === 'function')
          ct.set_ellipsize(Pango.EllipsizeMode.NONE);
      }

    }

    this._menuCover = new St.Bin({
      width:  MENU_COVER_SIZE,
      height: MENU_COVER_SIZE,
      style_class: 'eversolo-cover-menu'
    });

    const headerBox = new St.BoxLayout({ vertical: true, style_class: 'eversolo-header' });
    headerBox.add_child(this._title);
    headerBox.add_child(this._artist);
    headerBox.add_child(this._info);
    headerBox.add_child(this._extra);
    headerBox.set_x_expand(true);

    const topRow = new St.BoxLayout({ x_expand: true });
    topRow.add_child(this._menuCover);
    topRow.add_child(new St.Widget({ width: 12 }));
    topRow.add_child(headerBox);

    const topItem = new PopupMenu.PopupBaseMenuItem({ reactive: false, can_focus: false });
    topItem.add_child(topRow);
    topItem.add_style_class_name('eversolo-popup');
    this.menu.addMenuItem(topItem);

    this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());

    const ctrls = new St.BoxLayout({ style_class: 'eversolo-ctrls' });

    const prevCtl   = this._makeCtrlButton('media-skip-backward-symbolic', () => this._sendCmd(CMD_PREV));
    const toggleCtl = this._makeCtrlButton('media-playback-start-symbolic', () => this._sendCmd(CMD_TOGGLE));
    const nextCtl   = this._makeCtrlButton('media-skip-forward-symbolic', () => this._sendCmd(CMD_NEXT));

    this._btnToggle     = toggleCtl.btn;
    this._btnToggleIcon = toggleCtl.icon;

    ctrls.add_child(prevCtl.btn);
    ctrls.add_child(this._btnToggle);
    ctrls.add_child(nextCtl.btn);

    const ctrlItem = new PopupMenu.PopupBaseMenuItem({ reactive: false, can_focus: false });
    ctrlItem.add_child(ctrls);
    this.menu.addMenuItem(ctrlItem);

    this._timeout = GLib.timeout_add_seconds(GLib.PRIORITY_DEFAULT, REFRESH_SECS, () => {
      if (!this._alive) return GLib.SOURCE_REMOVE;
      this._refresh().catch(e => log(`[Eversolo] refresh error: ${e}`));
      return GLib.SOURCE_CONTINUE;
    });

    this._refresh().catch(e => log(`[Eversolo] initial refresh error: ${e}`));
  }

  _getBase()       { return (this._settings?.get_string('eversolo-base') || '').trim(); }
  _getPanelCover() { return !!(this._settings?.get_boolean('panel-cover')); }
  _getLfmKey()     { return (this._settings?.get_string('lastfm-apikey') || '').trim(); }

  _rebuildPanelActor() {
    if (this._panelIcon && !this._panelIcon.destroyed) this._panelIcon.destroy();
    if (this._coverBin  && !this._coverBin.destroyed)  this._coverBin.destroy();

    let child;
    if (this._getPanelCover()) {
      this._coverBin = new St.Bin({ style_class: 'eversolo-cover-bin', width: COVER_SIZE, height: COVER_SIZE });
      child = this._coverBin;
    } else {
      this._panelIcon = new St.Icon({ icon_name: 'media-playback-start-symbolic', icon_size: 16 });
      child = this._panelIcon;
    }

  if (this._panelBox && !this._panelBox.destroyed)
    this._panelBox.set_child(child);
}

  _applySettings() { this._rebuildPanelActor(); }

  _makeCtrlButton(iconName, cb) {
    const btn = new St.Button({ style_class: 'eversolo-btn' });
    const icon = new St.Icon({ icon_name: iconName, icon_size: 18 });
    btn.set_child(icon);

    btn.connect('clicked', () => {
      if (!this._alive) return;
      cb();
      GLib.timeout_add(GLib.PRIORITY_DEFAULT, 350, () => {
        this._refresh().catch(e => log(`[Eversolo] post-click refresh: ${e}`));
        return GLib.SOURCE_REMOVE;
      });
    });

    return { btn, icon };
  }

  _positionTooltip() {
    if (!actorAlive(this) || !actorAlive(this._tooltip)) return;

    const monitor = Main.layoutManager.monitorForActor?.(this) ?? { index: Main.layoutManager.primaryIndex };
    const workArea = Main.layoutManager.getWorkAreaForMonitor(monitor.index);

    const [minW, natW] = this._tooltip.get_preferred_width(-1);
    const width = Math.min(natW, workArea.width);
    const [minH, natH] = this._tooltip.get_preferred_height(width);
    const height = Math.min(natH, workArea.height);

    const [bx, by] = this.get_transformed_position();
    const [bw, bh] = this.get_transformed_size();
    let x = Math.round(bx + (bw - width) / 2);
    let y = Math.round(by + bh + 6);

    const minX = workArea.x;
    const minY = workArea.y;
    const maxX = workArea.x + workArea.width  - width;
    const maxY = workArea.y + workArea.height - height;
    x = Math.max(minX, Math.min(x, maxX));
    y = Math.max(minY, Math.min(y, maxY));

    this._tooltip.set_position(x, y);
  }

  _showTooltip() { if (actorAlive(this._tooltip)) { this._positionTooltip(); this._tooltip.show(); } }
  _hideTooltip() { if (actorAlive(this._tooltip)) this._tooltip.hide(); }

  async _sendCmd(path) {
    const base = this._getBase();
    if (!base) return;
    const msg = Soup.Message.new('GET', base + path);
    this._session.send_and_read_async(msg, GLib.PRIORITY_DEFAULT, null, () => {});
  }

  async _getLastFmCover(artist, track) {
    const key = this._getLfmKey();
    if (!key || !artist || !track) return null;
    const ck = `${norm(artist)}::${norm(track)}`;
    if (this._lfmCache[ck] !== undefined) return this._lfmCache[ck];

    const params = new URLSearchParams({ method: 'track.getInfo', api_key: key, artist, track, format: 'json' });
    const url = `${LASTFM_URL}?${params.toString()}`;
    const json = await httpGetJSON(this._session, url);
    const imgs = json?.track?.album?.image || [];
    const found = imgs.length ? imgs[imgs.length - 1]['#text'] : null;
    this._lfmCache[ck] = found;
    return found;
  }

  async _refresh() {
    if (!this._alive) return;
    try {
      const base = this._getBase();
      if (!base) {
        if (actorAlive(this._title))  this._title.text  = 'Eversolo';
        if (actorAlive(this._artist)) this._artist.text = 'Configure IP in Preferences';
        if (actorAlive(this._info))   this._info.text   = '';
        return;
      }

      const json = await httpGetJSON(this._session, base + STATE_PATH);
      if (!this._alive) return;

      let playing = false;

      const ps = json?.everSoloPlayInfo?.playStatus;
      if (ps === 2) playing = true;
      if (ps === 3 || ps === 1) playing = false;

      if (json?.state === 4) playing = true;

      const pos = json?.everSoloPlayInfo?.currentPosition ?? json?.position;
      if (typeof pos === 'number') {
        if (typeof this._lastPos === 'number') {
          const delta = pos - this._lastPos;
          if (delta > 200) playing = true;
          else if (delta <= 0) playing = false;
        }
        this._lastPos = pos;
      }

      const state = parseState(json);
      const safeSet = (label, text) => { if (actorAlive(label)) label.text = text; };

      if (!state) {
        safeSet(this._title, 'Eversolo');
        safeSet(this._artist, '— no signal —');
        safeSet(this._info, '');
        if (this._btnToggleIcon) this._btnToggleIcon.icon_name = 'media-playback-start-symbolic';
        if (!this._getPanelCover() && this._panelIcon && !this._panelIcon.destroyed)
          this._panelIcon.icon_name = 'media-playback-start-symbolic';
        return;
      }

      safeSet(this._title, state.title || '—');
      safeSet(this._artist, state.artist || '');
      safeSet(this._info, formatInfo(state));

      const toggleIcon = playing ? 'media-playback-pause-symbolic' : 'media-playback-start-symbolic';
      if (this._btnToggleIcon) this._btnToggleIcon.icon_name = toggleIcon;
      if (!this._getPanelCover() && this._panelIcon && !this._panelIcon.destroyed)
        this._panelIcon.icon_name = toggleIcon;

      let coverUrl = state.cover_url && /^https?:\/\//.test(state.cover_url) ? state.cover_url : null;
      if (!coverUrl) coverUrl = await this._getLastFmCover(state.artist, state.title);
      if (!this._alive) return;

      if (coverUrl && coverUrl !== this._lastCoverUrl) {
        const ok = await httpGetToFile(this._session, coverUrl, COVER_FILE_MENU);
        if (ok && this._alive) {
          this._lastCoverUrl = coverUrl;
          setTextureFromPath(this._menuCover, COVER_FILE_MENU, MENU_COVER_SIZE);
          if (this._getPanelCover() && this._coverBin && !this._coverBin.destroyed)
            setTextureFromPath(this._coverBin, COVER_FILE_MENU, COVER_SIZE);
        }
      }

    } catch (e) {
      log(`[Eversolo] _refresh exception: ${e}`);
    }
  }

  destroy() {
    this._alive = false;
    if (this._timeout) { GLib.source_remove(this._timeout); this._timeout = 0; }
    if (this._enterId) { this.disconnect(this._enterId); this._enterId = 0; }
    if (this._leaveId) { this.disconnect(this._leaveId); this._leaveId = 0; }
    if (this._tooltip) { this._tooltip.destroy(); this._tooltip = null; }
    try { this._session?.abort(); } catch {} // abort() al salir
    super.destroy();
  }
});

// === ENTRYPOINT ==============================================================
export default class EversoloExtension extends Extension {
  enable() {
    this._settings = this.getSettings();
    this._lfmCache = {};

    ensureCacheDir();

    this._indicator = new Indicator(this._settings, this._lfmCache);
    this._settingsChangedId = this._settings.connect('changed', () => {
      this._indicator?._applySettings();
      this._indicator?._refresh?.();
    });

    Main.panel.addToStatusArea('eversolo-nowplaying', this._indicator, 0, 'right');
  }

  disable() {
    if (this._settingsChangedId) {
      this._settings.disconnect(this._settingsChangedId);
      this._settingsChangedId = 0;
    }
    if (this._indicator) {
      this._indicator.destroy();
      this._indicator = null;
    }
    this._settings = null;
    this._lfmCache = null;
  }
}

