// SPDX-License-Identifier: MIT
// Numeric Clock — GNOME 45+ (ESM)
import Gio from 'gi://Gio';
import GLib from 'gi://GLib';
import St from 'gi://St';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as ExtensionUtils from 'resource:///org/gnome/shell/misc/extensionUtils.js';

const SCHEMA_ID = 'org.gnome.shell.extensions.numeric-clock';
const HOOKED = Symbol('nc-hooked');
const TIME_RE = /(^|\s)\d{1,2}:\d{2}(:\d{2})?(\s|$)/i;
const MONTHS = /\b(jan|feb|mar|apr|may|jun|jul|aug|sep|oct|nov|dec)\b/i;
const DAYS   = /\b(sun|mon|tue|wed|thu|fri|sat)\b/i;

function formatNow(fmt) {
  try { return GLib.DateTime.new_now_local().format(fmt); }
  catch (_) { return GLib.DateTime.new_now_local().format('%d/%m/%Y %H:%M'); }
}

function looksLikeClock(txt) {
  if (!txt) return false;
  const s = String(txt);
  return TIME_RE.test(s) || MONTHS.test(s) || DAYS.test(s);
}

function allStageLabels() {
  const out = [];
  (function walk(a) {
    if (!a || typeof a.get_children !== 'function') return;
    for (const c of a.get_children()) {
      if (c instanceof St.Label) out.push(c);
      walk(c);
    }
  })(global.stage);
  return out;
}

export default class Extension {
  constructor() {
    this._settings = ExtensionUtils.getSettings(SCHEMA_ID);
    this._timeoutId = 0;
    this._stageAddedId = 0;
    this._stageRemovedId = 0;
    this._settingsChangedIds = [];
    this._textSignalIds = []; // [ [actor, id], ... ]
  }

  _collectClockLabels() {
    const set = new Set();
    const dm = Main.panel?.statusArea?.dateMenu;
    if (dm) {
      if (dm._clockDisplay instanceof St.Label) set.add(dm._clockDisplay);
      if (dm._clock        instanceof St.Label) set.add(dm._clock);
      if (dm._time         instanceof St.Label) set.add(dm._time);
    }
    for (const lab of allStageLabels()) {
      try {
        const cls  = (lab.get_style_class_name?.() || '').toLowerCase();
        const name = (lab.get_name?.() || '').toLowerCase();
        const txt  = lab.get_text?.() || '';
        if (cls.includes('clock') || name.includes('clock') ||
            cls.includes('date')  || name.includes('date')  ||
            looksLikeClock(txt))
          set.add(lab);
      } catch (_) {}
    }
    return Array.from(set);
  }

  _forcePlain(lab) {
    try {
      if (lab?.clutter_text?.set_use_markup)
        lab.clutter_text.set_use_markup(false);
    } catch (_) {}
  }

  _applyTo(lab) {
    if (!lab) return;
    try {
      this._forcePlain(lab);
      if (lab.set_text)
        lab.set_text(formatNow(this._settings.get_string('format-string')));

      if (lab.clutter_text && !lab.clutter_text[HOOKED]) {
        lab.clutter_text[HOOKED] = true;
        const id = lab.clutter_text.connect('notify::text', () => {
          try {
            this._forcePlain(lab);
            lab.set_text(formatNow(this._settings.get_string('format-string')));
          } catch (_) {}
        });
        this._textSignalIds.push([lab.clutter_text, id]);
      }
    } catch (_) {}
  }

  _updateAllNow() {
    for (const lab of this._collectClockLabels())
      this._applyTo(lab);
  }

  _restartTimer() {
    if (this._timeoutId) {
      GLib.source_remove(this._timeoutId);
      this._timeoutId = 0;
    }
    const sec = Math.max(1, this._settings.get_int('update-interval'));
    this._updateAllNow();
    this._timeoutId = GLib.timeout_add_seconds(GLib.PRIORITY_DEFAULT, sec, () => {
      this._updateAllNow();
      return GLib.SOURCE_CONTINUE;
    });
  }

  enable() {
    this._settingsChangedIds.push(
      this._settings.connect('changed::format-string', () => this._updateAllNow()),
      this._settings.connect('changed::update-interval', () => this._restartTimer()),
    );
    this._stageAddedId   = global.stage.connect('actor-added',   () => this._updateAllNow());
    this._stageRemovedId = global.stage.connect('actor-removed', () => this._updateAllNow());
    this._restartTimer();
  }

  disable() {
    if (this._timeoutId) GLib.source_remove(this._timeoutId);
    this._timeoutId = 0;

    for (const id of this._settingsChangedIds) {
      try { this._settings.disconnect(id); } catch (_) {}
    }
    this._settingsChangedIds = [];

    if (this._stageAddedId)   { try { global.stage.disconnect(this._stageAddedId); } catch (_) {}   this._stageAddedId = 0; }
    if (this._stageRemovedId) { try { global.stage.disconnect(this._stageRemovedId); } catch (_) {} this._stageRemovedId = 0; }

    for (const [actor, id] of this._textSignalIds) {
      try { actor.disconnect(id); } catch (_) {}
      try { delete actor[HOOKED]; } catch (_) {}
    }
    this._textSignalIds = [];
  }
}
