import * as lib from './lib.js';
import * as log from './log.js';
import * as once_cell from './once_cell.js';
import * as Rect from './rectangle.js';
import * as Tags from './tags.js';
import * as utils from './utils.js';
import * as xprop from './xprop.js';
import * as focus from './focus.js';
import Meta from 'gi://Meta';
import Shell from 'gi://Shell';
import St from 'gi://St';
import GLib from 'gi://GLib';
import Gio from 'gi://Gio';
import Mtk from 'gi://Mtk';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as Config from 'resource:///org/gnome/shell/misc/config.js';
const { OnceCell } = once_cell;
const WM_TITLE_BLACKLIST = [
    'Firefox',
    'Nightly',
    'Tor Browser',
];
const [major] = Config.PACKAGE_VERSION.split('.').map((s) => Number(s));
export class ShellWindow {
    entity;
    meta;
    ext;
    known_workspace;
    grab = false;
    activate_after_move = false;
    ignore_detach = false;
    was_attached_to;
    destroying = false;
    active_hint_show_id = null;
    reassignment = false;
    smart_gapped = false;
    border = new St.Bin({
        style_class: 'gnome-mosaic-active-hint gnome-mosaic-border-normal',
    });
    prev_rect = null;
    window_app;
    was_hidden = false;
    extra = {
        normal_hints: new OnceCell(),
        wm_role_: new OnceCell(),
        xid_: new OnceCell(),
    };
    border_size = 0;
    constructor(entity, window, window_app, ext) {
        this.window_app = window_app;
        this.entity = entity;
        this.meta = window;
        this.ext = ext;
        this.known_workspace = this.workspace_id();
        if (this.meta.is_fullscreen()) {
            ext.add_tag(entity, Tags.Floating);
        }
        this.decorate(ext);
        this.bind_window_events();
        this.bind_hint_events();
        if (this.border)
            global.window_group.add_child(this.border);
        this.hide_border();
        this.update_border_layout();
        if (this.meta.get_compositor_private()?.get_stage())
            this.on_style_changed();
    }
    activate(move_mouse = true) {
        activate(this.ext, move_mouse, this.meta);
    }
    actor_exists() {
        return !this.destroying && this.meta.get_compositor_private() !== null;
    }
    bind_window_events() {
        this.ext.window_signals
            .get_or(this.entity, () => new Array())
            .push(this.meta.connect('size-changed', () => {
            this.window_changed();
        }), this.meta.connect('position-changed', () => {
            this.window_changed();
        }), this.meta.connect('workspace-changed', () => {
            this.workspace_changed();
        }), this.meta.connect('notify::wm-class', () => {
            this.wm_class_changed();
        }), this.meta.connect('raised', () => {
            this.window_raised();
        }));
    }
    bind_hint_events() {
        if (!this.border)
            return;
        let settings = this.ext.settings;
        let change_id = settings.ext.connect('changed', () => {
            return false;
        });
        this.border.connect('destroy', () => {
            settings.ext.disconnect(change_id);
        });
        this.border.connect('style-changed', () => {
            this.on_style_changed();
        });
    }
    cmdline() {
        let pid = this.meta.get_pid(), out = null;
        if (-1 === pid)
            return out;
        const path = '/proc/' + pid + '/cmdline';
        if (!utils.exists(path))
            return out;
        const result = utils.read_to_string(path);
        if (result.kind == 1) {
            out = result.value.trim();
        }
        else {
            log.error(`failed to fetch cmdline: ${result.value.format()}`);
        }
        return out;
    }
    async decorate(ext) {
        if (await this.may_decorate()) {
            if (!this.is_client_decorated()) {
                if (ext.settings.show_title()) {
                    this.decoration_show(ext);
                }
                else {
                    this.decoration_hide(ext);
                }
            }
        }
    }
    async decoration(_ext, callback) {
        if (await this.may_decorate()) {
            const xid = this.xid();
            if (xid)
                callback(xid);
        }
    }
    decoration_hide(ext) {
        if (this.ignore_decoration())
            return;
        this.was_hidden = true;
        this.decoration(ext, xid => xprop.set_hint(xid, xprop.MOTIF_HINTS, xprop.HIDE_FLAGS));
    }
    decoration_show(ext) {
        if (!this.was_hidden)
            return;
        this.decoration(ext, xid => xprop.set_hint(xid, xprop.MOTIF_HINTS, xprop.SHOW_FLAGS));
    }
    icon(_ext, size) {
        let icon = this.window_app.create_icon_texture(size);
        if (!icon) {
            icon = new St.Icon({
                icon_name: 'applications-other',
                icon_type: St.IconType.FULLCOLOR,
                icon_size: size,
            });
        }
        return icon;
    }
    ignore_decoration() {
        const name = this.meta.get_wm_class();
        if (name === null)
            return true;
        return WM_TITLE_BLACKLIST.findIndex(n => name.startsWith(n)) !== -1;
    }
    is_client_decorated() {
        const xid = this.xid();
        const extents = xid ? xprop.get_frame_extents(xid) : false;
        if (!extents)
            return false;
        return true;
    }
    is_maximized() {
        return this.meta.get_maximized() !== 0;
    }
    is_max_screen() {
        return (this.is_maximized() ||
            this.ext.settings.gap_inner() === 0 ||
            this.smart_gapped);
    }
    is_single_max_screen() {
        const display = this.meta.get_display();
        if (display) {
            let monitor_count = display.get_n_monitors();
            return ((this.is_maximized() || this.smart_gapped) && monitor_count == 1);
        }
        return false;
    }
    is_snap_edge() {
        return this.meta.get_maximized() == Meta.MaximizeFlags.VERTICAL;
    }
    is_tilable(ext) {
        let tile_checks = () => {
            let wm_class = this.meta.get_wm_class();
            if (wm_class !== null && wm_class.trim().length === 0) {
                wm_class = this.name(ext);
            }
            const role = this.meta.get_role();
            if (role === 'quake')
                return false;
            if (this.meta.get_title() === 'Steam') {
                const rect = this.rect();
                const is_dialog = rect.width < 400 && rect.height < 200;
                const is_first_login = rect.width === 432 && rect.height === 438;
                if (is_dialog || is_first_login)
                    return false;
            }
            if (wm_class !== null &&
                ext.conf.window_shall_float(wm_class, this.title())) {
                return ext.contains_tag(this.entity, Tags.ForceTile);
            }
            return (this.meta.window_type == Meta.WindowType.NORMAL &&
                !this.is_transient() &&
                wm_class !== null);
        };
        return !ext.contains_tag(this.entity, Tags.Floating) && tile_checks();
    }
    is_transient() {
        return this.meta.get_transient_for() !== null;
    }
    async may_decorate() {
        const xid = this.xid();
        return xid ? await xprop.may_decorate(xid) : false;
    }
    move(ext, rect, on_complete) {
        if (!this.same_workspace() && this.is_maximized()) {
            return;
        }
        this.hide_border();
        const max_width = ext.settings.max_window_width();
        if (max_width > 0 && rect.width > max_width) {
            rect.x += (rect.width - max_width) / 2;
            rect.width = max_width;
        }
        const clone = Rect.Rectangle.from_meta(rect);
        const meta = this.meta;
        const actor = meta.get_compositor_private();
        if (actor) {
            meta.unmaximize(Meta.MaximizeFlags.HORIZONTAL);
            meta.unmaximize(Meta.MaximizeFlags.VERTICAL);
            meta.unmaximize(Meta.MaximizeFlags.HORIZONTAL | Meta.MaximizeFlags.VERTICAL);
            actor.remove_all_transitions();
            ext.movements.insert(this.entity, clone);
            ext.register({ tag: 2, window: this, kind: { tag: 1 } });
            if (on_complete)
                ext.register_fn(on_complete);
            if (meta.appears_focused) {
                this.update_border_layout();
                ext.show_border_on_focused();
            }
        }
    }
    name(ext) {
        return ext.names.get_or(this.entity, () => 'unknown');
    }
    on_style_changed() {
        if (!this.border)
            return;
        this.border_size = this.border
            .get_theme_node()
            .get_border_width(St.Side.TOP);
    }
    rect() {
        return Rect.Rectangle.from_meta(this.meta.get_frame_rect());
    }
    async size_hint() {
        const xid = this.xid();
        const hint = xid ? await xprop.get_size_hints(xid) : null;
        return this.extra.normal_hints.get_or_init(() => hint);
    }
    swap(ext, other) {
        let ar = this.rect().clone();
        let br = other.rect().clone();
        other.move(ext, ar);
        this.move(ext, br, () => place_pointer_on(this.ext, this.meta));
    }
    title() {
        const title = this.meta.get_title();
        return title ? title : this.name(this.ext);
    }
    async wm_role() {
        const xid = this.xid();
        const role = xid ? await xprop.get_window_role(xid) : null;
        return this.extra.wm_role_.get_or_init(() => role);
    }
    workspace_id() {
        const workspace = this.meta.get_workspace();
        if (workspace) {
            return workspace.index();
        }
        else {
            this.meta.change_workspace_by_index(0, false);
            return 0;
        }
    }
    xid() {
        return this.extra.xid_.get_or_init(() => {
            if (utils.is_wayland())
                return null;
            return xprop.get_xid(this.meta);
        });
    }
    show_border() {
        if (!this.border)
            return;
        this.update_border_style();
        if (this.ext.settings.active_hint()) {
            const border = this.border;
            const permitted = () => {
                return (this.actor_exists() &&
                    this.ext.focus_window() == this &&
                    !this.meta.is_fullscreen() &&
                    (!this.is_single_max_screen() || this.is_snap_edge()) &&
                    !this.meta.minimized);
            };
            if (permitted()) {
                if (this.meta.appears_focused) {
                    border.show();
                    let applications = 0;
                    if (this.active_hint_show_id !== null) {
                        GLib.source_remove(this.active_hint_show_id);
                        this.active_hint_show_id = null;
                    }
                    this.active_hint_show_id = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 600, () => {
                        if ((applications > 4 && !this.same_workspace()) ||
                            !permitted()) {
                            this.active_hint_show_id = null;
                            return GLib.SOURCE_REMOVE;
                        }
                        applications += 1;
                        border.show();
                        return GLib.SOURCE_CONTINUE;
                    });
                }
            }
        }
    }
    same_workspace() {
        const workspace = this.meta.get_workspace();
        if (workspace) {
            let workspace_id = workspace.index();
            return (workspace_id ===
                global.workspace_manager.get_active_workspace_index());
        }
        return false;
    }
    same_monitor() {
        return this.meta.get_monitor() === global.display.get_current_monitor();
    }
    get always_top_windows() {
        let above_windows = new Array();
        for (const actor of global.get_window_actors()) {
            if (actor &&
                actor.get_meta_window() &&
                actor.get_meta_window().is_above())
                above_windows.push(actor);
        }
        return above_windows;
    }
    hide_border() {
        let b = this.border;
        if (b)
            b.hide();
    }
    update_border_layout() {
        let { x, y, width, height } = this.meta.get_frame_rect();
        const border = this.border;
        let borderSize = this.border_size;
        if (border) {
            if (!(this.is_max_screen() || this.is_snap_edge())) {
                border.remove_style_class_name('gnome-mosaic-border-maximize');
            }
            else {
                borderSize = 0;
                border.add_style_class_name('gnome-mosaic-border-maximize');
            }
            let dimensions = [
                x - borderSize,
                y - borderSize,
                width + 2 * borderSize,
                height + 2 * borderSize,
            ];
            if (dimensions) {
                [x, y, width, height] = dimensions;
                const workspace = this.meta.get_workspace();
                if (workspace === null)
                    return;
                const screen = workspace.get_work_area_for_monitor(this.meta.get_monitor());
                if (screen) {
                    width = Math.min(width, screen.x + screen.width);
                    height = Math.min(height, screen.y + screen.height);
                }
                border.set_position(x, y);
                border.set_size(width, height);
            }
        }
    }
    async update_border_style() {
        const { settings } = this.ext;
        const radii = await getBorderRadii(this.meta.get_compositor_private());
        const radii_values = radii?.map(v => `${v}px`).join(' ') || '0px 0px 0px 0px';
        if (this.border) {
            this.border.set_style(`border-radius: ${radii_values};` +
                `border-width: ${settings.active_hint_border_width()}px;` +
                `border-color: ${major > 46 ? '-st-accent-color' : settings.gnome_legacy_accent_color()}`);
        }
    }
    wm_class_changed() {
        if (this.is_tilable(this.ext)) {
            this.ext.connect_window(this);
            if (!this.meta.minimized) {
                this.ext.auto_tiler?.auto_tile(this.ext, this, this.ext.init);
            }
        }
    }
    window_changed() {
        this.update_border_layout();
        this.ext.show_border_on_focused();
    }
    window_raised() {
        this.ext.show_border_on_focused();
    }
    workspace_changed() { }
    timeouts_remove() {
        if (this.active_hint_show_id) {
            GLib.source_remove(this.active_hint_show_id);
            this.active_hint_show_id = null;
        }
    }
}
export function activate(ext, move_mouse, win) {
    try {
        if (!win.get_compositor_private())
            return;
        if (ext.get_window(win)?.destroying)
            return;
        if (win.is_override_redirect())
            return;
        const workspace = win.get_workspace();
        if (!workspace)
            return;
        win.unminimize();
        workspace.activate_with_focus(win, global.get_current_time());
        win.raise();
        const pointer_placement_permitted = move_mouse &&
            Main.modalCount === 0 &&
            ext.settings.mouse_cursor_follows_active_window() &&
            !pointer_already_on_window(win) &&
            pointer_in_work_area();
        if (pointer_placement_permitted) {
            place_pointer_on(ext, win);
        }
    }
    catch (error) {
        log.error(`failed to activate window: ${error}`);
    }
}
function pointer_in_work_area() {
    const cursor = lib.cursor_rect();
    const indice = global.display.get_current_monitor();
    const mon = global.display
        .get_workspace_manager()
        .get_active_workspace()
        .get_work_area_for_monitor(indice);
    return mon ? cursor.intersects(mon) : false;
}
function place_pointer_on(ext, win) {
    const rect = win.get_frame_rect();
    let x = rect.x;
    let y = rect.y;
    let key = Object.keys(focus.FocusPosition)[ext.settings.mouse_cursor_focus_location()];
    let pointer_position_ = focus.FocusPosition[key];
    switch (pointer_position_) {
        case focus.FocusPosition.TopLeft:
            x += 8;
            y += 8;
            break;
        case focus.FocusPosition.BottomLeft:
            x += 8;
            y += rect.height - 16;
            break;
        case focus.FocusPosition.TopRight:
            x += rect.width - 16;
            y += 8;
            break;
        case focus.FocusPosition.BottomRight:
            x += rect.width - 16;
            y += rect.height - 16;
            break;
        default:
            x += 8;
            y += 8;
    }
    global.stage
        .get_context()
        .get_backend()
        .get_default_seat()
        .warp_pointer(x, y);
}
function pointer_already_on_window(meta) {
    const cursor = lib.cursor_rect();
    return cursor.intersects(meta.get_frame_rect());
}
async function getBorderRadii(actor) {
    const opaqueLimit = 200;
    const margin = 6;
    const { x, y, width, height } = actor.get_meta_window().get_frame_rect();
    const monitorIndex = actor.get_meta_window().get_monitor();
    const scale = Math.ceil(global.display.get_monitor_scale(monitorIndex));
    if (height <= 0)
        return;
    const capture = actor.paint_to_content(new Mtk.Rectangle({ x, y, width, height }));
    if (!capture)
        return;
    const memoryBuffer = Gio.MemoryOutputStream.new_resizable();
    const surface = capture.get_texture();
    const imageBuf = await Shell.Screenshot.composite_to_stream(surface, 0, 0, width, height * scale, 1, null, 0, 0, 1, memoryBuffer);
    const rawPixels = imageBuf.get_pixels();
    if (!rawPixels)
        return;
    memoryBuffer.close(null);
    const scanAlpha = (start) => {
        for (let x = 0; x < width / 2; x++) {
            const idx = (start * width + x) * 4;
            const alpha = rawPixels[idx + 3];
            if (alpha > opaqueLimit) {
                return x;
            }
        }
        return -1;
    };
    let alphaTop = -1;
    for (var row = 0; row < 3; row++) {
        alphaTop = scanAlpha(row);
        if (alphaTop > -1)
            break;
    }
    if (alphaTop === -1)
        alphaTop = 0;
    let alphaBottom = -1;
    for (var row = height * scale - 1; row > height * scale - 4; row--) {
        alphaBottom = scanAlpha(row);
        if (alphaBottom > -1)
            break;
    }
    if (alphaBottom === -1)
        alphaBottom = 0;
    const radiusTop = alphaTop / scale + margin;
    const radiusBottom = alphaBottom / scale + margin;
    return [radiusTop, radiusTop, radiusBottom, radiusBottom];
}
