/* extension.js
 *
 * Copyright (c) 2025 Matteo Paone
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

import Clutter from 'gi://Clutter';
import GObject from 'gi://GObject';
import Gio from 'gi://Gio';
import St from 'gi://St';

import { Extension, gettext as _ } from 'resource:///org/gnome/shell/extensions/extension.js';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as MessageTray from 'resource:///org/gnome/shell/ui/messageTray.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';

const ICON_CAPS_LOCK = '/icons/caps-lock-symbolic.svg';
const ICON_NUM_LOCK = '/icons/num-lock-symbolic.svg';

const ICON_CAPS_LOCK_ON = '/icons/caps-lock-enabled-symbolic.svg';
const ICON_CAPS_LOCK_OFF = '/icons/caps-lock-disabled-symbolic.svg';
const ICON_NUM_LOCK_ON = '/icons/num-lock-enabled-symbolic.svg';
const ICON_NUM_LOCK_OFF = '/icons/num-lock-disabled-symbolic.svg';

function _get_keymap() {
  return Clutter.get_default_backend().get_default_seat().get_keymap();
}

/* custom icon */
class Icon extends St.Icon {
  static {
    GObject.registerClass(this);
  }

  constructor({settings, ...args}) {
    super(args);

    this._settings = settings;

    this.OPACITY_ON = 0xff;
    this.OPACITY_OFF = this._settings.get_double('icon-opacity-disabled') * 0xff;

    this.add_style_class_name('system-status-icon');
    this.set_style(`margin: 0; padding: 0 ${this._settings.get_int('icon-spacing')}pt;`);

    this._settings = settings;
    this._state = false;

    this._settings.connect('changed::always-show-icon', this._sync_always_show_icon.bind(this))
    this._settings.connect('changed::icon-opacity-disabled', this._sync_icon_opacity_disabled.bind(this))
    this._settings.connect('changed::icon-spacing', this._sync_icon_spacing.bind(this))
  }

  _sync_always_show_icon() {
    this.update(this._state);
  }

  _sync_icon_opacity_disabled() {
    this.update(this._state);
    this.OPACITY_OFF = this._settings.get_double('icon-opacity-disabled') * 0xff;
}

  _sync_icon_spacing() {
    this.set_style(`margin: 0; padding: 0 ${this._settings.get_int('icon-spacing')}pt;`);
  }

  update(state) {
    this._state = state;

    if (this._settings.get_boolean('always-show-icon')) {
      this.show();
      this.set_opacity(state ? this.OPACITY_ON : this.OPACITY_OFF);
    } else {
      this.set_opacity(this.OPACITY_ON);
      state ? this.show() : this.hide();
    }
  }
}

/* main button */
class LockKeysIndicator extends PanelMenu.Button {
  static {
    GObject.registerClass(this);
  }

  constructor({metadata, settings, ...args}) {
    super(0.0, metadata.name, false, args);

    this._settings = settings;

    this._notification = null;
    this._source = null; /* for the notification */

    // states
    this._caps_state = false;
    this._num_state = false;

    // icons
    this._caps_gicon = Gio.Icon.new_for_string(metadata.path + ICON_CAPS_LOCK);
    this._num_gicon = Gio.Icon.new_for_string(metadata.path + ICON_NUM_LOCK);

    this._caps_gicon_on = Gio.Icon.new_for_string(metadata.path + ICON_CAPS_LOCK_ON);
    this._caps_gicon_off = Gio.Icon.new_for_string(metadata.path + ICON_CAPS_LOCK_OFF);
    this._num_gicon_on = Gio.Icon.new_for_string(metadata.path + ICON_NUM_LOCK_ON);
    this._num_gicon_off = Gio.Icon.new_for_string(metadata.path + ICON_NUM_LOCK_OFF);

    // create the icons
    this._caps_icon = new Icon({ gicon: this._caps_gicon, settings: settings });
    this._num_icon = new Icon({ gicon: this._num_gicon, settings: settings });

    // add the box container for the icons
    const box = new St.BoxLayout();
    box.add_child(this._caps_icon);
    box.add_child(this._num_icon);

    this.add_child(box);


    // connect the signal for keymap
    this._icon_changed_id = _get_keymap().connect('state-changed', this.update.bind(this));

    this.update()
  }

  _notify_osd(icon, text) {
    Main.osdWindowManager.show( -1, icon, text );
  }

  _ensure_source() {
          if (!this._source) {
              this._source = new MessageTray.Source({title:'Lock Keys'});
              this._source.connect('destroy', () => {
                  this._source = null;
              });

              Main.messageTray.add(this._source);
          }

          return this._source;
  }

  _notify(title, body) {
    if (this._notification)
      this._notification.destroy();

    let source = this._ensure_source();
    
    this._notification = new MessageTray.Notification({
      source: source,
      title: title,
      body: body
    });
    this._notification.urgency = MessageTray.Urgency.HIGH;
    this._notification.connect('destroy', () => {
      this._notification = null;
    });
    this._source.addNotification(this._notification);
  }

  update() {
    let caps_state = _get_keymap().get_caps_lock_state();
    this._caps_icon.update(caps_state);

    if (caps_state != this._caps_state) {
      this._caps_state = caps_state;

      if (this._settings.get_boolean('emit-notifications'))
        switch (this._settings.get_string('notification-type')) {
          case 'osd':
            this._notify_osd(
              caps_state ? this._caps_gicon_on : this._caps_gicon_off,
              caps_state ? _('Caps lock on') : _('Caps lock off')
            );
            break;
          case 'notification':
            this._notify(
              caps_state ? _('Caps lock is enabled') : _('Caps lock is disabled'),
              caps_state ?
                _('You will now begin to type uppercase') :
                _('You will now begin to type normally')
            );
            break;
        }
    }

    let num_state = _get_keymap().get_num_lock_state();
    this._num_icon.update(num_state);

    if (num_state != this._num_state) {
      this._num_state = num_state;

      if (this._settings.get_boolean('emit-notifications'))
        switch (this._settings.get_string('notification-type')) {
          case 'osd':
            this._notify_osd(
              num_state ? this._num_gicon_on : this._num_gicon_off,
              num_state ? _('Num lock on') : _('Num lock off')
            );
            break;
          case 'notification':
            this._notify(
              num_state ? _('Num lock enabled') : _('Num lock disabled'),
              num_state ?
                _('From now on your numpad will be locked') :
                _('From now on your numpad will be unlocked')
            );
            break;
        }
    }
  }
}

/* extension */
export default class LockKeysExtension extends Extension {

  constructor({...args}) {
    super(args);
    this._indicator = null;
    this._settings = this.getSettings();
  }

  enable() {
    if (!this._settings)
      this._settings = this.getSettings();

    if (this._indicator)
      return;

    // create the indicator
    this._indicator = new LockKeysIndicator({
      metadata: this.metadata,
      settings: this._settings
    });

    // add the indicator to the panel
    Main.panel.addToStatusArea(this.uuid, this._indicator);
  }

  disable() {
    this._indicator?.destroy();
    this._indicator = null;
    this._settings = null;
  }
}
