/* prefs.js
 *
 * Copyright (c) 2025 Matteo Paone
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

import Adw from 'gi://Adw';
import Gtk from 'gi://Gtk';
import Gio from 'gi://Gio';

import { ExtensionPreferences, gettext as _ } from 'resource:///org/gnome/Shell/Extensions/js/extensions/prefs.js';

export default class GamemodePrefs extends ExtensionPreferences {
  fillPreferencesWindow(window) {
    /* create a preferences page, with a single group */
    const page = new Adw.PreferencesPage({
      title: _('General'),
      icon_name: 'dialog-information-symbolic',
    });
    window.add(page);

    const group = new Adw.PreferencesGroup({
      title: _('Behavior'),
      description: _('Configure the behavior of the extension'),
    });
    page.add(group);

    /* create preferences rows */
    // notification and notification type
    const osd_toggle = new Adw.Toggle({
      name: 'osd',
      label: _('On Screen Display'),
    });
    const not_toggle = new Adw.Toggle({
      name: 'notification',
      label: _('Notification'),
    });
    const nottype_group = new Adw.ToggleGroup();
    nottype_group.add(osd_toggle);
    nottype_group.add(not_toggle);
    const toggle_container = new Gtk.Box({ valign: Gtk.Align.CENTER });
    toggle_container.append(nottype_group);
    const nottype_row = new Adw.ActionRow({
      title: _('Type'),
      subtitle: _('Type of the notification displayed'),
    });
    nottype_row.add_suffix(toggle_container);

    const notification_row = new Adw.ExpanderRow({
      title: _('Notification'),
      subtitle: _('Emit a notification when the lock keys status changes'),
      show_enable_switch: true,
    })
    notification_row.add_row(nottype_row);
    group.add(notification_row);

    // opacity and always show
    const opacity_adj = Gtk.Adjustment.new(0.25, 0.2, 0.7, 0.05, 0.1, 0.1);
    const opacity_slider = new Gtk.Scale({
      orientation: Gtk.Orientation.HORIZONTAL,
      adjustment: opacity_adj,
      hexpand: true,
    });
    opacity_slider.add_mark(0.5, Gtk.PositionType.BOTTOM, null);
    opacity_slider.add_mark(0.4, Gtk.PositionType.BOTTOM, null);
    opacity_slider.add_mark(0.25, Gtk.PositionType.BOTTOM, null);
    const opacity_row = new Adw.ActionRow({
        title: _('Icons opacity'),
        subtitle: ('Set the opacity when the icons are disabled'),
    })
    opacity_row.add_suffix(opacity_slider);

    const show_icons_row = new Adw.ExpanderRow({
      title: _('Always show the icons'),
      subtitle: _('Display the icons even if lock keys are currently not active'),
      show_enable_switch: true,
    });
    show_icons_row.add_row(opacity_row);
    group.add(show_icons_row);
    
    // spacing icons
    const spacing_row = new Adw.SpinRow({
      title: _('Icon spacing'),
      subtitle: _('Spacing between icons'),
      climb_rate: 0.2,
      adjustment: Gtk.Adjustment.new( 2, 0, 8, 1, 0, 0 ),
    });
    group.add(spacing_row);
    
    /* bind settings */
    window._settings = this.getSettings();
    window._settings.bind('emit-notifications', notification_row, 'enable-expansion',
            Gio.SettingsBindFlags.DEFAULT);
    window._settings.bind('notification-type', nottype_group, 'active-name',
            Gio.SettingsBindFlags.DEFAULT);
    window._settings.bind('icon-spacing', spacing_row, 'value',
            Gio.SettingsBindFlags.DEFAULT);
    window._settings.bind('icon-opacity-disabled', opacity_adj, 'value',
            Gio.SettingsBindFlags.DEFAULT);
    window._settings.bind('always-show-icon', show_icons_row, 'enable-expansion',
            Gio.SettingsBindFlags.DEFAULT);

  }
}



