import GObject from 'gi://GObject';
import Direction from '../enums/direction.js';
import Settings from '../helpers/settings.js';
import * as windowHelper from '../helpers/window.js';
import * as screenHelper from '../helpers/screen.js';

export default GObject.registerClass(
    class Snapper extends GObject.Object {
        _keybind;

        constructor(keybind) {
            super();

            this._keybind = keybind;
            this._keybind.registerKeybind(Settings.KEY_SNAP_RIGHT, this.snapRight.bind(this));
            this._keybind.registerKeybind(Settings.KEY_SNAP_LEFT, this.snapLeft.bind(this));
            this._keybind.registerKeybind(Settings.KEY_SNAP_UP, this.snapUp.bind(this));
            this._keybind.registerKeybind(Settings.KEY_SNAP_DOWN, this.snapDown.bind(this));
        }

        snapRight() {
            this._snap(Direction.Right);
        }

        snapLeft() {
            this._snap(Direction.Left);
        }

        snapUp() {
            this._snap(Direction.Up);
        }

        snapDown() {
            this._snap(Direction.Down);
        }

        _snap(direction) {
            let window = windowHelper.getFocusedWindow();
            if (!window) {
                return;
            }

            let windows = this._getNearBySnapableWindow(window, direction);
            if (windows.length === 0) {
                this._snapToScreenEdge(direction, window);
                return;
            }

            let currentWindowSize = window.size;
            let otherWindowSize = windows[0].size;

            let updatePosition = Direction.isVertical(direction)
                ? direction === Direction.Down
                    ? otherWindowSize.y - currentWindowSize.height
                    : otherWindowSize.y + otherWindowSize.height
                : direction === Direction.Right
                    ? otherWindowSize.x - currentWindowSize.width
                    : otherWindowSize.x + otherWindowSize.width;

            let screen = screenHelper.getScreenSize(window.workspace);

            let isVertical = Direction.isVertical(direction);
            let overflowX = !isVertical && currentWindowSize.width + updatePosition > screen.width;
            let overflowY = isVertical && currentWindowSize.height + updatePosition > screen.height;

            if (windows.length === 0 || updatePosition < 0 || overflowX || overflowY) {
                this._snapToScreenEdge(direction, window);
                return;
            }

            if (isVertical) {
                currentWindowSize.y = updatePosition
            } else {
                currentWindowSize.x = updatePosition
            }

            windowHelper.resizeWindow(window, currentWindowSize);
        }

        _getNearBySnapableWindow(window, direction) {
            let windows = windowHelper.getNearbyWindows(window, direction);
            if (windows.length === 0) {
                return windows;
            }

            windows = windows.filter((otherWindow) => {
                if (
                    Math.abs(window.size.x - (otherWindow.size.x + otherWindow.size.width)) === 0 ||
                    Math.abs(window.size.x + window.size.width - otherWindow.size.x) === 0 ||
                    Math.abs(window.size.y - (otherWindow.size.y + otherWindow.size.height)) === 0 ||
                    Math.abs(window.size.y + window.size.height - otherWindow.size.y) === 0
                ) {
                    return false;
                }

                let otherWindowMin = Direction.isVertical(direction) ? otherWindow.size.x : otherWindow.size.y;

                let otherWindowMax = Direction.isVertical(direction)
                    ? otherWindow.size.x + otherWindow.size.width
                    : otherWindow.size.y + otherWindow.size.height;

                let windowMin = Direction.isVertical(direction) ? window.size.x : window.size.y;

                let windowMax = Direction.isVertical(direction)
                    ? window.size.x + window.size.width
                    : window.size.y + window.size.height;

                return (
                    (windowMin < otherWindowMax && windowMax > otherWindowMin) ||
                    (otherWindowMin < windowMax && otherWindowMax > windowMin)
                );
            });

            return windows;
        }

        _snapToScreenEdge(direction, window) {
            let workspace = window.workspace;
            let screenSize = screenHelper.getScreenSize(workspace);

            if (Direction.isVertical(direction)) {
                window.size.y =
                    direction === Direction.Up ? screenSize.y : screenSize.y + screenSize.height - window.size.height;
            } else {
                window.size.x =
                    direction === Direction.Left ? screenSize.x : screenSize.x + screenSize.width - window.size.width;
            }

            windowHelper.resizeWindow(window, window.size);
        }

        destroy() {
            this._keybind.removeKeybinding(Settings.KEY_SNAP_RIGHT);
            this._keybind.removeKeybinding(Settings.KEY_SNAP_LEFT);
            this._keybind.removeKeybinding(Settings.KEY_SNAP_UP);
            this._keybind.removeKeybinding(Settings.KEY_SNAP_DOWN);
        }
    }
);
