/* prefs.js
 *
 * This file is part of the Custom Command Toggle GNOME Shell extension
 * https://github.com/StorageB/custom-command-toggle
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */
 
import Gio from 'gi://Gio';
import Adw from 'gi://Adw';
import Gtk from 'gi://Gtk';
import Gdk from 'gi://Gdk';
import GObject from 'gi://GObject';

import {ExtensionPreferences, gettext as _} from 'resource:///org/gnome/Shell/Extensions/js/extensions/prefs.js';
import {releaseNotes} from './about.js';
import {KeybindingRow} from './keybinding.js';

import {exportConfiguration} from './backup.js';

let numButtons = 1;

export default class CustomCommandTogglePreferences extends ExtensionPreferences {
    fillPreferencesWindow(window) {

        window._settings = this.getSettings();

        // Number of toggle buttons to create
        numButtons = window._settings.get_int('numbuttons-setting');

        const pages = [];
        

        // Loop to create toggle button setting pages
        for (let pageIndex = 1; pageIndex <= numButtons; pageIndex++) {

            let buttonTitle = "";
            if (numButtons === 1) { buttonTitle = 'Toggle Button';
            } else { buttonTitle = `Button ${pageIndex}`; }

            const page = new Adw.PreferencesPage({
                title: _(buttonTitle),
                icon_name: 'utilities-terminal-symbolic',
            });
            window.add(page);
        
            const groups = [];
        

            //#region Appearance
            const group2 = new Adw.PreferencesGroup({
                title: _('Appearance'),
            });
            page.add(group2);
            groups.push(group2);
        
            const entryRow3 = new Adw.EntryRow({
                title: _('Button name:'),
            });
            group2.add(entryRow3);
        
            const entryRow4 = new Adw.EntryRow({
                title: _('Icon:'),
            });
            group2.add(entryRow4);
            //#endregion Appearance


            //#region Commands
            const group1 = new Adw.PreferencesGroup({
                title: _('Commands'),
            });
            page.add(group1);
            groups.push(group1);
        
            const entryRow1 = new Adw.EntryRow({
                title: _('Toggle ON Command:'),
            });
            group1.add(entryRow1);
        
            const entryRow2 = new Adw.EntryRow({
                title: _('Toggle OFF Command:'),
            });
            group1.add(entryRow2);

            const checkCommandRow = new Adw.EntryRow({
              title: _("Check Output Command:"),
            });
            group1.add(checkCommandRow);

            const checkRegexRow = new Adw.EntryRow({
                title: _("Search Term:"),
            });
            group1.add(checkRegexRow);
            //#endregion Commands
        

            //#region Startup Behavior
            const group3 = new Adw.PreferencesGroup({
                title: _('Startup Behavior'),
            });
            page.add(group3);
            groups.push(group3);

            const optionList = new Gtk.StringList();
            [_('On'), _('Off'), _('Previous state'), _('Command output')].forEach(choice => optionList.append(choice));
            const comboRow = new Adw.ComboRow({
                title: _('Initial State'),
                subtitle: _('State of the toggle button at login/startup'),
                model: optionList,
                selected: window._settings.get_int(`initialtogglestate${pageIndex}-setting`),
            });
            group3.add(comboRow);

            const checkCommandInfo = new Adw.ActionRow({
                title: _('Command Configuration'),
                subtitle: _(
                            'Enter the Check Output Command and Search Term in the Commands section above. If the specified Search Term appears ' +
                            'in the command\'s output, the button will be set to ON at startup. Otherwise, the button will be set to OFF.'
                           ),
                activatable: false,
            });
            checkCommandInfo.visible = comboRow.selected === 3;
            group3.add(checkCommandInfo);
  
            comboRow.connect("notify::selected", () => {
                checkCommandInfo.visible = comboRow.selected === 3;
                expanderRow.visible =  comboRow.selected === 0 || comboRow.selected === 1 || comboRow.selected === 2;
                spinRow2.visible = (comboRow.selected === 3);
            });

            const expanderRow = new Adw.ExpanderRow({
                title: _('Run Command at Startup'),
                subtitle: _('Run associated toggle command at login/startup'),
                show_enable_switch: true,
                expanded: window._settings.get_boolean(`runcommandatboot${pageIndex}-setting`),
                enable_expansion: window._settings.get_boolean(`runcommandatboot${pageIndex}-setting`),
            });
            expanderRow.visible =  comboRow.selected === 0 || comboRow.selected === 1 || comboRow.selected === 2;
            
            expanderRow.connect('notify::expanded', widget => {
                expanderRow.enable_expansion = widget.expanded;
            });
            group3.add(expanderRow);
        
            const spinRow = new Adw.SpinRow({
                title: _('Startup Delay (seconds)'),
                subtitle: _('Amount of time to delay command from running after startup \n' +
                            '(it may be required to allow other processes to finish loading before running the command)'),
                adjustment: new Gtk.Adjustment({
                    lower: 0,
                    upper: 10,
                    step_increment: 1,
                    page_increment: 1,
                }),
            });
            expanderRow.add_row(spinRow);

            const spinRow2 = new Adw.SpinRow({
                title: _('Startup Delay (seconds)'),
                subtitle: _('Amount of time to delay command from running after startup \n' +
                            '(it may be required to allow other processes to finish loading before running the command)'),
                adjustment: new Gtk.Adjustment({
                    lower: 0,
                    upper: 10,
                    step_increment: 1,
                    page_increment: 1,
                }),
            });
            spinRow2.visible =  (comboRow.selected === 3 && !window._settings.get_boolean(`checkcommandsync${pageIndex}-setting`));
            group3.add(spinRow2);
            //#endregion Startup Behavior


            //#region Toggle Behavior
            const group4 = new Adw.PreferencesGroup({
                title: _('Toggle Behavior'),
            });
            groups.push(group4);
            
            const toggleList = new Gtk.StringList();
            [_('Always on'), _('Always off'), _('Toggle')].forEach(choice => toggleList.append(choice));
        
            const comboRow2 = new Adw.ComboRow({
                title: _('Button Click Action'),
                subtitle: _(
                    '• Toggle: Button will toggle on/off when clicked (default action)\n' +
                    '• Always on/off: Button will remain in the selected on or off state when clicked ' +
                    'and only execute the associated on or off command'
                ),
                model: toggleList,
            });
            group4.add(comboRow2);
            page.add(group4);

            const switchRow3 = new Adw.SwitchRow({
                title: _('Check Command Exit Code'),
                subtitle: _('Only toggle if the command executes successfully (returns exit code 0)'),
                active: window._settings.get_boolean(`checkexitcode${pageIndex}-setting`),
            });
            group4.add(switchRow3);
            switchRow3.visible =  comboRow2.selected === 2;

            comboRow2.connect("notify::selected", () => {
                switchRow3.visible =  comboRow2.selected === 2;
                syncDisabledInfo.visible = comboRow2.selected !== 2;
                if (comboRow2.selected !== 2) {
                    commandSyncExpanderRow.show_enable_switch = false;
                    commandSyncExpanderRow.enable_expansion = false;
                    commandSyncExpanderRow.expanded = false;
                } else {
                    commandSyncExpanderRow.show_enable_switch = true;
                    commandSyncExpanderRow.enable_expansion = window._settings.get_boolean(`checkcommandsync${pageIndex}-setting`);
                    commandSyncExpanderRow.expanded = commandSyncExpanderRow.enable_expansion;
                }
            });

            const switchRow = new Adw.SwitchRow({
                title: _('Show Indicator Icon'),
                subtitle: _('Show top bar icon when toggle button is switched on'),
                active: window._settings.get_boolean(`showindicator${pageIndex}-setting`),
            });
            group4.add(switchRow);

            const switchRow2 = new Adw.SwitchRow({
                title: _('Close Menu After Button Press'),
                subtitle: _('Close the system menu immediately after clicking toggle button'),
                active: window._settings.get_boolean(`closemenu${pageIndex}-setting`),
            });
            group4.add(switchRow2);
            //#endregion Toggle Behavior


            //#region Sync Behavior
            const group5 = new Adw.PreferencesGroup({
                title: _('Command Sync Behavior'),
            });
            groups.push(group5);

            const commandSyncExpanderRow = new Adw.ExpanderRow({
                title: _('Keep Toggle State Synced'),
                subtitle: _('Keep toggle button state synced with a command\'s output'),
                show_enable_switch: true,
                expanded: window._settings.get_boolean(`checkcommandsync${pageIndex}-setting`),
                enable_expansion: window._settings.get_boolean(`checkcommandsync${pageIndex}-setting`),
            });
            if (comboRow2.selected !== 2) {
                commandSyncExpanderRow.show_enable_switch = false;
                commandSyncExpanderRow.expanded = false;
            } else {
                commandSyncExpanderRow.expanded = window._settings.get_boolean(`checkcommandsync${pageIndex}-setting`);
                commandSyncExpanderRow.show_enable_switch = true;
            }            
            group5.add(commandSyncExpanderRow);

            const checkCommandInfo2 = new Adw.ActionRow({
                title: _('Command Configuration'),
                subtitle: _(
                            'Enter the Check Output Command and Search Term in the Commands section above. If the specified Search Term appears ' +
                            'in the command\'s output, the button will be set to ON. Otherwise, the button will be set to OFF.'
                           ),
                activatable: false,
            });
            commandSyncExpanderRow.add_row(checkCommandInfo2);

            const pollingFreqSpinRow = new Adw.SpinRow({
                title: _('Polling Frequency (seconds)'),
                subtitle: _('How often to check the command output and update button state'),
                adjustment: new Gtk.Adjustment({
                    lower: 2,
                    upper: 900,
                    step_increment: 1,
                    page_increment: 10,
                }),
            });
            commandSyncExpanderRow.add_row(pollingFreqSpinRow);

            const syncDisabledInfo = new Adw.ActionRow({
                subtitle: _('Command sync is disabled. To enable, set the Button Click Action to Toggle.'),
                activatable: false,
            });
            syncDisabledInfo.visible = comboRow2.selected !== 2;
            group5.add(syncDisabledInfo);
            
            page.add(group5);
            //#endregion Sync Behavior


            //#region Shortcut
            const group6 = new Adw.PreferencesGroup({
                title: _('Keyboard Shortcut'),
            });
            page.add(group6);

            const keybindRow = new KeybindingRow(
                window._settings,
                `keybinding${pageIndex}-setting`,
                _('Assign Shortcut')
            );
            keybindRow.add_suffix(keybindRow.resetButton);
            group6.add(keybindRow);
            //#endregion Shortcut


            //#region Bindings 
            let i = pageIndex;
            if (pageIndex === 1) {i='';}

            window._settings.bind(`entryrow1${i}-setting`, entryRow1, 'text', Gio.SettingsBindFlags.DEFAULT);
            window._settings.bind(`entryrow2${i}-setting`, entryRow2, 'text', Gio.SettingsBindFlags.DEFAULT);
            window._settings.bind(`entryrow3${i}-setting`, entryRow3, 'text', Gio.SettingsBindFlags.DEFAULT);
            window._settings.bind(`entryrow4${i}-setting`, entryRow4, 'text', Gio.SettingsBindFlags.DEFAULT);
            
            window._settings.bind(`checkcommand${pageIndex}-setting`, checkCommandRow, "text", Gio.SettingsBindFlags.DEFAULT);
            window._settings.bind(`checkregex${pageIndex}-setting`, checkRegexRow, "text", Gio.SettingsBindFlags.DEFAULT);
            window._settings.bind(`initialtogglestate${pageIndex}-setting`, comboRow, 'selected', Gio.SettingsBindFlags.DEFAULT);
            window._settings.bind(`runcommandatboot${pageIndex}-setting`, expanderRow, 'expanded', Gio.SettingsBindFlags.DEFAULT);
            window._settings.bind(`delaytime${pageIndex}-setting`, spinRow, 'value', Gio.SettingsBindFlags.DEFAULT);
            window._settings.bind(`checkcommanddelaytime${pageIndex}-setting`, spinRow2, 'value', Gio.SettingsBindFlags.DEFAULT);
            window._settings.bind(`showindicator${pageIndex}-setting`, switchRow, 'active', Gio.SettingsBindFlags.DEFAULT);
            window._settings.bind(`closemenu${pageIndex}-setting`, switchRow2, 'active', Gio.SettingsBindFlags.DEFAULT);
            window._settings.bind(`checkexitcode${pageIndex}-setting`, switchRow3, 'active', Gio.SettingsBindFlags.DEFAULT);
            window._settings.bind(`buttonclick${pageIndex}-setting`, comboRow2, 'selected', Gio.SettingsBindFlags.DEFAULT);
            window._settings.bind(`checkcommandinterval${pageIndex}-setting`, pollingFreqSpinRow, 'value', Gio.SettingsBindFlags.DEFAULT);
            window._settings.bind(`checkcommandsync${pageIndex}-setting`, commandSyncExpanderRow, 'expanded', Gio.SettingsBindFlags.DEFAULT);
            //#endregion Bindings

            // Push the created page to the pages array
            pages.push(page);

        }// End of for loop to create toggle button settings pages
        

        //#region Information Page
        const infoPage = new Adw.PreferencesPage({
            title: _('Configuration'),
            icon_name: 'applications-system-symbolic',
        });
        window.add(infoPage);
        //#endregion Information Page

        
        //#region Settings
        const configGroup0 = new Adw.PreferencesGroup({
            title: _('Settings'),
        });
        infoPage.add(configGroup0);

        const spinRow0 = new Adw.SpinRow({
            title: _('Number of Toggle Buttons'),
            subtitle: _('Restart required for changes to take effect'),
            value: window._settings.get_int('numbuttons-setting'),
            adjustment: new Gtk.Adjustment({
                lower: 1,
                upper: 6,
                step_increment: 1,
                page_increment: 1,
            }),
        });
        configGroup0.add(spinRow0);
        
        window._settings.bind('numbuttons-setting', spinRow0, 'value', Gio.SettingsBindFlags.DEFAULT);

        const exportRow = new Adw.ActionRow({
            title: _('Export Button Configurations'),
            subtitle: _(`Click to export the toggles.ini backup file to the home directory`),
            activatable: true,
        });
        exportRow.connect('activated', () => {
            exportConfiguration(numButtons, window._settings, window);
        });
        configGroup0.add(exportRow);
        //#endregion Settings

        
        //#region Resources
        const configGroup1 = new Adw.PreferencesGroup({
            title: _('Resources'),
        });
        
        const configRow2 = new Adw.ActionRow({
            title: _('Icons'),
            subtitle: _(
                        '•  For a list of icons, refer to the Icon List link below.\n' +
                        '•  Enter the icon name in the Icon field, or leave blank for no icon.\n' +
                        '•  To use separate on/off icons, enter both names separated by a comma.\n' +
                        '•  To use custom icons, place them in ~/.local/share/icons/.   ' + 
                        'Then reboot and enter the icon name (without the file extension) in the Icon field.'
                       ),
            activatable: false,
        });
        
        const configRow3 = new Adw.ActionRow({
            title: _('Icon List'),
            subtitle: _('List of default symbolic icons'),
            activatable: true,
        });
        configRow3.connect('activated', () => {
            Gio.app_info_launch_default_for_uri('https://github.com/StorageB/icons/blob/main/GNOME48Adwaita/icons.md', null);
        });
        configRow3.add_prefix(new Gtk.Image({icon_name: 'web-browser-symbolic'}));
        configRow3.add_suffix(new Gtk.Image({icon_name: 'go-next-symbolic'}));
        
        const configRow4 = new Adw.ActionRow({
            title: _('Local Icons'),
            subtitle: _('Local icon directory (/usr/share/icons)'),
            activatable: true,
        });
        configRow4.connect('activated', () => {
            Gio.app_info_launch_default_for_uri('file:///usr/share/icons', null);
        });
        configRow4.add_prefix(new Gtk.Image({icon_name: 'folder-symbolic'}));
        configRow4.add_suffix(new Gtk.Image({icon_name: 'go-next-symbolic'}));
        //#endregion Resources

        
        //#region About
        const aboutGroup = new Adw.PreferencesGroup({
            title: _('About'),
        });
        
        const aboutRow0 = new Adw.ActionRow({
            title: _('What\'s New'),
            subtitle: _('List of recent changes and improvements'),
            activatable: true,
        });
        aboutRow0.connect('activated', () => {
            const dialog = new Gtk.MessageDialog({
                transient_for: window,
                modal: true,
                text: _('Release Notes'),
                secondary_text: releaseNotes,
                buttons: Gtk.ButtonsType.CLOSE,
            });
            dialog.connect('response', () => dialog.destroy());
            dialog.show();
        });
        aboutRow0.add_prefix(new Gtk.Image({icon_name: 'dialog-information-symbolic'}));
        aboutRow0.add_suffix(new Gtk.Image({icon_name: 'go-next-symbolic'}));

        const aboutRow1 = new Adw.ActionRow({
            title: _('Homepage'),
            subtitle: _('GitHub page for additional information and bug reporting'),
            activatable: true,
        });
        aboutRow1.connect('activated', () => {
            Gio.app_info_launch_default_for_uri('https://github.com/StorageB/custom-command-toggle', null);
        });
        aboutRow1.add_prefix(new Gtk.Image({icon_name: 'go-home-symbolic'}));
        aboutRow1.add_suffix(new Gtk.Image({icon_name: 'go-next-symbolic'}));
        
        const aboutRow2 = new Adw.ActionRow({
            title: _('Extension Page'),
            subtitle: _('GNOME extension page'),
            activatable: true,
        });
        aboutRow2.connect('activated', () => {
            Gio.app_info_launch_default_for_uri('https://extensions.gnome.org/extension/7012/custom-command-toggle/', null);
        });
        aboutRow2.add_prefix(new Gtk.Image({icon_name: 'web-browser-symbolic'}));
        aboutRow2.add_suffix(new Gtk.Image({icon_name: 'go-next-symbolic'}));
        
        infoPage.add(configGroup1);
        configGroup1.add(configRow2);
        configGroup1.add(configRow3);
        //configGroup1.add(configRow4);
        
        infoPage.add(aboutGroup);
        aboutGroup.add(aboutRow0);
        aboutGroup.add(aboutRow1);
        aboutGroup.add(aboutRow2);
        //#endregion About


        //#region Advanced
        const advancedGroup = new Adw.PreferencesGroup({
            title: _('Advanced'),
        });
        infoPage.add(advancedGroup);

        const debugSwitchRow = new Adw.SwitchRow({
            title: _('Detailed Logging'),
            subtitle: _(
                'To view output, run the following in a terminal then restart extension:\n' +
                'journalctl -f -o cat /usr/bin/gnome-shell | grep "Custom Command Toggle"\n\n' +
                '(May generate excessive logs. Use for setup/troubleshooting only.)'
            ),
            active: window._settings.get_boolean(`debug-setting`),
        });
        window._settings.bind(`debug-setting`, debugSwitchRow, 'active', Gio.SettingsBindFlags.DEFAULT); 

        const copyButton = new Gtk.Button({
            icon_name: 'edit-copy-symbolic',
            tooltip_text: _('Copy command to clipboard'),
            has_frame: false,
            valign: Gtk.Align.CENTER,
            halign: Gtk.Align.END,
        });

        copyButton.connect('clicked', () => {
            const value = new GObject.Value();
            value.init(GObject.TYPE_STRING);
            value.set_string('journalctl -f -o cat /usr/bin/gnome-shell | grep "Custom Command Toggle"');
            const clipboard = Gdk.Display.get_default().get_clipboard();
            const provider = Gdk.ContentProvider.new_for_value(value);
            clipboard.set_content(provider); 
            const toast = Adw.Toast.new(_(`Command copied to clipboard`));
            window.add_toast(toast);
        });

        debugSwitchRow.add_suffix(copyButton);
        advancedGroup.add(debugSwitchRow);
        //#endregion Advanced
    }
}
