// src/core/clipboard/clipboard-manager.ts
import Meta from "gi://Meta";
import Shell from "gi://Shell";
import St from "gi://St";

// src/utils/window-utils.ts
var getFocusedWindow = () => {
  const windowActors = global.get_window_actors();
  return windowActors.find((w) => w.meta_window.has_focus());
};
var getFocusedWindowApp = () => {
  const focusedWindow = getFocusedWindow();
  if (focusedWindow) {
    const wmClass = focusedWindow.meta_window.get_wm_class();
    const title = focusedWindow.meta_window.get_title();
    return wmClass || title || "unknown";
  }
  return "gnome-shell";
};

// src/utils/clipboard-utils.ts
function uint8ArrayToBase64(uint8Array) {
  try {
    const base64Chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
    let result = "";
    for (let i = 0; i < uint8Array.length; i += 3) {
      const byte1 = uint8Array[i] || 0;
      const byte2 = uint8Array[i + 1] || 0;
      const byte3 = uint8Array[i + 2] || 0;
      const chunk1 = byte1 >> 2 & 63;
      const chunk2 = (byte1 & 3) << 4 | byte2 >> 4 & 15;
      const chunk3 = (byte2 & 15) << 2 | byte3 >> 6 & 3;
      const chunk4 = byte3 & 63;
      result += base64Chars[chunk1] + base64Chars[chunk2] + base64Chars[chunk3] + base64Chars[chunk4];
    }
    const padding = 3 - uint8Array.length % 3;
    if (padding < 3) {
      result = result.slice(0, -padding) + "=".repeat(padding);
    }
    return result;
  } catch (error2) {
    error2("Error in direct base64 conversion", error2);
    throw error2;
  }
}
function uint8ArrayToBase64Fallback(uint8Array) {
  try {
    const chunkSize = 8192;
    let binaryString = "";
    for (let i = 0; i < uint8Array.length; i += chunkSize) {
      const chunk = uint8Array.slice(i, i + chunkSize);
      let chunkString = "";
      for (let j = 0; j < chunk.length; j++) {
        chunkString += String.fromCharCode(chunk[j]);
      }
      binaryString += chunkString;
    }
    const base64Result = btoa(binaryString);
    return base64Result;
  } catch (error2) {
    error2("Error in fallback base64 conversion", error2);
    throw error2;
  }
}
function bufferToBase64(buffer) {
  try {
    let uint8Array;
    if (buffer instanceof ArrayBuffer) {
      uint8Array = new Uint8Array(buffer);
    } else if (buffer instanceof Uint8Array) {
      uint8Array = buffer;
    } else if (buffer && typeof buffer === "object" && "length" in buffer && buffer.length > 0) {
      uint8Array = new Uint8Array(buffer.length);
      for (let i = 0; i < buffer.length; i++) {
        uint8Array[i] = buffer[i];
      }
    } else {
      return "";
    }
    try {
      return uint8ArrayToBase64(uint8Array);
    } catch (_directError) {
      try {
        return uint8ArrayToBase64Fallback(uint8Array);
      } catch (fallbackError) {
        if (uint8Array.length <= 1024) {
          try {
            let binaryString = "";
            for (let i = 0; i < uint8Array.length; i++) {
              binaryString += String.fromCharCode(uint8Array[i]);
            }
            return btoa(binaryString);
          } catch (_simpleError) {
          }
        }
        throw fallbackError;
      }
    }
  } catch (error2) {
    error2("Error converting buffer to base64", error2);
    return "";
  }
}
function isValidImageBuffer(buffer) {
  if (!buffer) return false;
  let uint8Array;
  if (buffer instanceof ArrayBuffer) {
    if (buffer.byteLength < 8) return false;
    uint8Array = new Uint8Array(buffer);
  } else if (buffer instanceof Uint8Array) {
    if (buffer.length < 8) return false;
    uint8Array = buffer;
  } else if ("length" in buffer && buffer.length >= 8) {
    uint8Array = new Uint8Array(buffer.slice(0, 8));
  } else {
    return false;
  }
  if (uint8Array[0] === 137 && uint8Array[1] === 80 && uint8Array[2] === 78 && uint8Array[3] === 71) {
    return true;
  }
  if (uint8Array[0] === 255 && uint8Array[1] === 216 && uint8Array[2] === 255) {
    return true;
  }
  if (uint8Array[0] === 71 && uint8Array[1] === 73 && uint8Array[2] === 70 && uint8Array[3] === 56) {
    return true;
  }
  if (uint8Array[0] === 82 && uint8Array[1] === 73 && uint8Array[2] === 70 && uint8Array[3] === 70) {
    let webpCheck;
    if (buffer instanceof Uint8Array && buffer.length >= 12) {
      webpCheck = buffer.slice(8, 12);
    } else if (buffer instanceof ArrayBuffer && buffer.byteLength >= 12) {
      webpCheck = new Uint8Array(buffer.slice(8, 12));
    } else if ("length" in buffer && buffer.length >= 12) {
      webpCheck = new Uint8Array(buffer.slice(8, 12));
    } else {
      return false;
    }
    if (webpCheck[0] === 87 && webpCheck[1] === 69 && webpCheck[2] === 66 && webpCheck[3] === 80) {
      return true;
    }
  }
  return false;
}
function getImageMimeType(buffer) {
  if (!buffer) return "application/octet-stream";
  let uint8Array;
  if (buffer instanceof ArrayBuffer) {
    if (buffer.byteLength < 8) return "application/octet-stream";
    uint8Array = new Uint8Array(buffer.slice(0, 12));
  } else if (buffer instanceof Uint8Array) {
    if (buffer.length < 8) return "application/octet-stream";
    uint8Array = buffer.slice(0, 12);
  } else if ("length" in buffer && buffer.length >= 8) {
    uint8Array = new Uint8Array(buffer.slice(0, 12));
  } else {
    return "application/octet-stream";
  }
  if (uint8Array[0] === 137 && uint8Array[1] === 80 && uint8Array[2] === 78 && uint8Array[3] === 71) {
    return "image/png";
  }
  if (uint8Array[0] === 255 && uint8Array[1] === 216 && uint8Array[2] === 255) {
    return "image/jpeg";
  }
  if (uint8Array[0] === 71 && uint8Array[1] === 73 && uint8Array[2] === 70 && uint8Array[3] === 56) {
    return "image/gif";
  }
  if (uint8Array[0] === 82 && uint8Array[1] === 73 && uint8Array[2] === 70 && uint8Array[3] === 70) {
    let webpCheck;
    if (buffer instanceof Uint8Array && buffer.length >= 12) {
      webpCheck = buffer.slice(8, 12);
    } else if (buffer instanceof ArrayBuffer && buffer.byteLength >= 12) {
      webpCheck = new Uint8Array(buffer.slice(8, 12));
    } else if ("length" in buffer && buffer.length >= 12) {
      webpCheck = new Uint8Array(buffer.slice(8, 12));
    } else {
      return "application/octet-stream";
    }
    if (webpCheck[0] === 87 && webpCheck[1] === 69 && webpCheck[2] === 66 && webpCheck[3] === 80) {
      return "image/webp";
    }
  }
  return "application/octet-stream";
}
function simpleHash(str) {
  let hash = 0;
  if (str.length === 0) return hash.toString();
  for (let i = 0; i < str.length; i++) {
    const char = str.charCodeAt(i);
    hash = (hash << 5) - hash + char;
    hash = hash & hash;
  }
  return Math.abs(hash).toString(16);
}
function calculateClipboardMetadata(event) {
  const content = event.content;
  let mimeType = "text/plain";
  let contentType = "text";
  let contentHash = "";
  let size = 0;
  const sourceApp = getFocusedWindowApp();
  if (event.source === "image") {
    contentType = "image";
    if (content.startsWith("data:image/")) {
      const match = content.match(/^data:(image\/[^;]+);/);
      mimeType = match ? match[1] : "image/png";
    }
    contentHash = simpleHash(content);
    size = content.length;
  } else if (content.startsWith("data:")) {
    const match = content.match(/^data:([^;]+);/);
    mimeType = match ? match[1] : "application/octet-stream";
    contentType = "binary";
    contentHash = simpleHash(content);
    size = content.length;
  } else {
    mimeType = "text/plain";
    contentType = "text";
    contentHash = simpleHash(content);
    size = content.length;
    if (content.includes("<") && content.includes(">") && (content.includes("<html") || content.includes("<div") || content.includes("<p"))) {
      mimeType = "text/html";
      contentType = "html";
    }
    if (content.includes("/") && (content.includes(".") || content.includes("~"))) {
      contentType = "file";
    }
  }
  return {
    mimeType,
    contentType,
    contentHash,
    size,
    sourceApp
  };
}

// src/utils/logger.ts
var PROJECT_NAME = "Vicinae";
var LogLevel = /* @__PURE__ */ ((LogLevel2) => {
  LogLevel2[LogLevel2["ERROR"] = 0] = "ERROR";
  LogLevel2[LogLevel2["WARN"] = 1] = "WARN";
  LogLevel2[LogLevel2["INFO"] = 2] = "INFO";
  LogLevel2[LogLevel2["DEBUG"] = 3] = "DEBUG";
  return LogLevel2;
})(LogLevel || {});
var currentLogLevel = 2 /* INFO */;
var log = (level, message, data) => {
  if (level > currentLogLevel) {
    return;
  }
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const levelName = LogLevel[level];
  const prefix = `[${PROJECT_NAME}] ${timestamp} ${levelName}`;
  if (data) {
    console.log(`${prefix}: ${message}`);
    if (typeof data === "object" && data !== null) {
      Object.entries(data).forEach(([key, value]) => {
        console.log(`${prefix}:   ${key}: ${value}`);
      });
    } else {
      console.log(`${prefix}: ${data}`);
    }
  } else {
    console.log(`${prefix}: ${message}`);
  }
};
var debug = (message, data) => {
  log(3 /* DEBUG */, message, data);
};
var info = (message, data) => {
  log(2 /* INFO */, message, data);
};
var warn = (message, data) => {
  log(1 /* WARN */, message, data);
};
var error = (message, error2) => {
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const prefix = `[${PROJECT_NAME}] ${timestamp} ERROR`;
  if (error2) {
    console.error(`${prefix}: ${message}`);
    console.error(`${prefix}: ${String(error2)}`);
  } else {
    console.error(`${prefix}: ${message}`);
  }
};

// src/core/clipboard/clipboard-manager.ts
var VicinaeClipboardManager = class {
  eventListeners = [];
  currentContent = "";
  clipboard = null;
  selection = null;
  _selectionOwnerChangedId = null;
  _debouncing = 0;
  settings = null;
  constructor() {
    this.setupClipboardMonitoring();
  }
  // Method to set settings from external source (like main extension)
  setSettings(settings) {
    this.settings = settings;
    info("Settings set in clipboard manager from external source");
    try {
      const blockedApps = this.settings.get_strv("blocked-applications");
      debug(`Current blocked applications: [${blockedApps.join(", ")}]`);
    } catch (err) {
      error("Error reading blocked applications from settings", err);
    }
  }
  // Method to update settings when they change
  updateSettings(settings) {
    this.settings = settings;
    info("Settings updated in clipboard manager");
    try {
      const blockedApps = this.settings.get_strv("blocked-applications");
      debug(`Updated blocked applications: [${blockedApps.join(", ")}]`);
    } catch (err) {
      error(
        "Error reading updated blocked applications from settings",
        err
      );
    }
  }
  isApplicationBlocked(sourceApp) {
    if (!this.settings) {
      warn(
        "No settings available in clipboard manager - blocking logic disabled"
      );
      return false;
    }
    try {
      const blockedApps = this.settings.get_strv("blocked-applications");
      debug(
        `Checking if ${sourceApp} is blocked. Blocked apps list: [${blockedApps.join(", ")}]`
      );
      const isBlocked = blockedApps.some(
        (blockedApp) => sourceApp.toLowerCase().includes(blockedApp.toLowerCase()) || blockedApp.toLowerCase().includes(sourceApp.toLowerCase())
      );
      if (isBlocked) {
        debug(
          `Application ${sourceApp} is blocked from clipboard access`
        );
      } else {
        debug(
          `Application ${sourceApp} is NOT blocked (not in blocked apps list)`
        );
      }
      return isBlocked;
    } catch (error2) {
      error(
        "Error checking blocked applications in clipboard manager",
        error2
      );
      return false;
    }
  }
  shouldBlockContentType(contentType, mimeType) {
    return contentType === "text" || mimeType.startsWith("text/");
  }
  setupClipboardMonitoring() {
    try {
      this.clipboard = St.Clipboard.get_default();
      this.selection = Shell.Global.get().get_display().get_selection();
      if (this.selection) {
        this._selectionOwnerChangedId = this.selection.connect(
          "owner-changed",
          this.onSelectionOwnerChanged.bind(this)
        );
        this.queryClipboard();
        info(
          "Clipboard monitoring set up successfully using selection listener"
        );
      } else {
        error("Failed to get selection instance");
      }
    } catch (error2) {
      error("Error setting up clipboard monitoring", error2);
    }
  }
  onSelectionOwnerChanged(_, selectionType) {
    if (selectionType === Meta.SelectionType.SELECTION_CLIPBOARD) {
      this.queryClipboard();
    }
  }
  queryClipboard() {
    if (!this.clipboard) return;
    try {
      this.clipboard.get_text(St.ClipboardType.CLIPBOARD, (_, text) => {
        if (text) {
          this.processClipboardContent(text, "system");
        }
      });
      this.clipboard.get_text(St.ClipboardType.PRIMARY, (_, text) => {
        if (text) {
          this.processClipboardContent(text, "system");
        }
      });
      const mimeTypes = this.clipboard.get_mimetypes(
        St.ClipboardType.CLIPBOARD
      );
      if (mimeTypes.length > 0) {
        if (mimeTypes.some((type) => type.startsWith("image/"))) {
          this.captureImageData();
        }
      }
    } catch (error2) {
      error("Error querying clipboard", error2);
    }
  }
  captureImageData() {
    if (!this.clipboard) return;
    try {
      try {
        this.clipboard.get_content(
          St.ClipboardType.CLIPBOARD,
          "image/png",
          // MIME type we want
          (_, content) => {
            if (content) {
              if (content && typeof content === "object") {
                if (content.constructor && (content.constructor.name === "GLib.Bytes" || content.constructor.name.includes(
                  "GLib.Bytes"
                ) || content.constructor.name.includes(
                  "Bytes"
                ))) {
                  this.extractGLibBytesData(
                    content
                  );
                } else {
                  if (content && typeof content === "object" && "data" in content) {
                    this.processImageContent(
                      content
                    );
                  } else {
                    this.processClipboardContent(
                      "[IMAGE_DATA_AVAILABLE]",
                      "system"
                    );
                  }
                }
              }
            }
          }
        );
      } catch (_contentError) {
        this.processClipboardContent(
          "[IMAGE_DATA_AVAILABLE]",
          "system"
        );
      }
    } catch (error2) {
      error("Error capturing image data", error2);
    }
  }
  processImageContent(content) {
    try {
      if (content.data && content.data.length > 0) {
        const isValid = isValidImageBuffer(content.data);
        const detectedMimeType = getImageMimeType(content.data);
        const finalMimeType = content.mimeType || detectedMimeType;
        if (isValid) {
          const base64Data = bufferToBase64(content.data);
          const imageContent = `data:${finalMimeType};base64,${base64Data}`;
          this.processClipboardContent(imageContent, "image");
        } else {
          this.processClipboardContent(
            "[BINARY_DATA_AVAILABLE]",
            "system"
          );
        }
      } else {
        this.processClipboardContent(
          "[IMAGE_DATA_AVAILABLE]",
          "system"
        );
      }
    } catch (error2) {
      error("Error processing image content", error2);
      this.processClipboardContent("[IMAGE_DATA_AVAILABLE]", "system");
    }
  }
  extractGLibBytesData(bytes) {
    try {
      if (typeof bytes.get_data === "function") {
        try {
          const data = bytes.get_data();
          if (data && data.length > 0) {
            const isValid = isValidImageBuffer(data);
            const mimeType = getImageMimeType(data);
            if (isValid) {
              const base64Data = bufferToBase64(data);
              const imageContent = `data:${mimeType};base64,${base64Data}`;
              this.processClipboardContent(imageContent, "image");
            } else {
              this.processClipboardContent(
                "[BINARY_DATA_AVAILABLE]",
                "system"
              );
            }
          }
        } catch (getDataError) {
          error("get_data failed", getDataError);
        }
      }
      if (typeof bytes.toArray === "function") {
        try {
          const array = bytes.toArray();
          if (array && array.length > 0) {
            const isValid = isValidImageBuffer(array);
            const mimeType = getImageMimeType(array);
            if (isValid) {
              const base64Data = bufferToBase64(array);
              const imageContent = `data:${mimeType};base64,${base64Data}`;
              this.processClipboardContent(imageContent, "image");
            }
          }
        } catch (toArrayError) {
          error("to_array failed", toArrayError);
        }
      }
    } catch (error2) {
      error("Error extracting GLib.Bytes data", error2);
    }
  }
  processClipboardContent(text, source) {
    if (this._debouncing > 0) {
      this._debouncing--;
      return;
    }
    if (!text || text === this.currentContent) {
      return;
    }
    this.currentContent = text;
    this.emitClipboardEvent(text, source);
  }
  // Method to emit clipboard change events
  emitClipboardEvent(content, source = "user") {
    const event = {
      type: "clipboard-changed",
      content,
      timestamp: Date.now(),
      source,
      contentType: source === "image" ? "image" : "text"
    };
    const metadata = calculateClipboardMetadata(event);
    const isBlocked = this.isApplicationBlocked(metadata.sourceApp);
    const shouldBlock = isBlocked && this.shouldBlockContentType(
      metadata.contentType,
      metadata.mimeType
    );
    debug("\u{1F3AF} CLIPBOARD EVENT EMITTED", {
      type: event.type,
      content: content.length > 100 ? `${content.substring(0, 100)}...` : content,
      contentLength: content.length,
      timestamp: new Date(event.timestamp).toISOString(),
      source: event.source,
      listeners: this.eventListeners.length,
      // Enhanced metadata fields (same as DBUS emission)
      mimeType: metadata.mimeType,
      contentType: metadata.contentType,
      contentHash: metadata.contentHash,
      size: metadata.size,
      sourceApp: metadata.sourceApp,
      // Blocking status for user visibility
      isBlocked,
      shouldBlock,
      note: shouldBlock ? "\u26A0\uFE0F This event will be blocked by clipboard manager" : "\u2705 Event will be processed normally"
    });
    if (shouldBlock) {
      debug(
        `\u{1F6AB} Clipboard access blocked for application: ${metadata.sourceApp} (${metadata.contentType}) - Event not forwarded to listeners`
      );
      return;
    }
    this.eventListeners.forEach((listener) => {
      try {
        listener(event);
      } catch (error2) {
        error("\u274C Error in clipboard event listener", error2);
      }
    });
  }
  // Method for external components to listen to clipboard events
  onClipboardChange(listener) {
    this.eventListeners.push(listener);
    debug("\u{1F442} Clipboard change listener added");
  }
  // Method to remove a listener
  removeClipboardListener(listener) {
    const index = this.eventListeners.indexOf(listener);
    if (index > -1) {
      this.eventListeners.splice(index, 1);
      debug("Clipboard change listener removed");
    }
  }
  // Method to get current clipboard content
  getCurrentContent() {
    return this.currentContent;
  }
  // Method to set clipboard content
  setContent(content) {
    if (this.clipboard) {
      try {
        this.clipboard.set_text(St.ClipboardType.CLIPBOARD, content);
        this.clipboard.set_text(St.ClipboardType.PRIMARY, content);
        this.currentContent = content;
        this.emitClipboardEvent(content, "user");
      } catch (error2) {
        error("Error setting clipboard content", error2);
      }
    }
  }
  // Method to manually trigger clipboard change (for testing or external triggers)
  triggerClipboardChange(content, source = "user") {
    this.emitClipboardEvent(content, source);
  }
  // Cleanup method
  destroy() {
    if (this.selection && this._selectionOwnerChangedId) {
      try {
        this.selection.disconnect(this._selectionOwnerChangedId);
        this._selectionOwnerChangedId = null;
      } catch (edit_error) {
        error("Error disconnecting selection listener", edit_error);
      }
    }
    this.eventListeners = [];
    this.currentContent = "";
    this.clipboard = null;
    this.selection = null;
    info("Clipboard manager destroyed");
  }
};
export {
  VicinaeClipboardManager
};
