// src/core/dbus/services/clipboard-service.ts
import GLib from "gi://GLib";

// src/utils/window-utils.ts
var getFocusedWindow = () => {
  const windowActors = global.get_window_actors();
  return windowActors.find((w) => w.meta_window.has_focus());
};
var getFocusedWindowApp = () => {
  const focusedWindow = getFocusedWindow();
  if (focusedWindow) {
    const wmClass = focusedWindow.meta_window.get_wm_class();
    const title = focusedWindow.meta_window.get_title();
    return wmClass || title || "unknown";
  }
  return "gnome-shell";
};

// src/utils/clipboard-utils.ts
function simpleHash(str) {
  let hash = 0;
  if (str.length === 0) return hash.toString();
  for (let i = 0; i < str.length; i++) {
    const char = str.charCodeAt(i);
    hash = (hash << 5) - hash + char;
    hash = hash & hash;
  }
  return Math.abs(hash).toString(16);
}
function calculateClipboardMetadata(event) {
  const content = event.content;
  let mimeType = "text/plain";
  let contentType = "text";
  let contentHash = "";
  let size = 0;
  const sourceApp = getFocusedWindowApp();
  if (event.source === "image") {
    contentType = "image";
    if (content.startsWith("data:image/")) {
      const match = content.match(/^data:(image\/[^;]+);/);
      mimeType = match ? match[1] : "image/png";
    }
    contentHash = simpleHash(content);
    size = content.length;
  } else if (content.startsWith("data:")) {
    const match = content.match(/^data:([^;]+);/);
    mimeType = match ? match[1] : "application/octet-stream";
    contentType = "binary";
    contentHash = simpleHash(content);
    size = content.length;
  } else {
    mimeType = "text/plain";
    contentType = "text";
    contentHash = simpleHash(content);
    size = content.length;
    if (content.includes("<") && content.includes(">") && (content.includes("<html") || content.includes("<div") || content.includes("<p"))) {
      mimeType = "text/html";
      contentType = "html";
    }
    if (content.includes("/") && (content.includes(".") || content.includes("~"))) {
      contentType = "file";
    }
  }
  return {
    mimeType,
    contentType,
    contentHash,
    size,
    sourceApp
  };
}

// src/utils/logger.ts
var PROJECT_NAME = "Vicinae";
var LogLevel = /* @__PURE__ */ ((LogLevel2) => {
  LogLevel2[LogLevel2["ERROR"] = 0] = "ERROR";
  LogLevel2[LogLevel2["WARN"] = 1] = "WARN";
  LogLevel2[LogLevel2["INFO"] = 2] = "INFO";
  LogLevel2[LogLevel2["DEBUG"] = 3] = "DEBUG";
  return LogLevel2;
})(LogLevel || {});
var currentLogLevel = 2 /* INFO */;
var log = (level, message, data) => {
  if (level > currentLogLevel) {
    return;
  }
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const levelName = LogLevel[level];
  const prefix = `[${PROJECT_NAME}] ${timestamp} ${levelName}`;
  if (data) {
    console.log(`${prefix}: ${message}`);
    if (typeof data === "object" && data !== null) {
      Object.entries(data).forEach(([key, value]) => {
        console.log(`${prefix}:   ${key}: ${value}`);
      });
    } else {
      console.log(`${prefix}: ${data}`);
    }
  } else {
    console.log(`${prefix}: ${message}`);
  }
};
var debug = (message, data) => {
  log(3 /* DEBUG */, message, data);
};
var info = (message, data) => {
  log(2 /* INFO */, message, data);
};
var warn = (message, data) => {
  log(1 /* WARN */, message, data);
};
var error = (message, error2) => {
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const prefix = `[${PROJECT_NAME}] ${timestamp} ERROR`;
  if (error2) {
    console.error(`${prefix}: ${message}`);
    console.error(`${prefix}: ${String(error2)}`);
  } else {
    console.error(`${prefix}: ${message}`);
  }
};

// src/core/dbus/services/clipboard-service.ts
var ClipboardService = class {
  clipboardManager;
  dbusObject = null;
  clipboardListener = null;
  isListening = false;
  constructor(clipboardManager, _extension) {
    this.clipboardManager = clipboardManager;
    info("ClipboardService initialized with shared clipboard manager");
  }
  // Method to set the D-Bus exported object (called by DBusManager)
  setDBusObject(dbusObject) {
    this.dbusObject = dbusObject;
    this.clipboardListener = (event) => {
      try {
        if (!event.content || event.content.length === 0) {
          debug("Skipping empty clipboard content");
          return;
        }
        const metadata = calculateClipboardMetadata(event);
        debug(
          `Emitting D-Bus signal for ${metadata.sourceApp} with data:`,
          {
            content: `${event.content.substring(0, 50)}...`,
            timestamp: event.timestamp,
            source: event.source,
            mimeType: metadata.mimeType,
            contentType: metadata.contentType,
            contentHash: metadata.contentHash,
            size: metadata.size,
            sourceApp: metadata.sourceApp
          }
        );
        this.dbusObject?.emit_signal(
          "ClipboardChanged",
          GLib.Variant.new("(stssssts)", [
            String(event.content),
            // string
            Number(event.timestamp),
            // uint64 (timestamp can be large)
            String(event.source),
            // string
            String(metadata.mimeType),
            // string
            String(metadata.contentType),
            // string
            String(metadata.contentHash),
            // string
            Number(metadata.size),
            // uint64
            String(metadata.sourceApp)
            // string
          ])
        );
        debug(`D-Bus signal emitted for ${metadata.sourceApp}`);
      } catch (signalError) {
        error("Error emitting D-Bus clipboard signal", {
          error: signalError,
          errorType: typeof signalError,
          errorMessage: signalError instanceof Error ? signalError.message : String(signalError),
          stack: signalError instanceof Error ? signalError.stack : void 0,
          data: {
            content: event.content.substring(0, 50),
            timestamp: event.timestamp,
            source: event.source,
            mimeType: "unknown",
            contentType: "unknown",
            contentHash: "unknown",
            size: 0,
            sourceApp: "unknown"
          }
        });
      }
    };
    this.clipboardManager.onClipboardChange(this.clipboardListener);
    this.isListening = true;
    info("\u{1F4E1} D-Bus clipboard listener activated");
  }
  // Method to start listening to clipboard changes (no-op since we start automatically)
  ListenToClipboardChanges() {
    try {
      if (!this.isListening && this.clipboardListener) {
        debug("D-Bus: Starting clipboard listener...");
        this.clipboardManager.onClipboardChange(this.clipboardListener);
        this.isListening = true;
        info(
          "\u{1F4E1} D-Bus clipboard listener activated via ListenToClipboardChanges"
        );
      } else if (this.isListening) {
        debug("D-Bus: Clipboard listener already active");
      } else {
        warn("D-Bus: No clipboard listener available");
      }
    } catch (error2) {
      error("D-Bus: Error starting clipboard listener", error2);
      throw error2;
    }
  }
  // Method to manually trigger a clipboard change (for testing)
  TriggerClipboardChange(content, source = "user") {
    try {
      this.clipboardManager.triggerClipboardChange(
        content,
        source
      );
    } catch (error2) {
      error("D-Bus: Error triggering clipboard change", error2);
      throw error2;
    }
  }
  // Method to get current clipboard content (if needed)
  GetCurrentContent() {
    try {
      const content = this.clipboardManager.getCurrentContent();
      return content;
    } catch (error2) {
      error("D-Bus: Error getting current clipboard content", error2);
      throw error2;
    }
  }
  // Method to set clipboard content
  SetContent(content) {
    try {
      this.clipboardManager.setContent(content);
    } catch (error2) {
      error("D-Bus: Error setting clipboard content", error2);
      throw error2;
    }
  }
  // Method to get available MIME types from current clipboard
  GetClipboardMimeTypes() {
    try {
      return [
        "text/plain",
        "text/html",
        "image/png",
        "image/jpeg",
        "image/gif",
        "image/webp",
        "application/x-vicinae-concealed"
      ];
    } catch (error2) {
      error("D-Bus: Error getting clipboard MIME types", error2);
      return [];
    }
  }
  // Method to stop listening to clipboard changes
  StopListening() {
    try {
      if (this.clipboardListener && this.isListening) {
        this.clipboardManager.removeClipboardListener(
          this.clipboardListener
        );
        this.clipboardListener = null;
        this.isListening = false;
        info("\u{1F515} D-Bus clipboard listener deactivated");
      }
    } catch (error2) {
      error("Error stopping clipboard listener", error2);
    }
  }
  // Cleanup method
  destroy() {
    this.StopListening();
    this.dbusObject = null;
  }
};
export {
  ClipboardService
};
