// src/core/dbus/services/windows-service.ts
import GLib from "gi://GLib";

// src/utils/logger.ts
var PROJECT_NAME = "Vicinae";
var LogLevel = /* @__PURE__ */ ((LogLevel2) => {
  LogLevel2[LogLevel2["ERROR"] = 0] = "ERROR";
  LogLevel2[LogLevel2["WARN"] = 1] = "WARN";
  LogLevel2[LogLevel2["INFO"] = 2] = "INFO";
  LogLevel2[LogLevel2["DEBUG"] = 3] = "DEBUG";
  return LogLevel2;
})(LogLevel || {});
var currentLogLevel = 2 /* INFO */;
var log = (level, message, data) => {
  if (level > currentLogLevel) {
    return;
  }
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const levelName = LogLevel[level];
  const prefix = `[${PROJECT_NAME}] ${timestamp} ${levelName}`;
  if (data) {
    console.log(`${prefix}: ${message}`);
    if (typeof data === "object" && data !== null) {
      Object.entries(data).forEach(([key, value]) => {
        console.log(`${prefix}:   ${key}: ${value}`);
      });
    } else {
      console.log(`${prefix}: ${data}`);
    }
  } else {
    console.log(`${prefix}: ${message}`);
  }
};
var debug = (message, data) => {
  log(3 /* DEBUG */, message, data);
};
var error = (message, error2) => {
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const prefix = `[${PROJECT_NAME}] ${timestamp} ERROR`;
  if (error2) {
    console.error(`${prefix}: ${message}`);
    console.error(`${prefix}: ${String(error2)}`);
  } else {
    console.error(`${prefix}: ${message}`);
  }
};

// src/core/windows/window-manager.ts
import Meta from "gi://Meta";

// src/utils/window-utils.ts
var getWindowById = (winid) => {
  if (!winid || winid <= 0) return null;
  try {
    const windowActors = global.get_window_actors();
    return windowActors.find((w) => {
      try {
        return w.meta_window && w.meta_window.get_id() === winid;
      } catch {
        return false;
      }
    });
  } catch (_error) {
    return null;
  }
};
var getCurrentTime = () => {
  return global.get_current_time();
};

// src/core/windows/workspace-manager.ts
var WorkspaceManager = class {
  getWorkspaceCount() {
    try {
      const workspaceManager = global.workspace_manager;
      return workspaceManager.get_n_workspaces();
    } catch (error2) {
      error("Error getting workspace count", error2);
      return 0;
    }
  }
  getCurrentWorkspaceIndex() {
    try {
      const workspaceManager = global.workspace_manager;
      const currentWorkspace = workspaceManager.get_active_workspace();
      return currentWorkspace.index();
    } catch (error2) {
      error("Error getting current workspace index", error2);
      return 0;
    }
  }
  getWorkspaceByIndex(index) {
    try {
      const workspaceManager = global.workspace_manager;
      return workspaceManager.get_workspace_by_index(index);
    } catch (error2) {
      error("Error getting workspace by index", error2);
      return null;
    }
  }
  switchToWorkspace(index) {
    try {
      const workspace = this.getWorkspaceByIndex(index);
      if (workspace) {
        workspace.activate(global.get_current_time());
      }
    } catch (error2) {
      error("Error switching to workspace", error2);
      throw error2;
    }
  }
  getWorkspaceInfo(index) {
    try {
      const workspace = this.getWorkspaceByIndex(index);
      if (workspace) {
        const windows = workspace.list_windows();
        const hasFullscreen = windows.some(
          (win) => win.get_maximized() === 3
        );
        let monitor = 0;
        if (windows.length > 0) {
          monitor = windows[0].get_monitor();
        }
        return {
          index: workspace.index(),
          name: `Workspace ${workspace.index() + 1}`,
          isActive: workspace === global.workspace_manager.get_active_workspace(),
          windowCount: windows.length,
          monitor,
          hasfullscreen: hasFullscreen
        };
      }
      return null;
    } catch (error2) {
      error("Error getting workspace info", error2);
      return null;
    }
  }
  getAllWorkspaces() {
    try {
      const workspaceManager = global.workspace_manager;
      const workspaces = [];
      for (let i = 0; i < workspaceManager.get_n_workspaces(); i++) {
        const workspace = workspaceManager.get_workspace_by_index(i);
        if (workspace) {
          const workspaceInfo = this.getWorkspaceInfo(i);
          if (workspaceInfo) {
            workspaces.push(workspaceInfo);
          }
        }
      }
      return workspaces;
    } catch (error2) {
      error("Error getting all workspaces", error2);
      return [];
    }
  }
};

// src/core/windows/window-manager.ts
var VicinaeWindowManager = class {
  list() {
    const windowActors = global.get_window_actors();
    const workspaceManager = global.workspace_manager;
    const windows = windowActors.map((w) => {
      const metaWindow = w.meta_window;
      const windowWorkspace = metaWindow.get_workspace();
      const frame = metaWindow.get_frame_rect();
      return {
        id: metaWindow.get_id(),
        title: metaWindow.get_title(),
        wm_class: metaWindow.get_wm_class(),
        wm_class_instance: metaWindow.get_wm_class_instance(),
        pid: metaWindow.get_pid(),
        maximized: metaWindow.get_maximized() !== 0,
        // 0 means not maximized
        display: metaWindow.get_display(),
        frame_type: metaWindow.get_frame_type(),
        window_type: metaWindow.get_window_type(),
        layer: metaWindow.get_layer(),
        monitor: metaWindow.get_monitor(),
        role: metaWindow.get_role(),
        width: frame.width,
        height: frame.height,
        x: frame.x,
        y: frame.y,
        in_current_workspace: metaWindow.located_on_workspace?.(
          workspaceManager.get_active_workspace?.()
        ),
        canclose: metaWindow.can_close(),
        canmaximize: metaWindow.can_maximize(),
        canminimize: metaWindow.can_minimize(),
        canshade: false,
        // can_shade() is not in the type definitions
        moveable: metaWindow.allows_move(),
        resizeable: metaWindow.allows_resize(),
        has_focus: metaWindow.has_focus(),
        workspace: windowWorkspace ? windowWorkspace.index() : -1
      };
    });
    return windows;
  }
  details(winid) {
    const w = getWindowById(winid);
    if (!w) {
      throw new Error("Window not found");
    }
    const metaWindow = w.meta_window;
    const workspaceManager = global.workspace_manager;
    const windowWorkspace = metaWindow.get_workspace();
    const frame = metaWindow.get_frame_rect();
    const win = {
      id: metaWindow.get_id(),
      title: metaWindow.get_title(),
      wm_class: metaWindow.get_wm_class(),
      wm_class_instance: metaWindow.get_wm_class_instance(),
      pid: metaWindow.get_pid(),
      maximized: metaWindow.get_maximized() !== 0,
      // 0 means not maximized
      display: metaWindow.get_display(),
      frame_type: metaWindow.get_frame_type(),
      window_type: metaWindow.get_window_type(),
      layer: metaWindow.get_layer(),
      monitor: metaWindow.get_monitor(),
      role: metaWindow.get_role(),
      width: frame.width,
      height: frame.height,
      x: frame.x,
      y: frame.y,
      in_current_workspace: metaWindow.located_on_workspace?.(
        workspaceManager.get_active_workspace?.()
      ),
      canclose: metaWindow.can_close(),
      canmaximize: metaWindow.can_maximize(),
      canminimize: metaWindow.can_minimize(),
      canshade: false,
      // can_shade() is not in the type definitions
      moveable: metaWindow.allows_move(),
      resizeable: metaWindow.allows_resize(),
      has_focus: metaWindow.has_focus(),
      workspace: windowWorkspace ? windowWorkspace.index() : -1
    };
    return win;
  }
  getTitle(winid) {
    const w = getWindowById(winid);
    if (w) {
      return w.meta_window.get_title();
    } else {
      throw new Error("Window not found");
    }
  }
  getFrameRect(winid) {
    const w = getWindowById(winid);
    if (w) {
      const frame = w.meta_window.get_frame_rect();
      return {
        x: frame.x,
        y: frame.y,
        width: frame.width,
        height: frame.height
      };
    } else {
      throw new Error("Window not found");
    }
  }
  getFrameBounds(winid) {
    const w = getWindowById(winid);
    if (w) {
      return {
        frame_bounds: w.meta_window.get_frame_bounds()
      };
    } else {
      throw new Error("Window not found");
    }
  }
  moveToWorkspace(winid, workspaceNum) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.change_workspace_by_index(workspaceNum, false);
    } else {
      throw new Error("Window not found");
    }
  }
  moveResize(winid, x, y, width, height) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta.MaximizeFlags.BOTH);
      }
      win.meta_window.move_resize_frame(true, x, y, width, height);
    } else {
      throw new Error("Window not found");
    }
  }
  resize(winid, width, height) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta.MaximizeFlags.BOTH);
      }
      win.meta_window.move_resize_frame(
        true,
        win.get_x(),
        win.get_y(),
        width,
        height
      );
    } else {
      throw new Error("Window not found");
    }
  }
  move(winid, x, y) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta.MaximizeFlags.BOTH);
      }
      win.meta_window.move_frame(true, x, y);
    } else {
      throw new Error("Window not found");
    }
  }
  maximize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.maximize(Meta.MaximizeFlags.BOTH);
    } else {
      throw new Error("Window not found");
    }
  }
  minimize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.minimize();
    } else {
      throw new Error("Window not found");
    }
  }
  unmaximize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.unmaximize(Meta.MaximizeFlags.BOTH);
    } else {
      throw new Error("Window not found");
    }
  }
  unminimize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.unminimize();
    } else {
      throw new Error("Window not found");
    }
  }
  activate(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      const currentTime = getCurrentTime();
      const workspace = win.get_workspace();
      if (workspace) {
        workspace.activate_with_focus(win, currentTime);
      } else {
        win.activate(currentTime);
      }
    } else {
      throw new Error("Window not found");
    }
  }
  close(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      try {
        if (win.get_id() === winid) {
          win.delete(getCurrentTime());
        } else {
          throw new Error(
            "Window ID mismatch - window may be destroyed"
          );
        }
      } catch (error2) {
        throw new Error(`Failed to close window ${winid}: ${error2}`);
      }
    } else {
      throw new Error("Window not found");
    }
  }
  listWorkspaces() {
    const workspaceManager = new WorkspaceManager();
    return workspaceManager.getAllWorkspaces();
  }
  getActiveWorkspace() {
    const workspaceManager = new WorkspaceManager();
    const currentIndex = workspaceManager.getCurrentWorkspaceIndex();
    const workspace = workspaceManager.getWorkspaceInfo(currentIndex);
    if (!workspace) {
      throw new Error("No active workspace found");
    }
    return workspace;
  }
};

// src/core/dbus/services/windows-service.ts
var WindowsService = class {
  windowManager;
  dbusObject = null;
  // Signal connection IDs for cleanup
  windowOpenedSignalId = 0;
  windowFocusSignalId = 0;
  workspaceChangedSignalId = 0;
  // Track individual window signal IDs
  windowDestroySignalIds = /* @__PURE__ */ new Map();
  windowSizeSignalIds = /* @__PURE__ */ new Map();
  constructor() {
    this.windowManager = new VicinaeWindowManager();
  }
  // Method to set the D-Bus exported object (called by DBusManager)
  setDBusObject(dbusObject) {
    this.dbusObject = dbusObject;
    debug("WindowsService: D-Bus object set for signal emission");
    this.setupWindowEventListeners();
  }
  setupWindowEventListeners() {
    debug("WindowsService: Setting up GNOME window event listeners");
    const _display = global.display;
    this.windowOpenedSignalId = global.display.connect(
      "window-created",
      (_display2, window) => {
        try {
          const windowInfo = this.getWindowInfo(window);
          this.emitOpenWindow(
            windowInfo.id.toString(),
            windowInfo.workspace.toString(),
            windowInfo.wm_class,
            windowInfo.title
          );
          this.connectToWindowDestroy(window);
          this.connectToWindowSizeChanges(window);
        } catch (error2) {
          debug(`Error handling window opened event: ${error2}`);
        }
      }
    );
    this.connectToExistingWindows();
    this.windowFocusSignalId = global.display.connect(
      "notify::focus-window",
      () => {
        try {
          GLib.idle_add(GLib.PRIORITY_DEFAULT_IDLE, () => {
            const focusWindow = global.display.focus_window;
            if (focusWindow) {
              this.emitFocusWindow(
                focusWindow.get_id().toString()
              );
            }
            return GLib.SOURCE_REMOVE;
          });
        } catch (error2) {
          debug(`Error handling window focus event: ${error2}`);
        }
      }
    );
    this.workspaceChangedSignalId = global.workspace_manager?.connect(
      "notify::active-workspace",
      () => {
        try {
          const activeWorkspace = global.workspace_manager?.get_active_workspace();
          if (activeWorkspace) {
            this.emitWorkspaceChanged(
              activeWorkspace.index().toString()
            );
          }
        } catch (error2) {
          debug(`Error handling workspace changed event: ${error2}`);
        }
      }
    );
    debug(
      "WindowsService: GNOME window event listeners set up successfully"
    );
  }
  connectToWindowDestroy(window) {
    const windowId = window.get_id();
    try {
      let signalId;
      let connectedSignal = "";
      try {
        debug(
          `Attempting to connect 'destroy' signal for window ${windowId}`
        );
        signalId = window.connect("destroy", () => {
          try {
            debug(
              `Window ${windowId} destroy signal triggered - emitting closewindow`
            );
            this.emitCloseWindow(windowId.toString());
            this.windowDestroySignalIds.delete(windowId);
          } catch (error2) {
            debug(
              `Error emitting closewindow for ${windowId}: ${error2}`
            );
          }
        });
        connectedSignal = "destroy";
        debug(
          `Successfully connected to 'destroy' signal for window ${windowId}`
        );
      } catch (_destroyError) {
        debug(
          `'destroy' signal not available for window ${windowId}, trying 'unmanaged'`
        );
        try {
          signalId = window.connect("unmanaged", () => {
            try {
              debug(
                `Window ${windowId} unmanaged signal triggered - emitting closewindow`
              );
              this.emitCloseWindow(windowId.toString());
              this.windowDestroySignalIds.delete(windowId);
            } catch (error2) {
              debug(
                `Error emitting closewindow for ${windowId}: ${error2}`
              );
            }
          });
          connectedSignal = "unmanaged";
          debug(
            `Successfully connected to 'unmanaged' signal for window ${windowId}`
          );
        } catch (_unmanagedError) {
          debug(
            `No suitable destroy signal for window ${windowId}, skipping signal connection`
          );
          return;
        }
      }
      if (signalId !== void 0) {
        this.windowDestroySignalIds.set(windowId, signalId);
        debug(
          `Successfully connected ${connectedSignal} signal for window ${windowId} (signal ID: ${signalId})`
        );
      }
    } catch (error2) {
      debug(
        `Failed to connect any destroy signal for window ${windowId}: ${error2}`
      );
    }
  }
  connectToWindowSizeChanges(window) {
    const windowId = window.get_id();
    try {
      const signalId = window.connect("size-changed", () => {
        try {
          const windowInfo = this.getWindowInfo(window);
          debug(
            `Window ${windowId} size changed - emitting movewindow`
          );
          this.emitMoveWindow(
            windowInfo.id.toString(),
            windowInfo.x,
            windowInfo.y,
            windowInfo.width,
            windowInfo.height
          );
        } catch (error2) {
          debug(
            `Error handling size change for window ${windowId}: ${error2}`
          );
        }
      });
      this.windowSizeSignalIds.set(windowId, signalId);
      debug(`Connected size-changed signal for window ${windowId}`);
    } catch (error2) {
      debug(
        `Failed to connect size-changed signal for window ${windowId}: ${error2}`
      );
    }
  }
  connectToExistingWindows() {
    try {
      const windowActors = global.get_window_actors();
      debug(
        `WindowsService: Connecting to ${windowActors.length} existing windows`
      );
      for (const actor of windowActors) {
        if (actor.meta_window) {
          this.connectToWindowDestroy(actor.meta_window);
          this.connectToWindowSizeChanges(actor.meta_window);
        }
      }
    } catch (error2) {
      debug(`Error connecting to existing windows: ${error2}`);
    }
  }
  getWindowInfo(window) {
    let x = 0, y = 0, width = 0, height = 0;
    try {
      if (typeof window.get_frame_rect === "function") {
        const frame = window.get_frame_rect();
        x = frame.x;
        y = frame.y;
        width = frame.width;
        height = frame.height;
      } else {
        debug(
          `Window ${window.get_id()} does not have get_frame_rect method`
        );
      }
    } catch (error2) {
      debug(
        `Error getting frame rect for window ${window.get_id()}: ${error2}`
      );
    }
    const workspace = window.get_workspace();
    return {
      id: window.get_id(),
      title: window.get_title(),
      wm_class: window.get_wm_class() || "",
      workspace: workspace ? workspace.index() : -1,
      x,
      y,
      width,
      height
    };
  }
  // Cleanup method to disconnect signals
  destroy() {
    debug("WindowsService: Cleaning up window event listeners");
    if (this.windowOpenedSignalId) {
      global.display.disconnect(this.windowOpenedSignalId);
    }
    if (this.windowFocusSignalId) {
      global.display.disconnect(this.windowFocusSignalId);
    }
    if (this.workspaceChangedSignalId && global.workspace_manager) {
      global.workspace_manager.disconnect(this.workspaceChangedSignalId);
    }
    const allWindowIds = /* @__PURE__ */ new Set([
      ...this.windowDestroySignalIds.keys(),
      ...this.windowSizeSignalIds.keys()
    ]);
    for (const windowId of allWindowIds) {
      try {
        const windowActors = global.get_window_actors();
        const windowActor = windowActors.find(
          (actor) => actor.meta_window?.get_id() === windowId
        );
        if (windowActor?.meta_window) {
          const destroySignalId = this.windowDestroySignalIds.get(windowId);
          if (destroySignalId) {
            windowActor.meta_window.disconnect(destroySignalId);
          }
          const sizeSignalId = this.windowSizeSignalIds.get(windowId);
          if (sizeSignalId) {
            windowActor.meta_window.disconnect(sizeSignalId);
          }
        }
      } catch (error2) {
        debug(
          `Error disconnecting signals for window ${windowId}: ${error2}`
        );
      }
    }
    this.windowDestroySignalIds.clear();
    this.windowSizeSignalIds.clear();
    debug("WindowsService: Window event listeners cleaned up");
  }
  List() {
    try {
      GLib.usleep(1e3);
      const windows = this.windowManager.list();
      return JSON.stringify(windows);
    } catch (error2) {
      error("D-Bus: Error listing windows", error2);
      throw error2;
    }
  }
  Details(winid) {
    try {
      const details = this.windowManager.details(winid);
      return JSON.stringify(details);
    } catch (error2) {
      error("D-Bus: Error getting window details", error2);
      throw error2;
    }
  }
  GetTitle(winid) {
    try {
      return this.windowManager.getTitle(winid);
    } catch (error2) {
      error("D-Bus: Error getting window title", error2);
      throw error2;
    }
  }
  GetFrameRect(winid) {
    try {
      const frameRect = this.windowManager.getFrameRect(winid);
      return JSON.stringify(frameRect);
    } catch (error2) {
      error("D-Bus: Error getting window frame rect", error2);
      throw error2;
    }
  }
  GetFrameBounds(winid) {
    try {
      const frameBounds = this.windowManager.getFrameBounds(winid);
      return JSON.stringify(frameBounds);
    } catch (error2) {
      error("D-Bus: Error getting window frame bounds", error2);
      throw error2;
    }
  }
  MoveToWorkspace(winid, workspaceNum) {
    try {
      this.windowManager.moveToWorkspace(winid, workspaceNum);
    } catch (error2) {
      error("D-Bus: Error moving window to workspace", error2);
      throw error2;
    }
  }
  MoveResize(winid, x, y, width, height) {
    try {
      this.windowManager.moveResize(winid, x, y, width, height);
    } catch (error2) {
      error("D-Bus: Error move resizing window", error2);
      throw error2;
    }
  }
  Resize(winid, width, height) {
    try {
      this.windowManager.resize(winid, width, height);
    } catch (error2) {
      error("D-Bus: Error resizing window", error2);
      throw error2;
    }
  }
  Move(winid, x, y) {
    try {
      this.windowManager.move(winid, x, y);
    } catch (error2) {
      error("D-Bus: Error moving window", error2);
      throw error2;
    }
  }
  Maximize(winid) {
    try {
      this.windowManager.maximize(winid);
    } catch (error2) {
      error("D-Bus: Error maximizing window", error2);
      throw error2;
    }
  }
  Minimize(winid) {
    try {
      this.windowManager.minimize(winid);
    } catch (error2) {
      error("D-Bus: Error minimizing window", error2);
      throw error2;
    }
  }
  Unmaximize(winid) {
    try {
      this.windowManager.unmaximize(winid);
    } catch (error2) {
      error("D-Bus: Error unmaximizing window", error2);
      throw error2;
    }
  }
  Unminimize(winid) {
    try {
      this.windowManager.unminimize(winid);
    } catch (error2) {
      error("D-Bus: Error unminimizing window", error2);
      throw error2;
    }
  }
  Activate(winid) {
    try {
      this.windowManager.activate(winid);
    } catch (error2) {
      error("D-Bus: Error activating window", error2);
      throw error2;
    }
  }
  Close(winid) {
    try {
      this.windowManager.close(winid);
    } catch (error2) {
      error("D-Bus: Error closing window", error2);
      throw error2;
    }
  }
  ListWorkspaces() {
    try {
      const workspaces = this.windowManager.listWorkspaces();
      return JSON.stringify(workspaces);
    } catch (error2) {
      error("D-Bus: Error listing workspaces", error2);
      throw error2;
    }
  }
  GetActiveWorkspace() {
    try {
      const workspace = this.windowManager.getActiveWorkspace();
      return JSON.stringify(workspace);
    } catch (error2) {
      error("D-Bus: Error getting active workspace", error2);
      throw error2;
    }
  }
  GetWorkspaceWindows(workspaceIndex) {
    try {
      const windows = this.windowManager.list();
      const workspaceWindows = windows.filter(
        (win) => win.workspace === workspaceIndex
      );
      return JSON.stringify(workspaceWindows);
    } catch (error2) {
      error("D-Bus: Error getting workspace windows", error2);
      throw error2;
    }
  }
  // Signal emission methods - called by window manager when events occur
  emitOpenWindow(windowAddress, workspaceName, wmClass, title) {
    try {
      debug(`Emitting openwindow signal for ${title} (${wmClass})`);
      this.dbusObject?.emit_signal(
        "openwindow",
        GLib.Variant.new("(ssss)", [
          String(windowAddress),
          String(workspaceName),
          String(wmClass),
          String(title)
        ])
      );
    } catch (signalError) {
      error("Error emitting openwindow signal", signalError);
    }
  }
  emitCloseWindow(windowAddress) {
    try {
      debug(`Emitting closewindow signal for ${windowAddress}`);
      this.dbusObject?.emit_signal(
        "closewindow",
        GLib.Variant.new("(s)", [String(windowAddress)])
      );
    } catch (signalError) {
      error("Error emitting closewindow signal", signalError);
    }
  }
  emitFocusWindow(windowAddress) {
    try {
      debug(`Emitting focuswindow signal for ${windowAddress}`);
      this.dbusObject?.emit_signal(
        "focuswindow",
        GLib.Variant.new("(s)", [String(windowAddress)])
      );
    } catch (signalError) {
      error("Error emitting focuswindow signal", signalError);
    }
  }
  emitMoveWindow(windowAddress, x, y, width, height) {
    try {
      debug(`Emitting movewindow signal for ${windowAddress}`);
      this.dbusObject?.emit_signal(
        "movewindow",
        GLib.Variant.new("(siiuu)", [
          String(windowAddress),
          x,
          y,
          width,
          height
        ])
      );
    } catch (signalError) {
      error("Error emitting movewindow signal", signalError);
    }
  }
  emitStateWindow(windowAddress, state) {
    try {
      debug(`Emitting statewindow signal for ${windowAddress}: ${state}`);
      this.dbusObject?.emit_signal(
        "statewindow",
        GLib.Variant.new("(ss)", [
          String(windowAddress),
          String(state)
        ])
      );
    } catch (signalError) {
      error("Error emitting statewindow signal", signalError);
    }
  }
  emitWorkspaceChanged(workspaceId) {
    try {
      debug(`Emitting workspacechanged signal for ${workspaceId}`);
      this.dbusObject?.emit_signal(
        "workspacechanged",
        GLib.Variant.new("(s)", [String(workspaceId)])
      );
    } catch (signalError) {
      error("Error emitting workspacechanged signal", signalError);
    }
  }
  emitMonitorLayoutChanged() {
    try {
      debug("Emitting monitorlayoutchanged signal");
      this.dbusObject?.emit_signal(
        "monitorlayoutchanged",
        GLib.Variant.new("()", [])
      );
    } catch (signalError) {
      error("Error emitting monitorlayoutchanged signal", signalError);
    }
  }
};
export {
  WindowsService
};
