// src/core/launcher/click-handler.ts
import Clutter from "gi://Clutter";

// src/utils/logger.ts
var PROJECT_NAME = "Vicinae";
var LogLevel = /* @__PURE__ */ ((LogLevel2) => {
  LogLevel2[LogLevel2["ERROR"] = 0] = "ERROR";
  LogLevel2[LogLevel2["WARN"] = 1] = "WARN";
  LogLevel2[LogLevel2["INFO"] = 2] = "INFO";
  LogLevel2[LogLevel2["DEBUG"] = 3] = "DEBUG";
  return LogLevel2;
})(LogLevel || {});
var currentLogLevel = 2 /* INFO */;
var log = (level, message, data) => {
  if (level > currentLogLevel) {
    return;
  }
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const levelName = LogLevel[level];
  const prefix = `[${PROJECT_NAME}] ${timestamp} ${levelName}`;
  if (data) {
    console.log(`${prefix}: ${message}`);
    if (typeof data === "object" && data !== null) {
      Object.entries(data).forEach(([key, value]) => {
        console.log(`${prefix}:   ${key}: ${value}`);
      });
    } else {
      console.log(`${prefix}: ${data}`);
    }
  } else {
    console.log(`${prefix}: ${message}`);
  }
};
var info = (message, data) => {
  log(2 /* INFO */, message, data);
};
var error = (message, error2) => {
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const prefix = `[${PROJECT_NAME}] ${timestamp} ERROR`;
  if (error2) {
    console.error(`${prefix}: ${message}`);
    console.error(`${prefix}: ${String(error2)}`);
  } else {
    console.error(`${prefix}: ${message}`);
  }
};

// src/core/launcher/click-handler.ts
var ClickHandler = class {
  constructor(appClass, onClickOutside) {
    this.appClass = appClass;
    this.onClickOutside = onClickOutside;
  }
  buttonPressHandler;
  enable() {
    try {
      this.buttonPressHandler = global.stage.connect(
        "captured-event",
        (_stage, event) => {
          if (event.type() === Clutter.EventType.BUTTON_PRESS) {
            this.handleClick();
          }
          return Clutter.EVENT_PROPAGATE;
        }
      );
      info("ClickHandler: Click tracking enabled");
    } catch (error2) {
      error("ClickHandler: Error enabling click tracking", error2);
      throw error2;
    }
  }
  disable() {
    if (this.buttonPressHandler) {
      global.stage.disconnect(this.buttonPressHandler);
      this.buttonPressHandler = void 0;
    }
    info("ClickHandler: Click tracking disabled");
  }
  handleClick() {
    const [x, y] = global.get_pointer();
    const windows = global.get_window_actors();
    const window = windows.find((actor) => {
      const win = actor.meta_window;
      if (!win) return false;
      const rect = win.get_frame_rect();
      return x >= rect.x && x <= rect.x + rect.width && y >= rect.y && y <= rect.y + rect.height;
    });
    const clickedWindow = window?.meta_window;
    const isOurWindow = clickedWindow?.get_wm_class()?.toLowerCase().includes(this.appClass.toLowerCase());
    if (!clickedWindow || !isOurWindow) {
      this.onClickOutside();
    }
  }
};
export {
  ClickHandler
};
