// src/utils/logger.ts
var PROJECT_NAME = "Vicinae";
var LogLevel = /* @__PURE__ */ ((LogLevel2) => {
  LogLevel2[LogLevel2["ERROR"] = 0] = "ERROR";
  LogLevel2[LogLevel2["WARN"] = 1] = "WARN";
  LogLevel2[LogLevel2["INFO"] = 2] = "INFO";
  LogLevel2[LogLevel2["DEBUG"] = 3] = "DEBUG";
  return LogLevel2;
})(LogLevel || {});
var currentLogLevel = 2 /* INFO */;
var log = (level, message, data) => {
  if (level > currentLogLevel) {
    return;
  }
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const levelName = LogLevel[level];
  const prefix = `[${PROJECT_NAME}] ${timestamp} ${levelName}`;
  if (data) {
    console.log(`${prefix}: ${message}`);
    if (typeof data === "object" && data !== null) {
      Object.entries(data).forEach(([key, value]) => {
        console.log(`${prefix}:   ${key}: ${value}`);
      });
    } else {
      console.log(`${prefix}: ${data}`);
    }
  } else {
    console.log(`${prefix}: ${message}`);
  }
};
var debug = (message, data) => {
  log(3 /* DEBUG */, message, data);
};
var info = (message, data) => {
  log(2 /* INFO */, message, data);
};
var warn = (message, data) => {
  log(1 /* WARN */, message, data);
};
var error = (message, error2) => {
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const prefix = `[${PROJECT_NAME}] ${timestamp} ERROR`;
  if (error2) {
    console.error(`${prefix}: ${message}`);
    console.error(`${prefix}: ${String(error2)}`);
  } else {
    console.error(`${prefix}: ${message}`);
  }
};

// src/core/windows/window-manager.ts
import Meta from "gi://Meta";

// src/utils/window-utils.ts
var getWindowById = (winid) => {
  if (!winid || winid <= 0) return null;
  try {
    const windowActors = global.get_window_actors();
    return windowActors.find((w) => {
      try {
        return w.meta_window && w.meta_window.get_id() === winid;
      } catch {
        return false;
      }
    });
  } catch (_error) {
    return null;
  }
};
var getCurrentTime = () => {
  return global.get_current_time();
};

// src/core/windows/workspace-manager.ts
var WorkspaceManager = class {
  getWorkspaceCount() {
    try {
      const workspaceManager = global.workspace_manager;
      return workspaceManager.get_n_workspaces();
    } catch (error2) {
      error("Error getting workspace count", error2);
      return 0;
    }
  }
  getCurrentWorkspaceIndex() {
    try {
      const workspaceManager = global.workspace_manager;
      const currentWorkspace = workspaceManager.get_active_workspace();
      return currentWorkspace.index();
    } catch (error2) {
      error("Error getting current workspace index", error2);
      return 0;
    }
  }
  getWorkspaceByIndex(index) {
    try {
      const workspaceManager = global.workspace_manager;
      return workspaceManager.get_workspace_by_index(index);
    } catch (error2) {
      error("Error getting workspace by index", error2);
      return null;
    }
  }
  switchToWorkspace(index) {
    try {
      const workspace = this.getWorkspaceByIndex(index);
      if (workspace) {
        workspace.activate(global.get_current_time());
      }
    } catch (error2) {
      error("Error switching to workspace", error2);
      throw error2;
    }
  }
  getWorkspaceInfo(index) {
    try {
      const workspace = this.getWorkspaceByIndex(index);
      if (workspace) {
        const windows = workspace.list_windows();
        const hasFullscreen = windows.some(
          (win) => win.get_maximized() === 3
        );
        let monitor = 0;
        if (windows.length > 0) {
          monitor = windows[0].get_monitor();
        }
        return {
          index: workspace.index(),
          name: `Workspace ${workspace.index() + 1}`,
          isActive: workspace === global.workspace_manager.get_active_workspace(),
          windowCount: windows.length,
          monitor,
          hasfullscreen: hasFullscreen
        };
      }
      return null;
    } catch (error2) {
      error("Error getting workspace info", error2);
      return null;
    }
  }
  getAllWorkspaces() {
    try {
      const workspaceManager = global.workspace_manager;
      const workspaces = [];
      for (let i = 0; i < workspaceManager.get_n_workspaces(); i++) {
        const workspace = workspaceManager.get_workspace_by_index(i);
        if (workspace) {
          const workspaceInfo = this.getWorkspaceInfo(i);
          if (workspaceInfo) {
            workspaces.push(workspaceInfo);
          }
        }
      }
      return workspaces;
    } catch (error2) {
      error("Error getting all workspaces", error2);
      return [];
    }
  }
};

// src/core/windows/window-manager.ts
var VicinaeWindowManager = class {
  list() {
    const windowActors = global.get_window_actors();
    const workspaceManager = global.workspace_manager;
    const windows = windowActors.map((w) => {
      const metaWindow = w.meta_window;
      const windowWorkspace = metaWindow.get_workspace();
      const frame = metaWindow.get_frame_rect();
      return {
        id: metaWindow.get_id(),
        title: metaWindow.get_title(),
        wm_class: metaWindow.get_wm_class(),
        wm_class_instance: metaWindow.get_wm_class_instance(),
        pid: metaWindow.get_pid(),
        maximized: metaWindow.get_maximized() !== 0,
        // 0 means not maximized
        display: metaWindow.get_display(),
        frame_type: metaWindow.get_frame_type(),
        window_type: metaWindow.get_window_type(),
        layer: metaWindow.get_layer(),
        monitor: metaWindow.get_monitor(),
        role: metaWindow.get_role(),
        width: frame.width,
        height: frame.height,
        x: frame.x,
        y: frame.y,
        in_current_workspace: metaWindow.located_on_workspace?.(
          workspaceManager.get_active_workspace?.()
        ),
        canclose: metaWindow.can_close(),
        canmaximize: metaWindow.can_maximize(),
        canminimize: metaWindow.can_minimize(),
        canshade: false,
        // can_shade() is not in the type definitions
        moveable: metaWindow.allows_move(),
        resizeable: metaWindow.allows_resize(),
        has_focus: metaWindow.has_focus(),
        workspace: windowWorkspace ? windowWorkspace.index() : -1
      };
    });
    return windows;
  }
  details(winid) {
    const w = getWindowById(winid);
    if (!w) {
      throw new Error("Window not found");
    }
    const metaWindow = w.meta_window;
    const workspaceManager = global.workspace_manager;
    const windowWorkspace = metaWindow.get_workspace();
    const frame = metaWindow.get_frame_rect();
    const win = {
      id: metaWindow.get_id(),
      title: metaWindow.get_title(),
      wm_class: metaWindow.get_wm_class(),
      wm_class_instance: metaWindow.get_wm_class_instance(),
      pid: metaWindow.get_pid(),
      maximized: metaWindow.get_maximized() !== 0,
      // 0 means not maximized
      display: metaWindow.get_display(),
      frame_type: metaWindow.get_frame_type(),
      window_type: metaWindow.get_window_type(),
      layer: metaWindow.get_layer(),
      monitor: metaWindow.get_monitor(),
      role: metaWindow.get_role(),
      width: frame.width,
      height: frame.height,
      x: frame.x,
      y: frame.y,
      in_current_workspace: metaWindow.located_on_workspace?.(
        workspaceManager.get_active_workspace?.()
      ),
      canclose: metaWindow.can_close(),
      canmaximize: metaWindow.can_maximize(),
      canminimize: metaWindow.can_minimize(),
      canshade: false,
      // can_shade() is not in the type definitions
      moveable: metaWindow.allows_move(),
      resizeable: metaWindow.allows_resize(),
      has_focus: metaWindow.has_focus(),
      workspace: windowWorkspace ? windowWorkspace.index() : -1
    };
    return win;
  }
  getTitle(winid) {
    const w = getWindowById(winid);
    if (w) {
      return w.meta_window.get_title();
    } else {
      throw new Error("Window not found");
    }
  }
  getFrameRect(winid) {
    const w = getWindowById(winid);
    if (w) {
      const frame = w.meta_window.get_frame_rect();
      return {
        x: frame.x,
        y: frame.y,
        width: frame.width,
        height: frame.height
      };
    } else {
      throw new Error("Window not found");
    }
  }
  getFrameBounds(winid) {
    const w = getWindowById(winid);
    if (w) {
      return {
        frame_bounds: w.meta_window.get_frame_bounds()
      };
    } else {
      throw new Error("Window not found");
    }
  }
  moveToWorkspace(winid, workspaceNum) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.change_workspace_by_index(workspaceNum, false);
    } else {
      throw new Error("Window not found");
    }
  }
  moveResize(winid, x, y, width, height) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta.MaximizeFlags.BOTH);
      }
      win.meta_window.move_resize_frame(true, x, y, width, height);
    } else {
      throw new Error("Window not found");
    }
  }
  resize(winid, width, height) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta.MaximizeFlags.BOTH);
      }
      win.meta_window.move_resize_frame(
        true,
        win.get_x(),
        win.get_y(),
        width,
        height
      );
    } else {
      throw new Error("Window not found");
    }
  }
  move(winid, x, y) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta.MaximizeFlags.BOTH);
      }
      win.meta_window.move_frame(true, x, y);
    } else {
      throw new Error("Window not found");
    }
  }
  maximize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.maximize(Meta.MaximizeFlags.BOTH);
    } else {
      throw new Error("Window not found");
    }
  }
  minimize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.minimize();
    } else {
      throw new Error("Window not found");
    }
  }
  unmaximize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.unmaximize(Meta.MaximizeFlags.BOTH);
    } else {
      throw new Error("Window not found");
    }
  }
  unminimize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.unminimize();
    } else {
      throw new Error("Window not found");
    }
  }
  activate(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      const currentTime = getCurrentTime();
      const workspace = win.get_workspace();
      if (workspace) {
        workspace.activate_with_focus(win, currentTime);
      } else {
        win.activate(currentTime);
      }
    } else {
      throw new Error("Window not found");
    }
  }
  close(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      try {
        if (win.get_id() === winid) {
          win.delete(getCurrentTime());
        } else {
          throw new Error(
            "Window ID mismatch - window may be destroyed"
          );
        }
      } catch (error2) {
        throw new Error(`Failed to close window ${winid}: ${error2}`);
      }
    } else {
      throw new Error("Window not found");
    }
  }
  listWorkspaces() {
    const workspaceManager = new WorkspaceManager();
    return workspaceManager.getAllWorkspaces();
  }
  getActiveWorkspace() {
    const workspaceManager = new WorkspaceManager();
    const currentIndex = workspaceManager.getCurrentWorkspaceIndex();
    const workspace = workspaceManager.getWorkspaceInfo(currentIndex);
    if (!workspace) {
      throw new Error("No active workspace found");
    }
    return workspace;
  }
};

// src/core/launcher/click-handler.ts
import Clutter from "gi://Clutter";
var ClickHandler = class {
  constructor(appClass, onClickOutside) {
    this.appClass = appClass;
    this.onClickOutside = onClickOutside;
  }
  buttonPressHandler;
  enable() {
    try {
      this.buttonPressHandler = global.stage.connect(
        "captured-event",
        (_stage, event) => {
          if (event.type() === Clutter.EventType.BUTTON_PRESS) {
            this.handleClick();
          }
          return Clutter.EVENT_PROPAGATE;
        }
      );
      info("ClickHandler: Click tracking enabled");
    } catch (error2) {
      error("ClickHandler: Error enabling click tracking", error2);
      throw error2;
    }
  }
  disable() {
    if (this.buttonPressHandler) {
      global.stage.disconnect(this.buttonPressHandler);
      this.buttonPressHandler = void 0;
    }
    info("ClickHandler: Click tracking disabled");
  }
  handleClick() {
    const [x, y] = global.get_pointer();
    const windows = global.get_window_actors();
    const window = windows.find((actor) => {
      const win = actor.meta_window;
      if (!win) return false;
      const rect = win.get_frame_rect();
      return x >= rect.x && x <= rect.x + rect.width && y >= rect.y && y <= rect.y + rect.height;
    });
    const clickedWindow = window?.meta_window;
    const isOurWindow = clickedWindow?.get_wm_class()?.toLowerCase().includes(this.appClass.toLowerCase());
    if (!clickedWindow || !isOurWindow) {
      this.onClickOutside();
    }
  }
};

// src/core/launcher/focus-tracker.ts
var FocusTracker = class {
  constructor(onFocusChange) {
    this.onFocusChange = onFocusChange;
  }
  focusHandler;
  enable() {
    try {
      this.focusHandler = global.display.connect(
        "notify::focus-window",
        (_display, _window) => {
          this.onFocusChange();
        }
      );
      info("FocusTracker: Focus tracking enabled");
    } catch (error2) {
      error("FocusTracker: Error enabling focus tracking", error2);
      throw error2;
    }
  }
  disable() {
    if (this.focusHandler) {
      global.display.disconnect(this.focusHandler);
      this.focusHandler = void 0;
    }
    info("FocusTracker: Focus tracking disabled");
  }
};

// src/core/launcher/window-tracker.ts
import GLib from "gi://GLib";
var WindowTracker = class {
  constructor(appClass, onWindowTracked, onWindowUntracked) {
    this.appClass = appClass;
    this.onWindowTracked = onWindowTracked;
    this.onWindowUntracked = onWindowUntracked;
  }
  trackedWindows = /* @__PURE__ */ new Set();
  windowCreatedHandler;
  windowDestroySignalIds = /* @__PURE__ */ new Map();
  windowValidators = /* @__PURE__ */ new Map();
  isDestroying = false;
  enable() {
    try {
      this.windowCreatedHandler = global.display.connect(
        "window-created",
        (_display, window) => {
          GLib.idle_add(GLib.PRIORITY_DEFAULT, () => {
            this.handleNewWindow(window);
            return GLib.SOURCE_REMOVE;
          });
        }
      );
      this.scanExistingWindows();
      info("WindowTracker: Window tracking enabled");
    } catch (error2) {
      error("WindowTracker: Error enabling window tracking", error2);
      throw error2;
    }
  }
  disable() {
    this.isDestroying = true;
    if (this.windowCreatedHandler) {
      global.display.disconnect(this.windowCreatedHandler);
      this.windowCreatedHandler = void 0;
    }
    for (const [windowId, signalId] of this.windowDestroySignalIds) {
      if (signalId) {
        try {
          const window = this.getWindowById(windowId);
          if (window && this.isWindowValid(window)) {
            window.disconnect(signalId);
          }
        } catch (_error) {
          debug(
            `WindowTracker: Signal already disconnected for window ${windowId}`
          );
        }
      }
    }
    this.windowDestroySignalIds.clear();
    this.windowValidators.clear();
    this.trackedWindows.clear();
    this.isDestroying = false;
    info("WindowTracker: Window tracking disabled");
  }
  scanExistingWindows() {
    try {
      const windowActors = global.get_window_actors();
      debug(
        `WindowTracker: Scanning ${windowActors.length} existing windows`
      );
      windowActors.forEach((actor) => {
        if (actor?.meta_window) {
          this.handleNewWindow(actor.meta_window);
        } else {
          debug("WindowTracker: Skipping invalid window actor");
        }
      });
    } catch (error2) {
      error("WindowTracker: Error scanning existing windows", error2);
    }
  }
  handleNewWindow(window) {
    if (this.isDestroying || !this.isValidWindow(window)) {
      debug(
        "WindowTracker: Invalid window object or destroying, skipping"
      );
      return;
    }
    try {
      const wmClass = window.get_wm_class();
      const title = window.get_title();
      const windowId = window.get_id();
      if (!wmClass || !title || windowId <= 0) {
        debug("WindowTracker: Invalid window properties", {
          wmClass,
          title,
          windowId
        });
        return;
      }
      if (wmClass.toLowerCase().includes(this.appClass.toLowerCase())) {
        if (!this.trackedWindows.has(windowId)) {
          this.trackedWindows.add(windowId);
          const validator = () => this.isWindowValid(window);
          this.windowValidators.set(windowId, validator);
          this.onWindowTracked(windowId);
          this.centerWindow(window);
          let signalId;
          try {
            signalId = window.connect("destroy", () => {
              this.handleWindowDestroyed(window);
            });
          } catch (_error) {
            try {
              signalId = window.connect("unmanaged", () => {
                this.handleWindowDestroyed(window);
              });
            } catch (_unmanagedError) {
              debug(
                `WindowTracker: No suitable destroy signal for window ${windowId}, skipping signal connection`
              );
            }
          }
          if (signalId) {
            this.windowDestroySignalIds.set(windowId, signalId);
          }
          debug(
            `WindowTracker: Tracking new window ${windowId} (${wmClass})`
          );
        }
      }
    } catch (error2) {
      error("WindowTracker: Error handling new window", {
        error: error2 instanceof Error ? error2.message : String(error2),
        stack: error2 instanceof Error ? error2.stack : void 0,
        windowId: window?.get_id?.() || "unknown"
      });
    }
  }
  isValidWindow(window) {
    if (!window) return false;
    if (typeof window.get_wm_class !== "function") return false;
    if (typeof window.get_id !== "function") return false;
    try {
      const windowId = window.get_id();
      return windowId > 0 && windowId !== void 0;
    } catch {
      return false;
    }
  }
  isWindowValid(window) {
    if (!window) return false;
    try {
      const windowActors = global.get_window_actors();
      const stillExists = windowActors.some(
        (actor) => actor.meta_window && actor.meta_window.get_id() === window.get_id()
      );
      if (!stillExists) return false;
      window.get_id();
      window.get_wm_class();
      return true;
    } catch {
      return false;
    }
  }
  handleWindowDestroyed(window) {
    if (this.isDestroying || !this.isValidWindow(window)) return;
    const windowId = window.get_id();
    if (this.trackedWindows.has(windowId)) {
      this.trackedWindows.delete(windowId);
      this.windowValidators.delete(windowId);
      this.onWindowUntracked(windowId);
      this.windowDestroySignalIds.delete(windowId);
      debug(`WindowTracker: Untracking destroyed window ${windowId}`);
    }
  }
  getWindowById(windowId) {
    try {
      const actors = global.get_window_actors();
      for (const actor of actors) {
        const window = actor.meta_window;
        if (window && window.get_id() === windowId) {
          return window;
        }
      }
    } catch (error2) {
      error(`WindowTracker: Error finding window ${windowId}`, error2);
    }
    return null;
  }
  // Public methods for external access
  getTrackedWindows() {
    return Array.from(this.trackedWindows);
  }
  getTrackedWindowsCount() {
    return this.trackedWindows.size;
  }
  /**
   * Centers a window on the current monitor
   */
  centerWindow(window) {
    if (this.isDestroying || !this.isWindowValid(window)) {
      debug(
        "WindowTracker: Skipping center window - window invalid or destroying"
      );
      return;
    }
    try {
      const { x, y } = this.getCenterPosition(window);
      window.move_frame(true, x, y);
      debug(
        `WindowTracker: Centered window ${window.get_id()} at (${Math.round(
          x
        )}, ${Math.round(y)})`
      );
    } catch (error2) {
      error("WindowTracker: Error centering window", error2);
    }
  }
  /**
   * Gets the center position of a window
   */
  getCenterPosition(window) {
    const monitor = window.get_monitor();
    const display = global.display;
    const monitorGeometry = display.get_monitor_geometry(monitor);
    const frame = window.get_frame_rect();
    const centerX = monitorGeometry.x + (monitorGeometry.width - frame.width) / 2;
    const centerY = monitorGeometry.y + (monitorGeometry.height - frame.height) / 2;
    return { x: centerX, y: centerY };
  }
};

// src/core/launcher/launcher-manager.ts
var LauncherManager = class {
  windowManager;
  windowTracker;
  focusTracker;
  clickHandler;
  config;
  isEnabled = false;
  trackedWindows = /* @__PURE__ */ new Set();
  constructor(config) {
    this.windowManager = new VicinaeWindowManager();
    this.config = config;
    this.windowTracker = new WindowTracker(
      config.appClass,
      this.handleWindowTracked.bind(this),
      this.handleWindowUntracked.bind(this)
    );
  }
  async enable() {
    if (this.isEnabled) {
      debug("LauncherManager: Already enabled, skipping");
      return;
    }
    try {
      await this.windowTracker.enable();
      if (this.config.autoCloseOnFocusLoss) {
        this.setupFocusTracking();
        this.setupClickHandling();
      }
      this.isEnabled = true;
      info("LauncherManager: Successfully enabled");
    } catch (error2) {
      error("LauncherManager: Error during enable", error2);
      this.cleanup();
      throw error2;
    }
  }
  handleWindowTracked(windowId) {
    this.trackedWindows.add(windowId);
    debug(`LauncherManager: Window ${windowId} is now tracked`);
  }
  handleWindowUntracked(windowId) {
    this.trackedWindows.delete(windowId);
    debug(`LauncherManager: Window ${windowId} is no longer tracked`);
  }
  disable() {
    info("LauncherManager: Disabling");
    this.isEnabled = false;
    this.cleanup();
  }
  setupFocusTracking() {
    this.focusTracker = new FocusTracker(() => this.handleFocusChange());
    this.focusTracker.enable();
  }
  setupClickHandling() {
    this.clickHandler = new ClickHandler(
      this.config.appClass,
      () => this.closeTrackedWindows()
    );
    this.clickHandler.enable();
  }
  handleFocusChange() {
    if (!this.isEnabled) return;
    const focusedWindow = global.display.get_focus_window();
    if (!focusedWindow) {
      this.closeTrackedWindows();
      return;
    }
    const focusedWmClass = focusedWindow.get_wm_class()?.toLowerCase() || "";
    if (!focusedWmClass.includes(this.config.appClass.toLowerCase())) {
      this.closeTrackedWindows();
    }
  }
  closeTrackedWindows() {
    if (this.trackedWindows.size === 0) return;
    debug(
      `LauncherManager: Closing ${this.trackedWindows.size} launcher windows due to focus loss`
    );
    const windowsToClose = Array.from(this.trackedWindows);
    this.trackedWindows.clear();
    windowsToClose.forEach((windowId) => {
      try {
        if (this.isValidWindowId(windowId)) {
          this.windowManager.close(windowId);
          debug(
            `LauncherManager: Successfully closed window ${windowId}`
          );
        } else {
          debug(
            `LauncherManager: Window ${windowId} no longer valid, skipping close`
          );
        }
      } catch (error2) {
        error(
          `LauncherManager: Error closing window ${windowId}`,
          error2
        );
      }
    });
  }
  isValidWindowId(windowId) {
    if (!windowId || windowId <= 0) return false;
    try {
      const windowActors = global.get_window_actors();
      const windowExists = windowActors.some((actor) => {
        try {
          return actor.meta_window && actor.meta_window.get_id() === windowId;
        } catch {
          return false;
        }
      });
      if (!windowExists) return false;
      const details = this.windowManager.details(windowId);
      return details && details.id === windowId;
    } catch {
      return false;
    }
  }
  cleanup() {
    try {
      this.windowTracker.disable();
      this.focusTracker?.disable();
      this.clickHandler?.disable();
      this.trackedWindows.clear();
    } catch (error2) {
      error("LauncherManager: Error during cleanup", error2);
    }
  }
  // Recovery method for error situations
  recover() {
    warn("LauncherManager: Attempting recovery from errors");
    try {
      this.cleanup();
      this.isEnabled = false;
      setTimeout(() => {
        this.enable();
      }, 500);
      info("LauncherManager: Recovery initiated");
    } catch (error2) {
      error("LauncherManager: Recovery failed", error2);
    }
  }
  // Public methods for dynamic configuration
  updateConfig(newConfig) {
    const oldConfig = { ...this.config };
    this.config = { ...this.config, ...newConfig };
    debug("LauncherManager: Configuration updated", {
      old: oldConfig,
      new: this.config
    });
    if (this.isEnabled) {
      info("LauncherManager: Re-enabling with new configuration");
      this.disable();
      setTimeout(() => this.enable(), 100);
    }
  }
  getTrackedWindows() {
    return Array.from(this.trackedWindows);
  }
  getStatus() {
    return {
      isEnabled: this.isEnabled,
      trackedWindowsCount: this.trackedWindows.size,
      config: this.config,
      hasFocusTracker: !!this.focusTracker,
      hasClickHandler: !!this.clickHandler,
      hasWindowTracker: !!this.windowTracker
    };
  }
  // Force refresh of tracked windows
  refresh() {
    debug("LauncherManager: Refreshing tracked windows");
    this.trackedWindows.clear();
  }
};
export {
  LauncherManager
};
