// src/core/windows/window-manager.ts
import Meta from "gi://Meta";

// src/utils/window-utils.ts
var getWindowById = (winid) => {
  if (!winid || winid <= 0) return null;
  try {
    const windowActors = global.get_window_actors();
    return windowActors.find((w) => {
      try {
        return w.meta_window && w.meta_window.get_id() === winid;
      } catch {
        return false;
      }
    });
  } catch (_error) {
    return null;
  }
};
var getCurrentTime = () => {
  return global.get_current_time();
};

// src/utils/logger.ts
var PROJECT_NAME = "Vicinae";
var error = (message, error2) => {
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const prefix = `[${PROJECT_NAME}] ${timestamp} ERROR`;
  if (error2) {
    console.error(`${prefix}: ${message}`);
    console.error(`${prefix}: ${String(error2)}`);
  } else {
    console.error(`${prefix}: ${message}`);
  }
};

// src/core/windows/workspace-manager.ts
var WorkspaceManager = class {
  getWorkspaceCount() {
    try {
      const workspaceManager = global.workspace_manager;
      return workspaceManager.get_n_workspaces();
    } catch (error2) {
      error("Error getting workspace count", error2);
      return 0;
    }
  }
  getCurrentWorkspaceIndex() {
    try {
      const workspaceManager = global.workspace_manager;
      const currentWorkspace = workspaceManager.get_active_workspace();
      return currentWorkspace.index();
    } catch (error2) {
      error("Error getting current workspace index", error2);
      return 0;
    }
  }
  getWorkspaceByIndex(index) {
    try {
      const workspaceManager = global.workspace_manager;
      return workspaceManager.get_workspace_by_index(index);
    } catch (error2) {
      error("Error getting workspace by index", error2);
      return null;
    }
  }
  switchToWorkspace(index) {
    try {
      const workspace = this.getWorkspaceByIndex(index);
      if (workspace) {
        workspace.activate(global.get_current_time());
      }
    } catch (error2) {
      error("Error switching to workspace", error2);
      throw error2;
    }
  }
  getWorkspaceInfo(index) {
    try {
      const workspace = this.getWorkspaceByIndex(index);
      if (workspace) {
        const windows = workspace.list_windows();
        const hasFullscreen = windows.some(
          (win) => win.get_maximized() === 3
        );
        let monitor = 0;
        if (windows.length > 0) {
          monitor = windows[0].get_monitor();
        }
        return {
          index: workspace.index(),
          name: `Workspace ${workspace.index() + 1}`,
          isActive: workspace === global.workspace_manager.get_active_workspace(),
          windowCount: windows.length,
          monitor,
          hasfullscreen: hasFullscreen
        };
      }
      return null;
    } catch (error2) {
      error("Error getting workspace info", error2);
      return null;
    }
  }
  getAllWorkspaces() {
    try {
      const workspaceManager = global.workspace_manager;
      const workspaces = [];
      for (let i = 0; i < workspaceManager.get_n_workspaces(); i++) {
        const workspace = workspaceManager.get_workspace_by_index(i);
        if (workspace) {
          const workspaceInfo = this.getWorkspaceInfo(i);
          if (workspaceInfo) {
            workspaces.push(workspaceInfo);
          }
        }
      }
      return workspaces;
    } catch (error2) {
      error("Error getting all workspaces", error2);
      return [];
    }
  }
};

// src/core/windows/window-manager.ts
var VicinaeWindowManager = class {
  list() {
    const windowActors = global.get_window_actors();
    const workspaceManager = global.workspace_manager;
    const windows = windowActors.map((w) => {
      const metaWindow = w.meta_window;
      const windowWorkspace = metaWindow.get_workspace();
      const frame = metaWindow.get_frame_rect();
      return {
        id: metaWindow.get_id(),
        title: metaWindow.get_title(),
        wm_class: metaWindow.get_wm_class(),
        wm_class_instance: metaWindow.get_wm_class_instance(),
        pid: metaWindow.get_pid(),
        maximized: metaWindow.get_maximized() !== 0,
        // 0 means not maximized
        display: metaWindow.get_display(),
        frame_type: metaWindow.get_frame_type(),
        window_type: metaWindow.get_window_type(),
        layer: metaWindow.get_layer(),
        monitor: metaWindow.get_monitor(),
        role: metaWindow.get_role(),
        width: frame.width,
        height: frame.height,
        x: frame.x,
        y: frame.y,
        in_current_workspace: metaWindow.located_on_workspace?.(
          workspaceManager.get_active_workspace?.()
        ),
        canclose: metaWindow.can_close(),
        canmaximize: metaWindow.can_maximize(),
        canminimize: metaWindow.can_minimize(),
        canshade: false,
        // can_shade() is not in the type definitions
        moveable: metaWindow.allows_move(),
        resizeable: metaWindow.allows_resize(),
        has_focus: metaWindow.has_focus(),
        workspace: windowWorkspace ? windowWorkspace.index() : -1
      };
    });
    return windows;
  }
  details(winid) {
    const w = getWindowById(winid);
    if (!w) {
      throw new Error("Window not found");
    }
    const metaWindow = w.meta_window;
    const workspaceManager = global.workspace_manager;
    const windowWorkspace = metaWindow.get_workspace();
    const frame = metaWindow.get_frame_rect();
    const win = {
      id: metaWindow.get_id(),
      title: metaWindow.get_title(),
      wm_class: metaWindow.get_wm_class(),
      wm_class_instance: metaWindow.get_wm_class_instance(),
      pid: metaWindow.get_pid(),
      maximized: metaWindow.get_maximized() !== 0,
      // 0 means not maximized
      display: metaWindow.get_display(),
      frame_type: metaWindow.get_frame_type(),
      window_type: metaWindow.get_window_type(),
      layer: metaWindow.get_layer(),
      monitor: metaWindow.get_monitor(),
      role: metaWindow.get_role(),
      width: frame.width,
      height: frame.height,
      x: frame.x,
      y: frame.y,
      in_current_workspace: metaWindow.located_on_workspace?.(
        workspaceManager.get_active_workspace?.()
      ),
      canclose: metaWindow.can_close(),
      canmaximize: metaWindow.can_maximize(),
      canminimize: metaWindow.can_minimize(),
      canshade: false,
      // can_shade() is not in the type definitions
      moveable: metaWindow.allows_move(),
      resizeable: metaWindow.allows_resize(),
      has_focus: metaWindow.has_focus(),
      workspace: windowWorkspace ? windowWorkspace.index() : -1
    };
    return win;
  }
  getTitle(winid) {
    const w = getWindowById(winid);
    if (w) {
      return w.meta_window.get_title();
    } else {
      throw new Error("Window not found");
    }
  }
  getFrameRect(winid) {
    const w = getWindowById(winid);
    if (w) {
      const frame = w.meta_window.get_frame_rect();
      return {
        x: frame.x,
        y: frame.y,
        width: frame.width,
        height: frame.height
      };
    } else {
      throw new Error("Window not found");
    }
  }
  getFrameBounds(winid) {
    const w = getWindowById(winid);
    if (w) {
      return {
        frame_bounds: w.meta_window.get_frame_bounds()
      };
    } else {
      throw new Error("Window not found");
    }
  }
  moveToWorkspace(winid, workspaceNum) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.change_workspace_by_index(workspaceNum, false);
    } else {
      throw new Error("Window not found");
    }
  }
  moveResize(winid, x, y, width, height) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta.MaximizeFlags.BOTH);
      }
      win.meta_window.move_resize_frame(true, x, y, width, height);
    } else {
      throw new Error("Window not found");
    }
  }
  resize(winid, width, height) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta.MaximizeFlags.BOTH);
      }
      win.meta_window.move_resize_frame(
        true,
        win.get_x(),
        win.get_y(),
        width,
        height
      );
    } else {
      throw new Error("Window not found");
    }
  }
  move(winid, x, y) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta.MaximizeFlags.BOTH);
      }
      win.meta_window.move_frame(true, x, y);
    } else {
      throw new Error("Window not found");
    }
  }
  maximize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.maximize(Meta.MaximizeFlags.BOTH);
    } else {
      throw new Error("Window not found");
    }
  }
  minimize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.minimize();
    } else {
      throw new Error("Window not found");
    }
  }
  unmaximize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.unmaximize(Meta.MaximizeFlags.BOTH);
    } else {
      throw new Error("Window not found");
    }
  }
  unminimize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.unminimize();
    } else {
      throw new Error("Window not found");
    }
  }
  activate(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      const currentTime = getCurrentTime();
      const workspace = win.get_workspace();
      if (workspace) {
        workspace.activate_with_focus(win, currentTime);
      } else {
        win.activate(currentTime);
      }
    } else {
      throw new Error("Window not found");
    }
  }
  close(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      try {
        if (win.get_id() === winid) {
          win.delete(getCurrentTime());
        } else {
          throw new Error(
            "Window ID mismatch - window may be destroyed"
          );
        }
      } catch (error2) {
        throw new Error(`Failed to close window ${winid}: ${error2}`);
      }
    } else {
      throw new Error("Window not found");
    }
  }
  listWorkspaces() {
    const workspaceManager = new WorkspaceManager();
    return workspaceManager.getAllWorkspaces();
  }
  getActiveWorkspace() {
    const workspaceManager = new WorkspaceManager();
    const currentIndex = workspaceManager.getCurrentWorkspaceIndex();
    const workspace = workspaceManager.getWorkspaceInfo(currentIndex);
    if (!workspace) {
      throw new Error("No active workspace found");
    }
    return workspace;
  }
};
export {
  VicinaeWindowManager
};
