// src/prefs.ts
import Adw from "gi://Adw";
import Gdk from "gi://Gdk";
import Gio from "gi://Gio";
import GLib from "gi://GLib";
import GObject from "gi://GObject";
import Gtk from "gi://Gtk";
import { ExtensionPreferences } from "resource:///org/gnome/Shell/Extensions/js/extensions/prefs.js";

// src/utils/logger.ts
var PROJECT_NAME = "Vicinae";
var LogLevel = /* @__PURE__ */ ((LogLevel2) => {
  LogLevel2[LogLevel2["ERROR"] = 0] = "ERROR";
  LogLevel2[LogLevel2["WARN"] = 1] = "WARN";
  LogLevel2[LogLevel2["INFO"] = 2] = "INFO";
  LogLevel2[LogLevel2["DEBUG"] = 3] = "DEBUG";
  return LogLevel2;
})(LogLevel || {});
var currentLogLevel = 2 /* INFO */;
var log = (level, message, data) => {
  if (level > currentLogLevel) {
    return;
  }
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const levelName = LogLevel[level];
  const prefix = `[${PROJECT_NAME}] ${timestamp} ${levelName}`;
  if (data) {
    console.log(`${prefix}: ${message}`);
    if (typeof data === "object" && data !== null) {
      Object.entries(data).forEach(([key, value]) => {
        console.log(`${prefix}:   ${key}: ${value}`);
      });
    } else {
      console.log(`${prefix}: ${data}`);
    }
  } else {
    console.log(`${prefix}: ${message}`);
  }
};
var debug = (message, data) => {
  log(3 /* DEBUG */, message, data);
};
var error = (message, error2) => {
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const prefix = `[${PROJECT_NAME}] ${timestamp} ERROR`;
  if (error2) {
    console.error(`${prefix}: ${message}`);
    console.error(`${prefix}: ${String(error2)}`);
  } else {
    console.error(`${prefix}: ${message}`);
  }
};

// src/prefs.ts
var LICENSE = `MIT License

Copyright (c) 2025 Dagim G. Astatkie

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.`;
var getTemplate = (name) => {
  const uri = GLib.uri_resolve_relative(
    import.meta.url,
    `ui/${name}.ui`,
    GLib.UriFlags.NONE
  );
  if (uri === null) {
    throw new Error(`Failed to resolve URI for template ${name}!`);
  }
  return uri;
};
var BlockedAppRow = GObject.registerClass(
  {
    GTypeName: "BlockedAppRow",
    Properties: {
      "window-class": GObject.ParamSpec.string(
        "window-class",
        "Window Class",
        "The window class of the blocked application",
        GObject.ParamFlags.READWRITE,
        ""
      )
    },
    Signals: {
      "delete-requested": {},
      "save-requested": {},
      "input-changed": {}
    }
  },
  class BlockedAppRow2 extends Adw.ExpanderRow {
    windowClass = "";
    inputValue = "";
    // Separate state for input field
    originalWindowClass = "";
    // Store the original value before editing
    windowEntry;
    checkButton;
    deleteButton;
    constructor() {
      super();
      this.set_title("Expand this row to enter window class");
      this.set_subtitle("");
      this.windowEntry = new Adw.EntryRow({
        title: "Window Class",
        text: ""
      });
      this.checkButton = new Gtk.Button({
        icon_name: "emblem-ok-symbolic",
        valign: Gtk.Align.CENTER,
        tooltip_text: "Save and close",
        css_classes: ["flat", "suggested-action"]
      });
      this.deleteButton = new Gtk.Button({
        icon_name: "user-trash-symbolic",
        valign: Gtk.Align.CENTER,
        tooltip_text: "Remove",
        css_classes: ["flat"]
      });
      this.add_suffix(this.checkButton);
      this.add_suffix(this.deleteButton);
      this.add_row(this.windowEntry);
      this.checkButton.visible = false;
      this.checkButton.connect("clicked", () => {
        this.saveChanges();
      });
      this.deleteButton.connect("clicked", () => {
        this.emit("delete-requested");
      });
      this.windowEntry.connect("changed", () => {
        this.inputValue = this.windowEntry.get_text().trim();
        this.updateCheckButtonState();
        this.emit("input-changed");
      });
      this.connect("notify::expanded", () => {
        this.updateButtonVisibility();
      });
    }
    setWindowClass(windowClass) {
      this.windowClass = windowClass;
      this.inputValue = windowClass;
      this.originalWindowClass = windowClass;
      this.windowEntry.set_text(windowClass);
      this.updateDisplay();
      this.updateCheckButtonState();
    }
    getWindowClass() {
      return this.windowClass;
    }
    getInputValue() {
      return this.inputValue;
    }
    // Get the current effective window class (either the original or the edited value)
    getCurrentWindowClass() {
      return this.inputValue.trim() || this.windowClass;
    }
    getOriginalWindowClass() {
      return this.originalWindowClass;
    }
    isEmpty() {
      return this.inputValue.trim() === "";
    }
    closeExpanded() {
      this.set_expanded(false);
    }
    saveChanges() {
      const oldValue = this.windowClass;
      const newValue = this.inputValue.trim();
      if (newValue !== oldValue) {
        this.originalWindowClass = oldValue;
        this.windowClass = newValue;
        this.updateDisplay();
        this.emit("save-requested");
      }
      this.closeExpanded();
    }
    updateButtonVisibility() {
      this.checkButton.visible = this.get_expanded();
      if (this.get_expanded()) {
        this.updateCheckButtonState();
      }
    }
    updateCheckButtonState() {
      this.checkButton.set_sensitive(this.inputValue.trim().length > 0);
    }
    updateDisplay() {
      if (this.windowClass) {
        this.set_title(this.windowClass);
        this.set_subtitle(
          `Expand this row to edit - ${this.windowClass}`
        );
      } else {
        this.set_title("Expand this row to enter window class");
        this.set_subtitle("");
      }
    }
  }
);
var GeneralPage = GObject.registerClass(
  {
    GTypeName: "TypescriptTemplateGeneralPage",
    Template: getTemplate("GeneralPage"),
    InternalChildren: [
      "blockedAppsGroup",
      "addWindowButton",
      "showStatusIndicator",
      "loggingLevel",
      "launcherAutoCloseFocusLoss",
      "launcherAppClass"
    ]
  },
  class GeneralPage2 extends Adw.PreferencesPage {
    settings;
    blockedAppRows = /* @__PURE__ */ new Map();
    emptyRows = /* @__PURE__ */ new Set();
    bindSettings(settings) {
      this.settings = settings;
      debug("Settings bound to GeneralPage");
      this.loadBlockedApplications();
      this.updateAddButtonState();
      const children = this;
      children._addWindowButton.connect("clicked", () => {
        this.addEmptyBlockedAppRow();
      });
      settings.bind(
        "show-status-indicator",
        children._showStatusIndicator,
        "active",
        Gio.SettingsBindFlags.DEFAULT
      );
      const loggingLevels = ["error", "warn", "info", "debug"];
      const currentLevel = settings.get_string("logging-level");
      const currentIndex = loggingLevels.indexOf(currentLevel);
      children._loggingLevel.set_selected(
        currentIndex >= 0 ? currentIndex : 2
      );
      children._loggingLevel.connect("notify::selected", () => {
        const selectedIndex = children._loggingLevel.get_selected();
        if (selectedIndex >= 0 && selectedIndex < loggingLevels.length) {
          settings.set_string(
            "logging-level",
            loggingLevels[selectedIndex]
          );
        }
      });
      settings.bind(
        "launcher-auto-close-focus-loss",
        children._launcherAutoCloseFocusLoss,
        "active",
        Gio.SettingsBindFlags.DEFAULT
      );
      settings.bind(
        "launcher-app-class",
        children._launcherAppClass,
        "text",
        Gio.SettingsBindFlags.DEFAULT
      );
    }
    loadBlockedApplications() {
      try {
        const blockedApps = this.settings.get_strv(
          "blocked-applications"
        );
        const uniqueBlockedApps = this.removeDuplicates(blockedApps);
        if (uniqueBlockedApps.length !== blockedApps.length) {
          this.settings.set_strv(
            "blocked-applications",
            uniqueBlockedApps
          );
        }
        const children = this;
        const existingRows = Array.from(this.blockedAppRows.values());
        existingRows.forEach((row) => {
          children._blockedAppsGroup.remove(row);
        });
        this.blockedAppRows.clear();
        this.emptyRows.clear();
        uniqueBlockedApps.forEach((windowClass) => {
          this.addBlockedAppRow(windowClass);
        });
      } catch (error2) {
        error("Error loading blocked applications", error2);
      }
    }
    removeDuplicates(apps) {
      const seen = /* @__PURE__ */ new Set();
      return apps.filter((app) => {
        const lowerApp = app.toLowerCase();
        if (seen.has(lowerApp)) {
          return false;
        }
        seen.add(lowerApp);
        return true;
      });
    }
    addEmptyBlockedAppRow() {
      this.addBlockedAppRow("");
    }
    addBlockedAppRow(windowClass) {
      const children = this;
      const row = new BlockedAppRow();
      row.setWindowClass(windowClass);
      row.connect("delete-requested", () => {
        this.removeBlockedAppRow(row);
      });
      row.connect("save-requested", () => {
        this.handleSaveRequest(row);
      });
      row.connect("input-changed", () => {
        this.handleInputChange(row);
      });
      children._blockedAppsGroup.add(row);
      if (windowClass) {
        this.blockedAppRows.set(windowClass, row);
      } else {
        this.emptyRows.add(row);
        row.updateCheckButtonState();
      }
      this.updateAddButtonState();
    }
    handleInputChange(row) {
      const isEmpty = row.isEmpty();
      const wasEmpty = this.emptyRows.has(row);
      if (isEmpty && !wasEmpty) {
        this.emptyRows.add(row);
      } else if (!isEmpty && wasEmpty) {
        this.emptyRows.delete(row);
      }
      this.updateAddButtonState();
    }
    updateAddButtonState() {
      const children = this;
      const hasEmptyRows = this.emptyRows.size > 0;
      children._addWindowButton.set_sensitive(!hasEmptyRows);
    }
    handleSaveRequest(row) {
      const oldClass = row.getOriginalWindowClass();
      const newClass = row.getInputValue().trim();
      if (newClass) {
        const currentBlockedApps = this.settings.get_strv(
          "blocked-applications"
        );
        const isDuplicate = currentBlockedApps.some(
          (app) => app !== oldClass && // Exclude the current row's old class
          app.toLowerCase() === newClass.toLowerCase()
        );
        if (isDuplicate) {
          const root = this.get_root();
          if (root && "add_toast" in root) {
            const toast = new Adw.Toast({
              title: `Can't add ${newClass} to the list, because it's already there`
            });
            root.add_toast(toast);
          }
          row.setWindowClass(oldClass);
          return;
        } else {
          this.updateBlockedAppInSettings(row, oldClass, newClass);
        }
      } else {
        this.removeBlockedAppFromSettings(row, oldClass);
      }
      this.updateAddButtonState();
    }
    updateBlockedAppInSettings(row, oldClass, newClass) {
      try {
        const currentBlockedApps = this.settings.get_strv(
          "blocked-applications"
        );
        let filteredApps;
        if (oldClass && oldClass.trim() !== "") {
          filteredApps = currentBlockedApps.filter(
            (app) => app !== oldClass
          );
        } else {
          filteredApps = [...currentBlockedApps];
        }
        filteredApps.push(newClass);
        this.settings.set_strv("blocked-applications", filteredApps);
        if (oldClass && oldClass.trim() !== "") {
          this.blockedAppRows.delete(oldClass);
        }
        this.blockedAppRows.set(newClass, row);
      } catch (error2) {
        error("Error updating blocked app in settings", error2);
      }
    }
    removeBlockedAppFromSettings(_row, oldClass) {
      try {
        if (!oldClass || oldClass.trim() === "") {
          return;
        }
        const currentBlockedApps = this.settings.get_strv(
          "blocked-applications"
        );
        const filteredApps = currentBlockedApps.filter(
          (app) => app !== oldClass
        );
        this.settings.set_strv("blocked-applications", filteredApps);
        this.blockedAppRows.delete(oldClass);
      } catch (error2) {
        error("Error removing blocked app from settings", error2);
      }
    }
    removeBlockedAppRow(row) {
      const children = this;
      const windowClass = row.getWindowClass();
      if (windowClass) {
        const currentApps = this.settings.get_strv(
          "blocked-applications"
        );
        const updatedApps = currentApps.filter(
          (app) => app !== windowClass
        );
        this.settings.set_strv("blocked-applications", updatedApps);
        this.blockedAppRows.delete(windowClass);
      } else {
        this.emptyRows.delete(row);
      }
      children._blockedAppsGroup.remove(row);
      this.updateAddButtonState();
    }
  }
);
var AboutPage = GObject.registerClass(
  {
    GTypeName: "TypescriptTemplateAboutPage",
    Template: getTemplate("AboutPage"),
    InternalChildren: [
      "extensionIcon",
      "extensionName",
      "extensionVersion",
      "linkWebsite",
      "linkIssues",
      "creditsRow",
      "legalRow",
      "extensionLicense"
    ]
  },
  class AboutPage2 extends Adw.PreferencesPage {
    setMetadata(metadata) {
      const children = this;
      const iconPath = GLib.uri_resolve_relative(
        import.meta.url,
        "assets/icons/vicinae-symbolic.svg",
        GLib.UriFlags.NONE
      );
      if (iconPath) {
        try {
          const iconFile = Gio.File.new_for_uri(iconPath);
          children._extensionIcon.set_from_file(iconFile.get_path());
        } catch (_error) {
          children._extensionIcon.set_from_icon_name(
            "application-x-executable"
          );
        }
      }
      children._extensionName.set_text(metadata.name);
      children._extensionVersion.set_text(
        `v${metadata["version-name"] || metadata.version}`
      );
      if (metadata.url) {
        children._linkWebsite.connect("clicked", () => {
          Gtk.show_uri(null, metadata.url || "", Gdk.CURRENT_TIME);
        });
        children._linkIssues.connect("clicked", () => {
          Gtk.show_uri(
            null,
            `${metadata.url}/issues`,
            Gdk.CURRENT_TIME
          );
        });
      } else {
        children._linkWebsite.visible = false;
        children._linkIssues.visible = false;
      }
      children._extensionLicense.buffer.set_text(LICENSE, -1);
    }
  }
);
var VicinaePrefs = class extends ExtensionPreferences {
  async fillPreferencesWindow(window) {
    const prefsWindow = window;
    prefsWindow._settings = this.getSettings();
    const generalPage = new GeneralPage();
    generalPage.bindSettings(prefsWindow._settings);
    prefsWindow.add(generalPage);
    const aboutPage = new AboutPage();
    aboutPage.setMetadata(this.metadata);
    prefsWindow.add(aboutPage);
  }
};
export {
  VicinaePrefs as default
};
