// src/utils/window-utils.ts
var getFocusedWindow = () => {
  const windowActors = global.get_window_actors();
  return windowActors.find((w) => w.meta_window.has_focus());
};
var getFocusedWindowApp = () => {
  const focusedWindow = getFocusedWindow();
  if (focusedWindow) {
    const wmClass = focusedWindow.meta_window.get_wm_class();
    const title = focusedWindow.meta_window.get_title();
    return wmClass || title || "unknown";
  }
  return "gnome-shell";
};

// src/utils/clipboard-utils.ts
function uint8ArrayToBase64(uint8Array) {
  try {
    const base64Chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
    let result = "";
    for (let i = 0; i < uint8Array.length; i += 3) {
      const byte1 = uint8Array[i] || 0;
      const byte2 = uint8Array[i + 1] || 0;
      const byte3 = uint8Array[i + 2] || 0;
      const chunk1 = byte1 >> 2 & 63;
      const chunk2 = (byte1 & 3) << 4 | byte2 >> 4 & 15;
      const chunk3 = (byte2 & 15) << 2 | byte3 >> 6 & 3;
      const chunk4 = byte3 & 63;
      result += base64Chars[chunk1] + base64Chars[chunk2] + base64Chars[chunk3] + base64Chars[chunk4];
    }
    const padding = 3 - uint8Array.length % 3;
    if (padding < 3) {
      result = result.slice(0, -padding) + "=".repeat(padding);
    }
    return result;
  } catch (error) {
    error("Error in direct base64 conversion", error);
    throw error;
  }
}
function uint8ArrayToBase64Fallback(uint8Array) {
  try {
    const chunkSize = 8192;
    let binaryString = "";
    for (let i = 0; i < uint8Array.length; i += chunkSize) {
      const chunk = uint8Array.slice(i, i + chunkSize);
      let chunkString = "";
      for (let j = 0; j < chunk.length; j++) {
        chunkString += String.fromCharCode(chunk[j]);
      }
      binaryString += chunkString;
    }
    const base64Result = btoa(binaryString);
    return base64Result;
  } catch (error) {
    error("Error in fallback base64 conversion", error);
    throw error;
  }
}
function bufferToBase64(buffer) {
  try {
    let uint8Array;
    if (buffer instanceof ArrayBuffer) {
      uint8Array = new Uint8Array(buffer);
    } else if (buffer instanceof Uint8Array) {
      uint8Array = buffer;
    } else if (buffer && typeof buffer === "object" && "length" in buffer && buffer.length > 0) {
      uint8Array = new Uint8Array(buffer.length);
      for (let i = 0; i < buffer.length; i++) {
        uint8Array[i] = buffer[i];
      }
    } else {
      return "";
    }
    try {
      return uint8ArrayToBase64(uint8Array);
    } catch (_directError) {
      try {
        return uint8ArrayToBase64Fallback(uint8Array);
      } catch (fallbackError) {
        if (uint8Array.length <= 1024) {
          try {
            let binaryString = "";
            for (let i = 0; i < uint8Array.length; i++) {
              binaryString += String.fromCharCode(uint8Array[i]);
            }
            return btoa(binaryString);
          } catch (_simpleError) {
          }
        }
        throw fallbackError;
      }
    }
  } catch (error) {
    error("Error converting buffer to base64", error);
    return "";
  }
}
function isValidImageBuffer(buffer) {
  if (!buffer) return false;
  let uint8Array;
  if (buffer instanceof ArrayBuffer) {
    if (buffer.byteLength < 8) return false;
    uint8Array = new Uint8Array(buffer);
  } else if (buffer instanceof Uint8Array) {
    if (buffer.length < 8) return false;
    uint8Array = buffer;
  } else if ("length" in buffer && buffer.length >= 8) {
    uint8Array = new Uint8Array(buffer.slice(0, 8));
  } else {
    return false;
  }
  if (uint8Array[0] === 137 && uint8Array[1] === 80 && uint8Array[2] === 78 && uint8Array[3] === 71) {
    return true;
  }
  if (uint8Array[0] === 255 && uint8Array[1] === 216 && uint8Array[2] === 255) {
    return true;
  }
  if (uint8Array[0] === 71 && uint8Array[1] === 73 && uint8Array[2] === 70 && uint8Array[3] === 56) {
    return true;
  }
  if (uint8Array[0] === 82 && uint8Array[1] === 73 && uint8Array[2] === 70 && uint8Array[3] === 70) {
    let webpCheck;
    if (buffer instanceof Uint8Array && buffer.length >= 12) {
      webpCheck = buffer.slice(8, 12);
    } else if (buffer instanceof ArrayBuffer && buffer.byteLength >= 12) {
      webpCheck = new Uint8Array(buffer.slice(8, 12));
    } else if ("length" in buffer && buffer.length >= 12) {
      webpCheck = new Uint8Array(buffer.slice(8, 12));
    } else {
      return false;
    }
    if (webpCheck[0] === 87 && webpCheck[1] === 69 && webpCheck[2] === 66 && webpCheck[3] === 80) {
      return true;
    }
  }
  return false;
}
function getImageMimeType(buffer) {
  if (!buffer) return "application/octet-stream";
  let uint8Array;
  if (buffer instanceof ArrayBuffer) {
    if (buffer.byteLength < 8) return "application/octet-stream";
    uint8Array = new Uint8Array(buffer.slice(0, 12));
  } else if (buffer instanceof Uint8Array) {
    if (buffer.length < 8) return "application/octet-stream";
    uint8Array = buffer.slice(0, 12);
  } else if ("length" in buffer && buffer.length >= 8) {
    uint8Array = new Uint8Array(buffer.slice(0, 12));
  } else {
    return "application/octet-stream";
  }
  if (uint8Array[0] === 137 && uint8Array[1] === 80 && uint8Array[2] === 78 && uint8Array[3] === 71) {
    return "image/png";
  }
  if (uint8Array[0] === 255 && uint8Array[1] === 216 && uint8Array[2] === 255) {
    return "image/jpeg";
  }
  if (uint8Array[0] === 71 && uint8Array[1] === 73 && uint8Array[2] === 70 && uint8Array[3] === 56) {
    return "image/gif";
  }
  if (uint8Array[0] === 82 && uint8Array[1] === 73 && uint8Array[2] === 70 && uint8Array[3] === 70) {
    let webpCheck;
    if (buffer instanceof Uint8Array && buffer.length >= 12) {
      webpCheck = buffer.slice(8, 12);
    } else if (buffer instanceof ArrayBuffer && buffer.byteLength >= 12) {
      webpCheck = new Uint8Array(buffer.slice(8, 12));
    } else if ("length" in buffer && buffer.length >= 12) {
      webpCheck = new Uint8Array(buffer.slice(8, 12));
    } else {
      return "application/octet-stream";
    }
    if (webpCheck[0] === 87 && webpCheck[1] === 69 && webpCheck[2] === 66 && webpCheck[3] === 80) {
      return "image/webp";
    }
  }
  return "application/octet-stream";
}
function simpleHash(str) {
  let hash = 0;
  if (str.length === 0) return hash.toString();
  for (let i = 0; i < str.length; i++) {
    const char = str.charCodeAt(i);
    hash = (hash << 5) - hash + char;
    hash = hash & hash;
  }
  return Math.abs(hash).toString(16);
}
function calculateClipboardMetadata(event) {
  const content = event.content;
  let mimeType = "text/plain";
  let contentType = "text";
  let contentHash = "";
  let size = 0;
  const sourceApp = getFocusedWindowApp();
  if (event.source === "image") {
    contentType = "image";
    if (content.startsWith("data:image/")) {
      const match = content.match(/^data:(image\/[^;]+);/);
      mimeType = match ? match[1] : "image/png";
    }
    contentHash = simpleHash(content);
    size = content.length;
  } else if (content.startsWith("data:")) {
    const match = content.match(/^data:([^;]+);/);
    mimeType = match ? match[1] : "application/octet-stream";
    contentType = "binary";
    contentHash = simpleHash(content);
    size = content.length;
  } else {
    mimeType = "text/plain";
    contentType = "text";
    contentHash = simpleHash(content);
    size = content.length;
    if (content.includes("<") && content.includes(">") && (content.includes("<html") || content.includes("<div") || content.includes("<p"))) {
      mimeType = "text/html";
      contentType = "html";
    }
    if (content.includes("/") && (content.includes(".") || content.includes("~"))) {
      contentType = "file";
    }
  }
  return {
    mimeType,
    contentType,
    contentHash,
    size,
    sourceApp
  };
}
export {
  bufferToBase64,
  calculateClipboardMetadata,
  getImageMimeType,
  isValidImageBuffer,
  simpleHash,
  uint8ArrayToBase64,
  uint8ArrayToBase64Fallback
};
