import { DayOfWeek, DevicesListType, DisplayMode, ResetSchedule, SettingKeys } from "./utils/Constants.js";
import { kRefreshInterval, kSchemaName } from "./utils/Constants.js";
import { compareJsonStrings } from "./utils/GenUtils.js";
import { getExtension } from "../extension.js";
/*
 * AppSettingsModel represents application setttings and user prefrences.
 */
export class AppSettingsModel {
    _logger;
    _schema;
    _settingsC;
    _settingListeners = [];
    _refreshInterval = kRefreshInterval;
    _displayMode = DisplayMode.DEFAULT;
    _devicesListType = DevicesListType.DEFAULT;
    _resetSchedule = ResetSchedule.DAILY;
    _resetDayOfWeek = DayOfWeek.MONDAY;
    _resetDayOfMonth = 1;
    _resetHours = 0;
    _resetMinutes = 0;
    _preferedDeviceName = undefined;
    _devicesInfoMap = {};
    _statusFontSize = 16;
    _displayBytes = true;
    _statusShowIcon = true;
    _resetAllStats = false;
    constructor(_logger) {
        this._logger = _logger;
    }
    init() {
        this.load();
        this._settingsC = this.settings.connect("changed", () => {
            // setting changed - get the new values
            this._logger.info("Prefrences/Settings value changed");
            this.load();
            this.notifyListerners();
        });
    }
    deinit() {
        if (this._settingsC !== undefined) {
            this.settings.disconnect(this._settingsC);
            this._settingsC = undefined;
        }
    }
    get settings() {
        if (!this._schema) {
            this._schema = getExtension().getSettings(kSchemaName);
        }
        return this._schema;
    }
    /**
     * Loads all the setting values
     */
    load() {
        this._refreshInterval = this.settings.get_int(SettingKeys.REFRESH_INTERVAL);
        this._displayMode = this.settings.get_string(SettingKeys.DISPLAY_MODE);
        this._devicesListType = this.settings.get_string(SettingKeys.DEVICES_LIST_TYPE);
        this._resetSchedule = this.settings.get_string(SettingKeys.RESET_SCHEDULE);
        this._resetDayOfWeek = this.settings.get_string(SettingKeys.RESET_WEEK_DAY);
        this._resetDayOfMonth = this.settings.get_int(SettingKeys.RESET_MONTH_DAY);
        this._resetHours = this.settings.get_int(SettingKeys.RESET_HOURS);
        this._resetMinutes = this.settings.get_int(SettingKeys.RESET_MINUTES);
        const str = this.settings.get_string(SettingKeys.DEVICES_INFO);
        this._devicesInfoMap = JSON.parse(str);
        this._preferedDeviceName = this.settings.get_string(SettingKeys.PREFERED_DEVICE);
        this._displayBytes = this.settings.get_boolean(SettingKeys.DISPLAY_BYTES);
        this._statusFontSize = this.settings.get_int(SettingKeys.STATUS_FONT_SIZE);
        this._statusShowIcon = this.settings.get_boolean(SettingKeys.STATUS_SHOW_ICON);
        this._resetAllStats = this.settings.get_boolean(SettingKeys.RESET_ALL_STATS);
        // this._logger.debug(`new values [ refreshInterval: ${this._refreshInterval} displayMode: ${this._displayMode} resetTime: ${this._resetHours} : ${this._resetMinutes}]`);
        // this._logger.debug(`deivicesInfoMap ${str}`);
    }
    /**
     * Save/Write all the setting values
     */
    save() {
        // write back the changed values.
        if (this.settings.get_string(SettingKeys.DISPLAY_MODE) !== this._displayMode) {
            this.settings.set_string(SettingKeys.DISPLAY_MODE, this._displayMode);
        }
        if (this.settings.get_string(SettingKeys.PREFERED_DEVICE) !== this._preferedDeviceName) {
            this.settings.set_string(SettingKeys.PREFERED_DEVICE, this._preferedDeviceName ?? "");
        }
        const devicesJson = JSON.stringify(this._devicesInfoMap);
        //this._logger.info("devicesInfoMap", devicesJson);
        if (!compareJsonStrings(this.settings.get_string(SettingKeys.DEVICES_INFO), devicesJson)) {
            this.settings.set_string(SettingKeys.DEVICES_INFO, devicesJson);
        }
        if (this.settings.get_boolean(SettingKeys.RESET_ALL_STATS) !== this._resetAllStats) {
            this.settings.set_boolean(SettingKeys.RESET_ALL_STATS, this._resetAllStats);
        }
    }
    get refreshInterval() {
        return this._refreshInterval || kRefreshInterval;
    }
    get displayMode() {
        return this._displayMode || DisplayMode.DEFAULT;
    }
    set displayMode(mode) {
        this._displayMode = mode;
        this.save();
    }
    get preferedDeviceName() {
        return this._preferedDeviceName;
    }
    set preferedDeviceName(deviceName) {
        this._preferedDeviceName = deviceName;
        this.save();
    }
    get devicesListType() {
        return this._devicesListType || DevicesListType.ALL;
    }
    get resetSchedule() {
        return this._resetSchedule;
    }
    get resetDayOfWeek() {
        return this._resetDayOfWeek;
    }
    get resetDayOfMonth() {
        return this._resetDayOfMonth;
    }
    get resetHours() {
        return this._resetHours;
    }
    get resetMinutes() {
        return this._resetMinutes;
    }
    get displayBytes() {
        return this._displayBytes;
    }
    get statusShowIcon() {
        return this._statusShowIcon;
    }
    get resetAllStats() {
        return this._resetAllStats;
    }
    clearResetAllStats() {
        this._resetAllStats = false;
        this.save();
    }
    getResetTime() {
        const date = new Date();
        date.setHours(this._resetHours);
        date.setMinutes(this._resetMinutes);
        date.setSeconds(0);
        return date;
    }
    getLastResetDateTime(deviceName) {
        const { resetedAt } = this.getDeviceInfo(deviceName);
        let lastResetedAt = undefined;
        if (resetedAt) {
            lastResetedAt = new Date(resetedAt);
        }
        return lastResetedAt;
    }
    get devicesInfoMap() {
        return this._devicesInfoMap;
    }
    set devicesInfoMap(info) {
        this._devicesInfoMap = { ...this._devicesInfoMap, ...info };
        this.save();
    }
    get statusFontSize() {
        return this._statusFontSize;
    }
    getDeviceInfo(name) {
        return this._devicesInfoMap[name] || {};
    }
    replaceDeviceInfo(name, info) {
        this._devicesInfoMap[name] = info;
        this.save();
    }
    updateDeviceInfo(name, info) {
        this._devicesInfoMap[name] = { ...this.devicesInfoMap[name], ...info };
        this.save();
    }
    // pub-sub
    notifyListerners() {
        for (const listener of this._settingListeners) {
            listener();
        }
    }
    subscribe(listener) {
        this._settingListeners.push(listener);
        return listener;
    }
    unsubscribe(listener) {
        const index = this._settingListeners.indexOf(listener);
        if (index != -1) {
            this._settingListeners.splice(index, 1);
        }
        return listener;
    }
}
