'use strict';
/* Gigabyte Laptop using dkms https://github.com/tangalbert919/gigabyte-laptop-wmi  */
import GLib from 'gi://GLib';
import GObject from 'gi://GObject';
import * as Helper from '../lib/helper.js';

const {exitCode, fileExists, readFileInt, runCommandCtl} = Helper;

const GIGABYTE_MODE = '/sys/devices/platform/gigabyte_laptop/charge_mode';
const GIGABYTE_LIMIT = '/sys/devices/platform/gigabyte_laptop/charge_limit';
const AORUS_MODE = '/sys/devices/platform/aorus_laptop/charge_mode';
const AORUS_LIMIT = '/sys/devices/platform/aorus_laptop/charge_limit';

export const GigabyteSingleBattery = GObject.registerClass({
    Signals: {'threshold-applied': {param_types: [GObject.TYPE_STRING]}},
}, class GigabyteSingleBattery extends GObject.Object {
    constructor(settings) {
        super();
        this.name = 'Gigabyte Laptop';
        this.type = 28;
        this.deviceNeedRootPermission = true;
        this.deviceHaveDualBattery = false;
        this.deviceHaveStartThreshold = false;
        this.deviceHaveVariableThreshold = true;
        this.deviceHaveBalancedMode = true;
        this.deviceHaveAdaptiveMode = false;
        this.deviceHaveExpressMode = false;
        this.deviceUsesModeNotValue = false;
        this.iconForFullCapMode = '100';
        this.iconForBalanceMode = '080';
        this.iconForMaxLifeMode = '060';
        this.endFullCapacityRangeMax = 100;
        this.endFullCapacityRangeMin = 80;
        this.endBalancedRangeMax = 95;
        this.endBalancedRangeMin = 65;
        this.endMaxLifeSpanRangeMax = 85;
        this.endMaxLifeSpanRangeMin = 60;
        this.incrementsStep = 1;
        this.incrementsPage = 5;

        this._settings = settings;
        this.ctlPath = null;
    }

    isAvailable() {
        const usesGIGA = fileExists(GIGABYTE_MODE) && fileExists(GIGABYTE_LIMIT);
        const usesAORUS = fileExists(AORUS_MODE) && fileExists(AORUS_LIMIT);

        if (usesGIGA || usesAORUS) {
            this._modPath = usesAORUS ? AORUS_MODE : GIGABYTE_MODE;
            this._limitPath = usesAORUS ? AORUS_LIMIT : GIGABYTE_LIMIT;

            this._cmd = usesAORUS ? 'AORUS_THRESHOLD' : 'GIGABYTE_THRESHOLD';
        } else {
            return false;
        }
        return true;
    }

    async setThresholdLimit(chargingMode) {
        this._updateMode = 'true';
        this._endValue = this._settings.get_int(`current-${chargingMode}-end-threshold`);

        if (this._verifyThreshold())
            return exitCode.SUCCESS;

        const [status] = await runCommandCtl(this.ctlPath, this._cmd, this._updateMode, `${this._endValue}`);
        if (status === exitCode.ERROR) {
            this.emit('threshold-applied', 'error');
            return exitCode.ERROR;
        } else if (status === exitCode.TIMEOUT) {
            this.emit('threshold-applied', 'timeout');
            return exitCode.ERROR;
        }

        if (this._verifyThreshold())
            return exitCode.SUCCESS;

        if (this._delayReadTimeoutId)
            GLib.source_remove(this._delayReadTimeoutId);
        this._delayReadTimeoutId = null;

        await new Promise(resolve => {
            this._delayReadTimeoutId = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 200, () => {
                resolve();
                this._delayReadTimeoutId = null;
                return GLib.SOURCE_REMOVE;
            });
        });

        if (this._verifyThreshold())
            return exitCode.SUCCESS;

        this.emit('threshold-applied', 'not-updated');
        return exitCode.ERROR;
    }

    _verifyThreshold() {
        if (readFileInt(this._modPath) === 1)
            this._updateMode = 'false';
        this.endLimitValue = readFileInt(this._limitPath);
        if (this._endValue === this.endLimitValue) {
            this.emit('threshold-applied', 'success');
            return true;
        }
        return false;
    }

    destroy() {
        if (this._delayReadTimeoutId)
            GLib.source_remove(this._delayReadTimeoutId);
        this._delayReadTimeoutId = null;
    }
});
