import St from 'gi://St';
import GLib from 'gi://GLib';
import GObject from 'gi://GObject';
import Clutter from 'gi://Clutter';
import Settings from '../helpers/settings.js';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import * as PopupMenu from 'resource:///org/gnome/shell/ui/popupMenu.js';

export default GObject.registerClass(
    class ClipboardManager extends PanelMenu.Button {
        _timer;
        _keybinds;

        _lastItem;
        _copyItems;
        _clipboard;

        constructor(keybinds) {
            super();

            this._lastItem = null;
            this._copyItems = [];

            this._keybinds = keybinds;
            this._keybinds.registerKeybind(Settings.KEY_CLIPBOARD, this.openClipboard.bind(this));

            this._clipboard = St.Clipboard.get_default();

            this._startMonitoring();

            Main.panel.addToStatusArea('window-tricks-clipboard', this, 0);
        }

        _init(extension) {
            super._init(0.0, null, false);

            this._icon = new St.Icon({ icon_name: 'edit-paste-symbolic', style_class: 'system-status-icon', });
            this.add_child(this._icon);

            this._setupMenu();
        }

        openClipboard() {
            this.menu.toggle();
        }

        _startMonitoring() {
            this._timer = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 500, () => {
                this._checkClipboard();
                return GLib.SOURCE_CONTINUE;
            });
        }

        _checkClipboard() {
            this._clipboard.get_text(St.ClipboardType.CLIPBOARD, (_, data) => {
                let text = data?.toString().trim();
                if (!text || text.length === 0 || text === this._lastItem) {
                    return;
                }

                this._lastItem = text;

                this._copyItems = [text, ...this._copyItems];
                this._copyItems = this._copyItems.slice(0, this._getClipboardMax());

                this._setupMenu();
            });
        }

        _setupMenu() {
            if (this._copyItems && this._copyItems.length > 0) {
                this._setupMenuItems();
            } else {
                this._setupMenuEmpty();
            }
        }

        _setupMenuEmpty() {
            this.menu.removeAll();

            let label = new St.Label({
                text: "Clipboard empty.",
                x_expand: true,
                x_align: Clutter.ActorAlign.CENTER
            });

            let menuItem = new PopupMenu.PopupMenuItem('');
            menuItem.actor.add_child(label);

            this.menu.addMenuItem(menuItem);
        }

        _setupMenuItems() {
            this.menu.removeAll();

            this._copyItems.forEach(item => {
                this._setupMenuItem(item);
            });
        }

        _setupMenuItem(item) {
            let label = new St.Label({
                text: item,
                x_expand: true
            });

            let button = new St.Button({
                x_align: Clutter.ActorAlign.END
            });
            button.connect('clicked', () => {
                this._copyItems = this._copyItems.filter(x => x !== item);
                this._setupMenu();
            });
            button.child = new St.Icon({
                icon_name: 'window-close-symbolic',
                style_class: 'popup-menu-icon',
            });

            let menuItem = new PopupMenu.PopupMenuItem('');
            menuItem.actor.add_child(label);
            menuItem.actor.add_child(button);

            menuItem.connect('activate', () => {
                this._clipboard.set_text(St.ClipboardType.CLIPBOARD, item);
                this.menu.close();
            });

            this.menu.addMenuItem(menuItem);
        }

        _getClipboardMax() {
            return Settings.tryGetInteger(Settings.CLIPBOARD_MAX) ?? 10;
        }

        destroy() {
            GLib.source_remove(this._timer);
            this._keybinds.removeKeybinding(Settings.KEY_CLIPBOARD);

            this._timer = null;
            this._keybinds = null;
            this._lastItem = null;
            this._copyItems = null;
            this._clipboard = null;
        }
    }
);


