import GObject from 'gi://GObject';
import Direction from '../enums/direction.js';
import Settings from '../helpers/settings.js';
import * as windowHelper from '../helpers/window.js';
import * as screenHelper from '../helpers/screen.js';

export default GObject.registerClass(
    class Snapper extends GObject.Object {
        _keybinds;

        constructor(keybinds) {
            super();

            this._keybinds = keybinds;
            this._keybinds.registerKeybind(Settings.KEY_SNAP_RIGHT, this.snapRight.bind(this));
            this._keybinds.registerKeybind(Settings.KEY_SNAP_LEFT, this.snapLeft.bind(this));
            this._keybinds.registerKeybind(Settings.KEY_SNAP_UP, this.snapUp.bind(this));
            this._keybinds.registerKeybind(Settings.KEY_SNAP_DOWN, this.snapDown.bind(this));
        }

        snapRight() {
            this._snap(Direction.Right);
        }

        snapLeft() {
            this._snap(Direction.Left);
        }

        snapUp() {
            this._snap(Direction.Up);
        }

        snapDown() {
            this._snap(Direction.Down);
        }

        _snap(direction) {
            let window = windowHelper.getFocusedWindow();
            if (!window) {
                return;
            }

            let windows = this._getNearBySnapableWindow(window, direction);
            if (windows.length === 0) {
                this._snapToScreenEdge(direction, window);
                return;
            }

            let currentWindowSize = window.size;
            let otherWindowSize = windows[0].size;

            let updatePosition = Direction.isVertical(direction)
                ? direction === Direction.Down
                    ? otherWindowSize.y - currentWindowSize.height
                    : otherWindowSize.y + otherWindowSize.height
                : direction === Direction.Right
                    ? otherWindowSize.x - currentWindowSize.width
                    : otherWindowSize.x + otherWindowSize.width;

            let screen = screenHelper.getScreenSize(window.workspace);

            let isVertical = Direction.isVertical(direction);
            let overflowX = !isVertical && currentWindowSize.width + updatePosition > screen.width;
            let overflowY = isVertical && currentWindowSize.height + updatePosition > screen.height;

            if (windows.length === 0 || updatePosition < 0 || overflowX || overflowY) {
                this._snapToScreenEdge(direction, window);
                return;
            }

            if (isVertical) {
                currentWindowSize.y = updatePosition
            } else {
                currentWindowSize.x = updatePosition
            }

            windowHelper.resizeWindow(window, currentWindowSize);
        }

        _snapToScreenEdge(direction, window) {
            let workspace = window.workspace;
            let screenSize = screenHelper.getScreenSize(workspace);

            if (Direction.isVertical(direction)) {
                window.size.y =
                    direction === Direction.Up ? screenSize.y : screenSize.y + screenSize.height - window.size.height;
            } else {
                window.size.x =
                    direction === Direction.Left ? screenSize.x : screenSize.x + screenSize.width - window.size.width;
            }

            windowHelper.resizeWindow(window, window.size);
        }


        _getNearBySnapableWindow(window, direction) {
            let windows = windowHelper.getNearbyWindows(window, direction);
            if (windows.length === 0) {
                return windows;
            }

            windows = this._filterSnappedWindows(window, windows);
            windows = this._filterPerpendicularWindows(window, direction, windows);
            windows = this._filterOverlappedWindows(window, direction, windows);

            return windows;
        }

        _filterSnappedWindows(window, windows) {
            return windows.filter((otherWindow) => {
                let touching = (
                    window.size.x - (otherWindow.size.x + otherWindow.size.width) === 0 ||
                    window.size.y - (otherWindow.size.y + otherWindow.size.height) === 0 ||
                    (window.size.x + window.size.width) - otherWindow.size.x === 0 ||
                    (window.size.y + window.size.height) - otherWindow.size.y === 0
                )
                return !touching;
            })
        }

        _filterPerpendicularWindows(window, direction, windows) {
            return windows.filter((otherWindow) => {
                let otherWindowMin = Direction.isVertical(direction) ? otherWindow.size.x : otherWindow.size.y;

                let otherWindowMax = Direction.isVertical(direction)
                    ? otherWindow.size.x + otherWindow.size.width
                    : otherWindow.size.y + otherWindow.size.height;

                let windowMin = Direction.isVertical(direction) ? window.size.x : window.size.y;

                let windowMax = Direction.isVertical(direction)
                    ? window.size.x + window.size.width
                    : window.size.y + window.size.height;

                return (
                    (windowMin < otherWindowMax && windowMax > otherWindowMin) ||
                    (otherWindowMin < windowMax && otherWindowMax > windowMin)
                );
            });
        }

        _filterOverlappedWindows(window, direction, windows) {
            return windows.filter((otherWindow) => {
                switch (direction) {
                    case Direction.Up:
                        return otherWindow.size.y < window.size.y;
                    case Direction.Down:
                        return otherWindow.size.y > window.size.y;
                    case Direction.Left:
                        return otherWindow.size.x < window.size.x;
                    case Direction.Right:
                        return otherWindow.size.x > window.size.x;
                }
                return false;
            });
        }

        destroy() {
            this._keybinds.removeKeybinding(Settings.KEY_SNAP_RIGHT);
            this._keybinds.removeKeybinding(Settings.KEY_SNAP_LEFT);
            this._keybinds.removeKeybinding(Settings.KEY_SNAP_UP);
            this._keybinds.removeKeybinding(Settings.KEY_SNAP_DOWN);

            this._keybinds = null;
        }
    }
);
