import Clutter from "gi://Clutter";
import Cogl from "gi://Cogl";
import GObject from "gi://GObject";
import St from "gi://St";
import { PopupAnimation } from "resource:///org/gnome/shell/ui/boxpointer.js";
import * as Main from "resource:///org/gnome/shell/ui/main.js";
import { current_extension_uuid, registerClass, set_style_value } from "./utils.js";
const QuickSettingsLayoutConstructor = Main.panel.statusArea.quickSettings.menu._grid.layout_manager.constructor;
// Base panel, reproducing gnome's QuickSettingsMenu
const BasePanel = registerClass(class BasePanel extends St.Widget {
    panel_id;
    is_destroyed;
    // Do not rename. Those are the same names that the ones used by QuickSettingsMenu
    _overlay;
    _grid;
    _dimEffect;
    _activeMenu;
    constructor(id, n_columns = 2, properties) {
        super({
            // Enable this so the menu block click events from propagating through
            reactive: true,
            ...properties
        });
        this.panel_id = `${current_extension_uuid()}/${id}`;
        // Overlay layer that will contain sub-menus
        this._overlay = new Clutter.Actor({ layout_manager: new Clutter.BinLayout() });
        // Placeholder to make empty space when opening a sub-menu
        const placeholder = new Clutter.Actor({
            // The placeholder have the same height as the overlay, which means
            // it have the same height as the opened sub-menu
            constraints: new Clutter.BindConstraint({
                coordinate: Clutter.BindCoordinate.HEIGHT,
                source: this._overlay,
            }),
        });
        this._grid = new St.Widget({
            style_class: "popup-menu-content quick-settings quick-settings-grid",
            layout_manager: new QuickSettingsLayoutConstructor(placeholder, { nColumns: n_columns }),
        });
        // Force the grid to take up all the available width
        this._grid.add_constraint(new Clutter.BindConstraint({
            coordinate: Clutter.BindCoordinate.WIDTH,
            source: this,
        }));
        this.add_child(this._grid);
        this._grid.add_child(placeholder);
        this._overlay.add_constraint(new Clutter.BindConstraint({
            coordinate: Clutter.BindCoordinate.WIDTH,
            source: this._grid,
        }));
        this.add_child(this._overlay);
        this._dimEffect = new Clutter.BrightnessContrastEffect({ enabled: false });
        this._grid.add_effect_with_name("dim", this._dimEffect);
        this.is_destroyed = true;
        this.connect("destroy", () => { this.is_destroyed = true; });
    }
    getItems() {
        // Every child except the placeholder
        return this._grid.get_children().filter(item => item != this._grid.layout_manager._overlay);
    }
    getFirstItem() {
        return this.getItems()[0];
    }
    addItem(item, column_span = 1) {
        this._grid.add_child(item);
        this._completeAddItem(item, column_span);
    }
    insertItemBefore(item, sibling, column_span = 1) {
        this._grid.insert_child_below(item, sibling);
        this._completeAddItem(item, column_span);
    }
    _completeAddItem(item, column_span) {
        this.setColumnSpan(item, column_span);
        if ("menu" in item && item.menu) {
            this._overlay.add_child(item.menu.actor);
            item.menu.connect_object("open-state-changed", (_, is_open) => {
                this._setDimmed(is_open);
                this._activeMenu = is_open ? item.menu : undefined;
                // The sub-popup for the power menu is too high.
                // I don't know if it's the real source of the issue, but I suspect that the constraint that fixes its y position
                // isn't accounting for the padding of the grid, so we add it to the offset manually
                // Later: I added the name check because it breaks on the audio panel
                // so I'm almost certain that this is not a proper fix
                if (is_open && this.getItems().indexOf(item) == 0 && this.panel_id.startsWith("main@gnome-shell/")) {
                    const constraint = item.menu.actor.get_constraints()[0];
                    constraint.offset =
                        // the offset is normally bound to the height of the source
                        constraint.source.height
                            + this._grid.get_theme_node().get_padding(St.Side.TOP);
                    // note: we don't reset this property when the item is removed from this panel because
                    // we hope that it will reset itself (because it's bound to the height of the source),
                    // which in the case in my tests, but maybe some issue will arise because of this
                }
                return false;
            }, this);
        }
        if ("_menuButton" in item && item._menuButton) {
            // @ts-expect-error: hack
            item._menuButton.__libpanel_y_expand_backup = item._menuButton.y_expand;
            item._menuButton.y_expand = false;
        }
    }
    removeItem(item) {
        if (!this._grid.get_children().includes(item))
            console.error(`[LibPanel] ${current_extension_uuid()} tried to remove an item not in the panel`);
        item.get_parent()?.remove_child(item);
        if ("menu" in item && item.menu) {
            item.menu.disconnect_object(this);
            item.menu.actor?.get_parent()?.remove_child(item.menu.actor);
        }
        if ("_menuButton" in item && item._menuButton) {
            // @ts-expect-error: hack
            item._menuButton.y_expand = item._menuButton.__libpanel_y_expand_backup;
            // @ts-expect-error: hack
            delete item._menuButton.__libpanel_y_expand_backup;
        }
    }
    getColumnSpan(item) {
        if (!this._grid.get_children().includes(item))
            console.error(`[LibPanel] ${current_extension_uuid()} tried to get the column span of an item not in the panel`);
        const value = new GObject.Value();
        this._grid.layout_manager.child_get_property(this._grid, item, "column-span", value);
        const column_span = value.get_int();
        value.unset();
        return column_span;
    }
    setColumnSpan(item, column_span) {
        if (!this._grid.get_children().includes(item))
            console.error(`[LibPanel] ${current_extension_uuid()} tried to set the column span of an item not in the panel`);
        this._grid.layout_manager.child_set_property(this._grid, item, "column-span", column_span);
    }
    close() {
        this._activeMenu?.close(PopupAnimation.NONE);
    }
    _setDimmed(dim) {
        // copied from https://gitlab.gnome.org/GNOME/gnome-shell/-/blob/main/js/ui/quickSettings.js
        const DIM_BRIGHTNESS = -0.4;
        const POPUP_ANIMATION_TIME = 400;
        const val = 127 * (1 + (dim ? 1 : 0) * DIM_BRIGHTNESS);
        const color = new Cogl.Color({
            red: val,
            green: val,
            blue: val,
            alpha: 255,
        });
        this._grid.ease_property("@effects.dim.brightness", color, {
            mode: Clutter.AnimationMode.LINEAR,
            duration: POPUP_ANIMATION_TIME,
            onStopped: () => (this._dimEffect.enabled = dim),
        });
        this._dimEffect.enabled = true;
    }
    set_padding(padding) {
        set_style_value(this._grid, "padding", padding !== null ? `${padding}px` : null);
    }
    set_row_spacing(row_spacing) {
        set_style_value(this._grid, "spacing-rows", row_spacing !== null ? `${row_spacing}px` : null);
    }
    set_column_spacing(column_spacing) {
        set_style_value(this._grid, "spacing-columns", column_spacing !== null ? `${column_spacing}px` : null);
    }
});
const DraggablePanel = registerClass(class DraggablePanel extends BasePanel {
});
const AutohidingPanel = registerClass(class AutohidingPanel extends DraggablePanel {
    constructor(id, n_columns = 2, properties) {
        super(id, n_columns, properties);
        this._grid.connect("child-added", (_, child) => {
            const handler_id = child.connect("notify::visible", () => this._update_visibility());
            // @ts-expect-error: hack
            child.__libpanel_handler_id = handler_id;
            this._update_visibility();
        });
        this._grid.connect("child-removed", (_, child) => {
            // The check is needed because the placeholder doesn't have the signal
            // @ts-expect-error: hack
            if (child.__libpanel_handler_id) {
                // @ts-expect-error: hack
                child.disconnect(child.__libpanel_handler_id);
                // @ts-expect-error: hack
                delete child.__libpanel_handler_id;
            }
            this._update_visibility();
        });
    }
    _update_visibility() {
        for (const child of this.getItems()) {
            if (child.visible) {
                this.show();
                return;
            }
        }
        this.hide();
    }
});
export default AutohidingPanel;
