// GNOME 46–48 — Eversolo Now Playing (UI + Last.fm + GSettings + toggle miniature)
import St from 'gi://St';
import Gio from 'gi://Gio';
import GLib from 'gi://GLib';
import GObject from 'gi://GObject';
import Soup from 'gi://Soup';
import Pango from 'gi://Pango';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import * as PopupMenu from 'resource:///org/gnome/shell/ui/popupMenu.js';
import { Extension } from 'resource:///org/gnome/shell/extensions/extension.js';
// === CONSTS ==================================================================
const REFRESH_SECS = 5;
const COVER_SIZE = 24;
const MENU_COVER_SIZE = 230;
const STATE_PATH = '/ZidooMusicControl/v2/getState';
const CMD_PREV = '/ZidooMusicControl/v2/playLast';
const CMD_TOGGLE = '/ZidooMusicControl/v2/playOrPause';
const CMD_NEXT = '/ZidooMusicControl/v2/playNext';
const LASTFM_URL = 'https://ws.audioscrobbler.com/2.0/';
const CACHE_DIR = GLib.build_filenamev([GLib.get_user_cache_dir(), 'eversolo-nowplaying']);
const COVER_FILE_MENU = GLib.build_filenamev([CACHE_DIR, 'cover_menu.jpg']);
// === UTILS ===================================================================
function ensureCacheDir() {
  try {
    const f = Gio.File.new_for_path(CACHE_DIR);
    if (!f.query_exists(null)) f.make_directory_with_parents(null);
  } catch (e) { log(`[Eversolo] couldn't create cache: ${e}`); }
}
function httpSession() {
  const s = new Soup.Session();
  s.timeout = REFRESH_SECS;
  return s;
}
async function httpGetJSON(session, url) {
  return new Promise((resolve) => {
    const msg = Soup.Message.new('GET', url);
    session.send_and_read_async(msg, GLib.PRIORITY_DEFAULT, null, (sess, res) => {
      try {
        const bytes = sess.send_and_read_finish(res);
        const data = new TextDecoder('utf-8').decode(bytes.get_data());
        try { resolve(JSON.parse(data)); } catch { resolve(null); }
      } catch { resolve(null); }
    });
  });
}
async function httpGetToFile(session, url, destPath) {
  return new Promise((resolve) => {
    const msg = Soup.Message.new('GET', url);
    session.send_and_read_async(msg, GLib.PRIORITY_DEFAULT, null, (sess, res) => {
      try {
        const bytes = sess.send_and_read_finish(res);
        const data = bytes.get_data();
        if (!data?.length) return resolve(false);
        Gio.File.new_for_path(destPath)
          .replace_contents(data, null, false, Gio.FileCreateFlags.REPLACE_DESTINATION, null);
        resolve(true);
      } catch (e) { log(`[Eversolo] Error saving cover: ${e}`); resolve(false); }
    });
  });
}
function formatInfo(o) {
  const parts = [];
  if (o.quality) parts.push(String(o.quality));
  if (o.bits) parts.push(String(o.bits).replace(/-bit$/, ''));
  if (o.samplerate) parts.push(String(o.samplerate));
  return parts.join(' – ');
}
function parseState(json) {
  if (!json) return null;
  const roonActive = !!(json?.everSoloPlayInfo?.everSoloPlayAudioInfo?.songName);
  if (roonActive) {
    const i = json.everSoloPlayInfo.everSoloPlayAudioInfo;
    const o = json.everSoloPlayOutputInfo || {};
    return {
      title: i.songName || 'Unknown',
      artist: i.artistName || '',
      album: i.albumName || '',
      cover_url: i.albumUrl || '',
      quality: o.outPutDecodec || 'PCM',
      bits: i.audioBitsPerSample > 0 ? `${i.audioBitsPerSample}` : '',
      samplerate: i.audioSampleRate > 0 ? `${Math.floor(i.audioSampleRate / 1000)} kHz` : '',
    };
  }
  const p = json.playingMusic || {};
  return {
    title: p.title || 'Unknown',
    artist: p.artist || '',
    album: p.album || '',
    cover_url: p.albumArt || '',
    quality: p.audioQuality || '',
    bits: p.bits || '',
    samplerate: p.sampleRate || '',
  };
}
function setTextureFromPath(bin, path, size) {
  try {
    const child = bin.get_child && bin.get_child();
    if (child) child.destroy();
    const uri = Gio.File.new_for_path(path).get_uri();
    const bust = `${uri}?t=${Date.now()}`;
    bin.set_style(
      `background-image: url("${bust}");` +
      `background-size: cover; background-position: center;` +
      `width: ${size}px; height: ${size}px; border-radius: ${Math.round(size*0.15)}px;`
    );
  } catch (e) { log(`[Eversolo] Error showing cover: ${e}`); }
}
function actorAlive(a) { return a && !a.destroyed && a.get_stage?.(); }
const norm = s => (s || '').toLowerCase().trim();
function enableWrap(label, maxPx) {
  if (!label) return;
  const ct = label.get_clutter_text?.();
  if (ct) {
    if (typeof ct.set_single_line_mode === 'function') ct.set_single_line_mode(false);
    else if (typeof ct.set_single_line === 'function') ct.set_single_line(false);
    if (typeof ct.set_line_wrap === 'function') ct.set_line_wrap(true);
    if (typeof ct.set_line_wrap_mode === 'function') ct.set_line_wrap_mode(Pango.WrapMode.WORD_CHAR);
    if (typeof ct.set_ellipsize === 'function') ct.set_ellipsize(Pango.EllipsizeMode.NONE);
  }
  label.set_style(`max-width: ${maxPx}px;`);
}
// === INDICATOR ===============================================================
const Indicator = GObject.registerClass(
class Indicator extends PanelMenu.Button {
  _init(settings, lfmCacheRef) {
    super._init(0.0, 'Eversolo Now Playing');
    this.name = 'eversolo-nowplaying';
    this._alive = true;
    this._lastCoverUrl = '';
    this._settings = settings;
    this._lfmCache = lfmCacheRef;
    this._session = httpSession();
    this._timeout = 0;
    this._postClickId = 0;
    this._lastPlaying = false;
    ensureCacheDir();
    this.add_style_class_name('eversolo-button');
    this._panelIcon = null;
    this._coverBin = null;
    this._rebuildPanelActor();
    // Tooltip con wrap
    this._tooltip = new St.Label({ text: 'Eversolo', style_class: 'eversolo-tooltip' });
    enableWrap(this._tooltip, 480);
    this._tooltip.hide();
    Main.layoutManager.addTopChrome(this._tooltip);
    this._enterId = this.connect('enter-event', () => this._showTooltip());
    this._leaveId = this.connect('leave-event', () => this._hideTooltip());
    // Info en menú (con wrap)
    this._title = new St.Label({ text: '—', style_class: 'eversolo-title', x_expand: true });
    this._artist = new St.Label({ text: '', style_class: 'eversolo-artist', x_expand: true });
    this._info = new St.Label({ text: '', style_class: 'eversolo-info', x_expand: true });
    const MENU_TEXT_MAX_W = 300;
    enableWrap(this._title, MENU_TEXT_MAX_W);
    enableWrap(this._artist, MENU_TEXT_MAX_W);
    enableWrap(this._info, MENU_TEXT_MAX_W);
    this._menuCover = new St.Bin({ width: MENU_COVER_SIZE, height: MENU_COVER_SIZE, style_class: 'eversolo-cover-menu' });
    const headerBox = new St.BoxLayout({ vertical: true, style_class: 'eversolo-header', x_expand: true, style: `max-width: ${MENU_TEXT_MAX_W}px;` });
    headerBox.add_child(this._title);
    headerBox.add_child(this._artist);
    headerBox.add_child(this._info);
    const topRow = new St.BoxLayout({ x_expand: true });
    topRow.add_child(this._menuCover);
    topRow.add_child(new St.Widget({ width: 12 }));
    topRow.add_child(headerBox);
    const topItem = new PopupMenu.PopupBaseMenuItem({ reactive: false, can_focus: false });
    topItem.add_child(topRow);
    this.menu.addMenuItem(topItem);
    this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
    const ctrls = new St.BoxLayout({ style_class: 'eversolo-ctrls' });
    ctrls.add_child(this._makeCtrlButton('media-skip-backward-symbolic', () => this._sendCmd(CMD_PREV)));
    ctrls.add_child(this._makeCtrlButton('media-playback-start-symbolic', () => this._sendCmd(CMD_TOGGLE)));
    ctrls.add_child(this._makeCtrlButton('media-skip-forward-symbolic', () => this._sendCmd(CMD_NEXT)));
    const ctrlItem = new PopupMenu.PopupBaseMenuItem({ reactive: false, can_focus: false });
    ctrlItem.add_child(ctrls);
    this.menu.addMenuItem(ctrlItem);
    this._startRefreshTimer();
    this._refresh().catch(e => log(`[Eversolo] initial refresh error: ${e}`));
  }
  _getBase() { return (this._settings?.get_string('eversolo-base') || '').trim(); }
  _getPanelCover() { return !!(this._settings?.get_boolean('panel-cover')); }
  _getLfmKey() { return (this._settings?.get_string('lastfm-apikey') || '').trim(); }
  _rebuildPanelActor() {
    if (this._panelIcon && !this._panelIcon.destroyed) this._panelIcon.destroy();
    if (this._coverBin && !this._coverBin.destroyed) this._coverBin.destroy();
    if (this._getPanelCover()) {
      this._coverBin = new St.Bin({ style_class: 'eversolo-cover-bin', width: COVER_SIZE, height: COVER_SIZE });
      this.add_child(this._coverBin);
    } else {
      this._panelIcon = new St.Icon({ icon_name: 'media-playback-start-symbolic', icon_size: 16 });
      this.add_child(this._panelIcon);
    }
    this._updatePlayPauseIcons(this._lastPlaying);
  }
  _applySettings() {
    this._rebuildPanelActor();
    this._startRefreshTimer();
  }
  _updatePlayPauseIcons(playing) {
    const iconName = playing ? 'media-playback-pause-symbolic' : 'media-playback-start-symbolic';
    if (this._btnToggleIcon) this._btnToggleIcon.icon_name = iconName;
    if (!this._getPanelCover() && this._panelIcon && !this._panelIcon.destroyed)
      this._panelIcon.icon_name = iconName;
  }
  _startRefreshTimer() {
    if (this._timeout) {
      GLib.source_remove(this._timeout);
      this._timeout = 0;
    }
    this._timeout = GLib.timeout_add_seconds(GLib.PRIORITY_DEFAULT, REFRESH_SECS, () => {
      if (!this._alive) return GLib.SOURCE_REMOVE;
      this._refresh().catch(e => log(`[Eversolo] refresh error: ${e}`));
      return GLib.SOURCE_CONTINUE;
    });
  }
  _makeCtrlButton(iconName, cb) {
    const btn = new St.Button({ style_class: 'eversolo-btn' });
    const icon = new St.Icon({ icon_name: iconName, icon_size: 18 });
    btn.set_child(icon);
    if (iconName === 'media-playback-start-symbolic')
      this._btnToggleIcon = icon;
    btn.connect('clicked', () => { if (this._alive) cb(); });
    return btn;
  }
  _positionTooltip() {
    if (!actorAlive(this) || !actorAlive(this._tooltip)) return;
    const [x, y] = this.get_transformed_position();
    const [w, h] = this.get_transformed_size();
    this._tooltip.set_position(Math.round(x), Math.round(y + h + 6));
  }
  _showTooltip() { if (actorAlive(this._tooltip)) { this._positionTooltip(); this._tooltip.show(); } }
  _hideTooltip() { if (actorAlive(this._tooltip)) this._tooltip.hide(); }
  async _sendCmd(path) {
    const base = this._getBase();
    if (!base) return;
    const msg = Soup.Message.new('GET', base + path);
    this._session.send_and_read_async(msg, GLib.PRIORITY_DEFAULT, null, () => {});
  }
  async _getLastFmCover(artist, track) {
    const key = this._getLfmKey();
    if (!key || !artist || !track) return null;
    const ck = `${norm(artist)}::${norm(track)}`;
    if (this._lfmCache[ck] !== undefined) return this._lfmCache[ck];
    const params = new URLSearchParams({ method: 'track.getInfo', api_key: key, artist, track, format: 'json' });
    const url = `${LASTFM_URL}?${params.toString()}`;
    const json = await httpGetJSON(this._session, url);
    const imgs = json?.track?.album?.image || [];
    const found = imgs.length ? imgs[imgs.length - 1]['#text'] : null;
    this._lfmCache[ck] = found;
    return found;
  }
  async _refresh() {
    if (!this._alive) return;
    try {
      const base = this._getBase();
      if (!base) {
        if (actorAlive(this._title)) this._title.text = 'Eversolo';
        if (actorAlive(this._artist)) this._artist.text = 'Configure IP in Preferences';
        if (actorAlive(this._info)) this._info.text = '';
        this._lastPlaying = false;
        this._updatePlayPauseIcons(false);
        return;
      }
      const json = await httpGetJSON(this._session, base + STATE_PATH);
      if (!this._alive) return;
      // === playing? ==========================================================
      let playing = false;
      const ps = json?.everSoloPlayInfo?.playStatus;
      if (ps === 2) playing = true;
      if (ps === 3 || ps === 1) playing = false;
      if (json?.state === 4) playing = true;
      const pos = json?.everSoloPlayInfo?.currentPosition ?? json?.position;
      if (typeof pos === 'number') {
        if (typeof this._lastPos === 'number') {
          const delta = pos - this._lastPos;
          if (delta > 200) playing = true;
          else if (delta <= 0) playing = false;
        }
        this._lastPos = pos;
      }
      const state = parseState(json);
      const safeSet = (label, text) => { if (actorAlive(label)) label.text = text; };
      if (!state) {
        safeSet(this._title, 'Eversolo');
        safeSet(this._artist, '— no signal —');
        safeSet(this._info, '');
        this._lastPlaying = false;
        this._updatePlayPauseIcons(false);
        return;
      }
      safeSet(this._title, state.title || '—');
      safeSet(this._artist, state.artist || '');
      safeSet(this._info, formatInfo(state));
      if (actorAlive(this._tooltip))
        this._tooltip.text = `${state.title || ''} — ${state.artist || ''}`.trim();
      // sincroniza ▶/⏸
      this._lastPlaying = !!playing;
      this._updatePlayPauseIcons(this._lastPlaying);
      // portada
      let coverUrl = state.cover_url;
      // Fix for Roon: if cover_url is local cache path, construct device API URL to fetch it
      if (coverUrl && coverUrl.startsWith('/storage/emulated/0/.zcontrol/cache/phonecontrol/')) {
        const filename = GLib.basename(coverUrl);
        coverUrl = `${base}/SystemSettings/getItemSettingIcon?iconName=${filename}&type=1`;
      }
      if (coverUrl && /^https?:\/\//.test(coverUrl)) {
        // Valid HTTP/HTTPS URL from device API
      } else {
        coverUrl = null;
      }
      if (!coverUrl) coverUrl = await this._getLastFmCover(state.artist, state.title);
      if (!this._alive) return;
      if (coverUrl && coverUrl !== this._lastCoverUrl) {
        const ok = await httpGetToFile(this._session, coverUrl, COVER_FILE_MENU);
        if (ok && this._alive) {
          this._lastCoverUrl = coverUrl;
          setTextureFromPath(this._menuCover, COVER_FILE_MENU, MENU_COVER_SIZE);
          if (this._getPanelCover() && this._coverBin && !this._coverBin.destroyed)
            setTextureFromPath(this._coverBin, COVER_FILE_MENU, COVER_SIZE);
        }
      }
    } catch (e) {
      log(`[Eversolo] _refresh exception: ${e}`);
    }
  }
  destroy() {
    this._alive = false;
    if (this._timeout) { GLib.source_remove(this._timeout); this._timeout = 0; }
    if (this._enterId) { this.disconnect(this._enterId); this._enterId = 0; }
    if (this._leaveId) { this.disconnect(this._leaveId); this._leaveId = 0; }
    if (this._tooltip) { this._tooltip.destroy(); this._tooltip = null; }
    try { this._session?.abort(); } catch {}
    super.destroy();
  }
});
// === ENTRYPOINT ==============================================================
export default class EversoloExtension extends Extension {
  enable() {
    this._settings = this.getSettings();
    this._lfmCache = {};
    ensureCacheDir();
    this._indicator = new Indicator(this._settings, this._lfmCache);
    this._settingsChangedId = this._settings.connect('changed', () => {
      this._indicator?._applySettings();
      this._indicator?._refresh?.();
    });
    Main.panel.addToStatusArea('eversolo-nowplaying', this._indicator, 0, 'right');
  }
  disable() {
    if (this._settingsChangedId) {
      this._settings.disconnect(this._settingsChangedId);
      this._settingsChangedId = 0;
    }
    if (this._indicator) {
      this._indicator.destroy();
      this._indicator = null;
    }
    this._settings = null;
    this._lfmCache = null;
  }
}
