import Gio from 'gi://Gio';
import Meta from 'gi://Meta';
import Shell from 'gi://Shell';
import { Extension } from 'resource:///org/gnome/shell/extensions/extension.js';
import { wm } from 'resource:///org/gnome/shell/ui/main.js';
import { E as ExtensionSettings } from './shared/extension-settings.js';

class InterfaceSettings {
  #settings;
  #originalShouldAnimate;
  #timeout;
  constructor() {
    this.#settings = new Gio.Settings({ schema_id: 'org.gnome.desktop.interface' });
    this.#originalShouldAnimate = wm._shouldAnimate;
  }
  /**
   * From https://extensions.gnome.org/extension/119/disable-window-animations/
   */
  maybeSuppressAnimation(shouldSuppress, callback) {
    const windowManager = wm;
    if (
      shouldSuppress &&
      typeof windowManager._shouldAnimate === 'function' &&
      this.#settings.get_boolean('enable-animations')
    ) {
      if (this.#timeout) {
        clearTimeout(this.#timeout);
      }
      windowManager._shouldAnimate = () => false;
      callback();
      this.#timeout = setTimeout(() => {
        windowManager._shouldAnimate = this.#originalShouldAnimate;
      }, 100);
    } else {
      callback();
    }
  }
}
class HotkeyListener {
  #actionMap;
  #appSystem;
  #appSystemHandlerId;
  #displayHandlerId;
  #extension;
  #interface;
  #settings;
  #settingsHandlerId;
  #windowTracker;
  constructor(settings) {
    this.#appSystem = Shell.AppSystem.get_default();
    this.#extension = new ExtensionSettings(settings);
    this.#interface = new InterfaceSettings();
    this.#settings = settings;
    this.#windowTracker = Shell.WindowTracker.get_default();
    this.#init();
  }
  #isAppMatch(app, appConfig) {
    return app ? app.get_name() === appConfig.name : false;
  }
  #getFirstWindowByAppConfig(appConfig) {
    for (const windowActor of global.get_window_actors()) {
      const window = windowActor.get_meta_window();
      if (!window || window.is_override_redirect()) {
        continue;
      }
      if (this.#isAppMatch(this.#windowTracker.get_window_app(window), appConfig)) {
        return window;
      }
    }
    return void 0;
  }
  #getFocusedWindow(app) {
    if (app) {
      return app.get_windows().find((window) => {
        return window.has_focus();
      });
    }
    return void 0;
  }
  #maybeMoveOrHideWindow(appConfig) {
    const app = this.#windowTracker.get_focus_app();
    if (!this.#isAppMatch(app, appConfig)) {
      return false;
    }
    const window = this.#getFocusedWindow(app);
    if (!window) {
      return false;
    }
    this.#interface.maybeSuppressAnimation(this.#extension.shouldSkipAnimations, () => {
      const activeMonitorIdx = global.display.get_current_monitor();
      if (window.get_monitor() !== activeMonitorIdx) {
        window.move_to_monitor(activeMonitorIdx);
      } else if (window.can_minimize()) {
        window.minimize();
      }
    });
    return true;
  }
  #maybeShowWindow(appConfig) {
    const window = this.#getFirstWindowByAppConfig(appConfig);
    if (!window) {
      return false;
    }
    this.#interface.maybeSuppressAnimation(this.#extension.shouldSkipAnimations, () => {
      if (!window.get_workspace().active) {
        window.change_workspace(global.get_workspace_manager().get_active_workspace());
      }
      const activeMonitorIdx = global.display.get_current_monitor();
      if (window.get_monitor() !== activeMonitorIdx) {
        window.move_to_monitor(activeMonitorIdx);
      }
      window.activate(global.get_current_time());
      window.focus(global.get_current_time());
    });
    return true;
  }
  #maybeLaunchWindow(appConfig) {
    const appInfo = Gio.DesktopAppInfo.new(appConfig.id);
    return appInfo ? appInfo.launch([], null) : false;
  }
  #onAcceleratorActivated(actionId) {
    const appConfig = this.#actionMap?.get(actionId);
    if (appConfig) {
      void (
        this.#maybeMoveOrHideWindow(appConfig) ||
        this.#maybeShowWindow(appConfig) ||
        this.#maybeLaunchWindow(appConfig)
      );
    }
  }
  #registerAppHotkey(appConfig) {
    if (!appConfig.hotkey) {
      return;
    }
    const actionId = global.display.grab_accelerator(appConfig.hotkey, Meta.KeyBindingFlags.NONE);
    switch (actionId) {
      case Meta.KeyBindingAction.NONE: {
        break;
      }
      default: {
        const bindingName = Meta.external_binding_name_for_action(actionId);
        wm.allowKeybinding(bindingName, Shell.ActionMode.NORMAL);
        this.#actionMap?.set(actionId, { ...appConfig, bindingName });
        break;
      }
    }
  }
  #init() {
    const reinitialise = () => {
      this.dispose();
      this.#init();
    };
    this.#actionMap = /* @__PURE__ */ new Map();
    this.#appSystemHandlerId = this.#appSystem.connect('installed-changed', reinitialise);
    this.#displayHandlerId = global.display.connect('accelerator-activated', (_, actionId) => {
      this.#onAcceleratorActivated(actionId);
    });
    this.#settingsHandlerId = this.#settings.connect(`changed::${'configs'}`, reinitialise);
    this.#extension.getAppConfigs().forEach((appConfig) => {
      this.#registerAppHotkey(appConfig);
    });
  }
  dispose() {
    if (this.#actionMap) {
      this.#actionMap.forEach(({ bindingName }, actionId) => {
        wm.removeKeybinding(bindingName);
        global.display.ungrab_accelerator(actionId);
      });
      this.#actionMap.clear();
      this.#actionMap = void 0;
    }
    if (typeof this.#appSystemHandlerId === 'number') {
      this.#appSystem.disconnect(this.#appSystemHandlerId);
      this.#appSystemHandlerId = void 0;
    }
    if (typeof this.#displayHandlerId === 'number') {
      global.display.disconnect(this.#displayHandlerId);
      this.#displayHandlerId = void 0;
    }
    if (typeof this.#settingsHandlerId === 'number') {
      this.#settings.disconnect(this.#settingsHandlerId);
      this.#settingsHandlerId = void 0;
    }
  }
}
class ApplicationHotkeysExtension extends Extension {
  #hotkeyListener;
  enable() {
    this.#hotkeyListener = new HotkeyListener(this.getSettings());
  }
  disable() {
    this.#hotkeyListener?.dispose();
    this.#hotkeyListener = void 0;
  }
}
export { ApplicationHotkeysExtension as default };
