import GnomeBluetooth from "gi://GnomeBluetooth";
import { Extension, gettext as _, } from "resource:///org/gnome/shell/extensions/extension.js";
import { notifyError, panel } from "resource:///org/gnome/shell/ui/main.js";
import { LogType, devicesObjectToArray, logIfEnabled, sendSignal, supportedDeviceNames, updateLogging, } from "./common.js";
// Internal
import NoiseclapperIndicator from "./indicator.js";
// ----------------------- Extension -----------------------
export default class NoiseclapperExtension extends Extension {
    settings;
    bluetoothClient;
    indicator;
    settingsHandler;
    enable() {
        logIfEnabled(LogType.Info, "Enabling Noiseclapper...");
        // We enable the bluetooth client
        logIfEnabled(LogType.Debug, "Enabling Bluetooth client...");
        this.bluetoothClient = new GnomeBluetooth.Client();
        // And create the indicator
        logIfEnabled(LogType.Debug, "Creating and adding Noiseclapper indicator...");
        this.indicator = new NoiseclapperIndicator(this);
        panel.addToStatusArea(this.uuid, this.indicator);
        // Apply settings and position
        this.settings = this.getSettings();
        this.settingsHandler = this.settings.connect("changed", this.applySettings.bind(this));
        this.applySettings();
        logIfEnabled(LogType.Info, "Startup successful.");
    }
    disable() {
        logIfEnabled(LogType.Info, "Disabling Noiseclapper...");
        // Disable Bluetooth client if enabled
        this.bluetoothClient = undefined;
        // Remove the indicator
        logIfEnabled(LogType.Debug, "Removing Noiseclapper indicator...");
        this.indicator?.destroy();
        this.indicator = undefined;
        // Disconnect settings change handler
        if (this.settingsHandler !== undefined) {
            this.settings.disconnect(this.settingsHandler);
            this.settingsHandler = undefined;
        }
        this.settings = undefined;
    }
    signalHandler(signal) {
        logIfEnabled(LogType.Debug, `Preparing to send signal : [${signal}]`);
        const devices = devicesObjectToArray(this.bluetoothClient.get_devices());
        let hasFoundAtLeastOneDevice = false;
        for (const device of devices) {
            if (device.connected && supportedDeviceNames.includes(device.name)) {
                hasFoundAtLeastOneDevice = true;
                const { name, address } = device;
                logIfEnabled(LogType.Info, `Sending signal [${signal}] to device [${name}] with MAC address [${address}]`);
                sendSignal(signal, address).catch((error) => {
                    logIfEnabled(LogType.Error, `Failed to send signal: ${error}`);
                });
            }
        }
        if (!hasFoundAtLeastOneDevice) {
            logIfEnabled(LogType.Error, "No compatible devices found.");
            notifyError(`Noiseclapper - ${_("Error")}`, _("No connected compatible devices found."));
        }
    }
    applySettings() {
        logIfEnabled(LogType.Debug, "Applying settings...");
        updateLogging(this.settings.get_boolean("logging-enabled"));
        this.indicator.applyPosition();
    }
}
