import Gio from 'gi://Gio';
import { appInfoToAppConfig } from '../utils/apps.js';

class ExtensionSettings {
  #settings;

  constructor(settings) {
    this.#settings = settings;
  }

  get shouldSkipAnimations() {
    return this.#settings.get_boolean('skip-animations');
  }

  get serializedAppConfigs() {
    return this.#settings.get_strv('configs');
  }

  writeSerializedAppConfigs(serializedAppConfigs) {
    return this.#settings.set_strv('configs', serializedAppConfigs);
  }

  serializeAppConfig({ id, hotkey }) {
    return JSON.stringify([id, hotkey]);
  }

  deserializeAppConfig(serializedAppConfig) {
    try {
      const [appId, hotkey] = JSON.parse(serializedAppConfig);

      if (appId && typeof appId === 'string' && typeof hotkey === 'string') {
        return [appId, hotkey];
      }
    } catch {}

    return void 0;
  }

  getConfiguredAppIds() {
    return this.serializedAppConfigs.reduce((accumulator, serializedAppConfig) => {
      const [appId] = this.deserializeAppConfig(serializedAppConfig) ?? [];

      return appId ? [...accumulator, appId] : accumulator;
    }, []);
  }

  getAppConfigs() {
    return this.serializedAppConfigs.reduce((appConfigs, serializedAppConfig) => {
      const [appId, hotkey] = this.deserializeAppConfig(serializedAppConfig) ?? [];

      if (!appId) {
        return appConfigs;
      }

      const appInfo =
        /**
         * There might be a case where an appId cannot be matched with a desktop
         * file while executing this. This can happen if an app was uninstalled or
         * if the desktop file isn't available right now (e.g. for distrobox
         * apps). Don't filter them out here, if a user wants to remove an app,
         * they can do so from the extension settings.
         */
        Gio.DesktopAppInfo.new(appId);

      appConfigs.push(appInfoToAppConfig({ appId, appInfo, hotkey }));

      return appConfigs;
    }, []);
  }

  addAppConfig(appConfig) {
    return this.writeSerializedAppConfigs([
      ...this.serializedAppConfigs,
      this.serializeAppConfig(appConfig),
    ]);
  }
  removeAppConfigByAppId = (targetAppId) => {
    const serializedAppConfigs = [...this.serializedAppConfigs];

    const idx = serializedAppConfigs.findIndex((serializedAppConfig) => {
      const [currentAppId] = this.deserializeAppConfig(serializedAppConfig) ?? [];

      return currentAppId && currentAppId === targetAppId;
    });

    if (idx > -1) {
      serializedAppConfigs.splice(idx, 1);

      return this.writeSerializedAppConfigs(serializedAppConfigs);
    }

    return false;
  };

  updateAppConfigById(appId, properties) {
    const appConfigs = this.getAppConfigs();

    for (const appConfig of appConfigs) {
      if (appConfig.id !== appId) {
        continue;
      }

      if (typeof properties.hotkey === 'string') {
        appConfig.hotkey = properties.hotkey;

        return this.writeSerializedAppConfigs(
          appConfigs.map((appConfig2) => {
            return this.serializeAppConfig(appConfig2);
          }),
        );
      }

      break;
    }

    return false;
  }

  setShouldSkipAnimations(shouldSkipAnimations) {
    return this.#settings.set_boolean('skip-animations', shouldSkipAnimations);
  }
}

export { ExtensionSettings as default };
