import Gio from 'gi://Gio';
import Meta from 'gi://Meta';
import Shell from 'gi://Shell';
import { wm } from 'resource:///org/gnome/shell/ui/main.js';
import { isAppMatch, getFocusedWindowByApp, findWindowByAppConfig } from './apps.js';
import ExtensionUtils from './extension.js';
import InterfaceUtils from './interface.js';

class HotkeyListener {
  #appSystem;
  #extension;
  #interface;
  #settings;
  #windowTracker;
  #actionMap;
  #appSystemHandlerId;
  #displayHandlerId;
  #settingsHandlerId;

  constructor(settings) {
    this.#appSystem = Shell.AppSystem.get_default();
    this.#extension = new ExtensionUtils(settings);
    this.#interface = new InterfaceUtils();
    this.#settings = settings;
    this.#windowTracker = Shell.WindowTracker.get_default();
    this.#init();
  }

  #maybeMoveOrHideWindow(appConfig) {
    const app = this.#windowTracker.get_focus_app();

    if (!isAppMatch(app, appConfig)) {
      return false;
    }

    const window = getFocusedWindowByApp(app);

    if (!window) {
      return false;
    }

    this.#interface.maybeSuppressAnimation(this.#extension.shouldSkipAnimations, () => {
      const activeMonitorIdx = global.display.get_current_monitor();

      if (window.get_monitor() !== activeMonitorIdx) {
        window.move_to_monitor(activeMonitorIdx);
      } else if (window.can_minimize()) {
        window.minimize();
      }
    });

    return true;
  }

  #maybeShowWindow(appConfig) {
    const window = findWindowByAppConfig(this.#windowTracker, appConfig);

    if (!window) {
      return false;
    }

    this.#interface.maybeSuppressAnimation(this.#extension.shouldSkipAnimations, () => {
      if (!window.get_workspace().active) {
        window.change_workspace(global.get_workspace_manager().get_active_workspace());
      }

      const activeMonitorIdx = global.display.get_current_monitor();

      if (window.get_monitor() !== activeMonitorIdx) {
        window.move_to_monitor(activeMonitorIdx);
      }

      window.activate(global.get_current_time());
      window.focus(global.get_current_time());
    });

    return true;
  }

  #maybeLaunchWindow(appConfig) {
    const appInfo = Gio.DesktopAppInfo.new(appConfig.id);

    return appInfo ? appInfo.launch([], null) : false;
  }

  #onAcceleratorActivated(actionId) {
    const appConfig = this.#actionMap?.get(actionId);

    if (appConfig) {
      void (
        this.#maybeMoveOrHideWindow(appConfig) ||
        this.#maybeShowWindow(appConfig) ||
        this.#maybeLaunchWindow(appConfig)
      );
    }
  }

  /**
   * Inspiration taken from p2t2p's answer:
   * https://superuser.com/a/1182899
   */
  #registerAppHotkey(appConfig) {
    if (!appConfig.hotkey) {
      return;
    }

    const actionId = global.display.grab_accelerator(appConfig.hotkey, Meta.KeyBindingFlags.NONE);

    switch (actionId) {
      case Meta.KeyBindingAction.NONE: {
        break;
      }

      default: {
        const bindingName = Meta.external_binding_name_for_action(actionId);

        wm.allowKeybinding(bindingName, Shell.ActionMode.NORMAL);
        this.#actionMap?.set(actionId, { ...appConfig, bindingName });
        break;
      }
    }
  }

  #init() {
    const reinitialise = () => {
      this.#unregisterListeners();
      this.#init();
    };

    this.#actionMap = /* @__PURE__ */ new Map();
    this.#appSystemHandlerId = this.#appSystem.connect('installed-changed', reinitialise);

    this.#displayHandlerId = global.display.connect('accelerator-activated', (_, actionId) => {
      this.#onAcceleratorActivated(actionId);
    });

    this.#settingsHandlerId = this.#settings.connect(`changed::${'configs'}`, reinitialise);

    this.#extension.getAppConfigs().forEach((appConfig) => {
      this.#registerAppHotkey(appConfig);
    });
  }

  #unregisterListeners() {
    if (this.#actionMap) {
      this.#actionMap.forEach(({ bindingName }, actionId) => {
        wm.removeKeybinding(bindingName);
        global.display.ungrab_accelerator(actionId);
      });

      this.#actionMap.clear();
    }

    if (typeof this.#appSystemHandlerId === 'number') {
      this.#appSystem.disconnect(this.#appSystemHandlerId);
    }

    if (typeof this.#displayHandlerId === 'number') {
      global.display.disconnect(this.#displayHandlerId);
    }

    if (typeof this.#settingsHandlerId === 'number') {
      this.#settings.disconnect(this.#settingsHandlerId);
    }

    this.#actionMap = void 0;
    this.#appSystemHandlerId = void 0;
    this.#displayHandlerId = void 0;
    this.#settingsHandlerId = void 0;
  }

  dispose() {
    this.#unregisterListeners();
    this.#interface.dispose();
  }
}

export { HotkeyListener as default };
