// src/extension.ts
import * as Main from "resource:///org/gnome/shell/ui/main.js";

// src/utils.ts
import Gio from "gi://Gio";
function toError(thing) {
  return thing instanceof Error ? thing : new Error(`${thing}`);
}
async function exec(argv, input = null, cancellable = null) {
  let flags = Gio.SubprocessFlags.STDOUT_PIPE;
  if (input !== null) flags |= Gio.SubprocessFlags.STDIN_PIPE;
  const proc = new Gio.Subprocess({
    argv,
    flags
  });
  try {
    proc.init(cancellable);
  } catch (error) {
    return { error: toError(error), output: null };
  }
  return await new Promise((resolve) => {
    proc.communicate_utf8_async(input, cancellable, (proc2, res) => {
      try {
        const output = proc2.communicate_utf8_finish(res)[1];
        const status = proc2.get_exit_status();
        if (status === 0) {
          resolve({ output, error: null });
        } else {
          resolve({
            output: null,
            error: new Error(
              `Command failed with status ${status}. Output: "${output}"`
            )
          });
        }
      } catch (e) {
        resolve({
          output: null,
          error: toError(e)
        });
      }
    });
  });
}
function titleCase(str) {
  return str.charAt(0).toUpperCase() + str.slice(1);
}

// src/extension.ts
import GObject from "gi://GObject";
import Clutter from "gi://Clutter";
import GLib from "gi://GLib";
import Gio2 from "gi://Gio";
import St from "gi://St";
import {
  PopupMenuSection,
  PopupMenuItem,
  Ornament
} from "resource:///org/gnome/shell/ui/popupMenu.js";
import {
  QuickMenuToggle,
  SystemIndicator
} from "resource:///org/gnome/shell/ui/quickSettings.js";
import {
  Extension,
  gettext as _
} from "resource:///org/gnome/shell/extensions/extension.js";
var QuickSettingsMenu = GObject.registerClass(
  class QuickSettingsMenu2 extends QuickMenuToggle {
    constructor(_extension) {
      super({
        title: _("Fan Control"),
        toggleMode: true,
        gicon: Gio2.icon_new_for_string(
          `${_extension.path}/icons/fw-fanctrl-revived.svg`
        )
      });
      this._extension = _extension;
      this.menu.setHeader(
        Gio2.icon_new_for_string(
          `${_extension.path}/icons/fw-fanctrl-revived.svg`
        ),
        _("Framework Fan Control")
      );
      this.connect("clicked", async () => {
        const newState = this.checked ? "resume" : "pause";
        const { error } = await exec(["fw-fanctrl", newState]);
        if (error) {
          return this._extension?.logger.error(
            `Error switching to ${newState}d state`,
            error
          );
        }
        await this._extension?._sync();
      });
      const settingsButton = new St.Button({
        style_class: "fw-fctrl-preferences-button",
        y_align: Clutter.ActorAlign.CENTER,
        x_align: Clutter.ActorAlign.END,
        x_expand: true,
        child: new St.Icon({
          icon_name: "applications-system-symbolic",
          icon_size: 14
        })
      });
      this.menu.addHeaderSuffix(settingsButton);
      settingsButton.connect("clicked", () => {
        this._extension?.openPreferences();
      });
      this.connect("destroy", () => {
        this._section?.destroy();
        this._section = null;
        settingsButton.destroy();
        this._items.clear();
      });
    }
    _section = null;
    _items = /* @__PURE__ */ new Map();
    async _setup(extension) {
      const { error, output } = await exec([
        "fw-fanctrl",
        "--output-format=JSON",
        "print",
        "list"
      ]);
      if (error) {
        return this._extension.logger.error(
          "Error fetching fan control strategies",
          error
        );
      }
      const { strategies } = JSON.parse(output);
      this._section = new PopupMenuSection();
      for (const strategy of strategies) {
        const item = new PopupMenuItem(titleCase(strategy));
        this._section?.addMenuItem(item);
        this._items?.set(strategy, item);
        item.connect("activate", async () => {
          const { error: error2 } = await exec([
            "fw-fanctrl",
            "use",
            strategy
          ]);
          if (error2) {
            return this._extension?.logger.error(
              `Error switching to ${strategy} strategy`,
              error2
            );
          }
          await extension._sync();
        });
      }
      this.menu.addMenuItem(this._section);
    }
    _setHeaderSubtitle(text) {
      this.menu._headerSubtitle.set({
        visible: true,
        text
      });
    }
    _setFailed() {
      this._setHeaderSubtitle("fw-fanctrl failed, please check logs");
      this.subtitle = "fw-fanctrl failed";
    }
    _setFanState(state) {
      this._setHeaderSubtitle(
        `Temperature: ${state.temperature}\xB0C Fan Speed ${state.speed}%`
      );
      this.subtitle = `${state.temperature}\xB0C ${state.speed}%`;
      this.set_checked(state.active);
      for (const [itemStrategy, menuItem] of this._items.entries()) {
        menuItem.setOrnament(
          itemStrategy === state.strategy ? Ornament.CHECK : Ornament.NONE
        );
      }
    }
  }
);
var FwFanCtrl = class extends Extension {
  _indicator = null;
  _menu = null;
  _sourceId = null;
  _settings = null;
  logger;
  constructor(metadata) {
    super(metadata);
    if ("getLogger" in this) {
      this.logger = this.getLogger();
    } else {
      this.logger = console;
    }
  }
  enable() {
    this._settings = this.getSettings();
    this._menu = new QuickSettingsMenu(this);
    this._indicator = new SystemIndicator();
    this._indicator.quickSettingsItems.push(this._menu);
    Main.panel.statusArea.quickSettings.addExternalIndicator(
      // @ts-expect-error who knows
      this._indicator
    );
    this._createLoop(this._settings.get_int("refresh-interval"));
    this._settings.connect(
      "changed::refresh-interval",
      (settings, _key) => {
        this._createLoop(settings.get_int("refresh-interval"));
      }
    );
    this._sync();
  }
  disable() {
    this._indicator?.destroy();
    this._indicator = null;
    this._menu?.destroy();
    this._menu = null;
    this._settings = null;
    if (this._sourceId) {
      GLib.Source.remove(this._sourceId);
      this._sourceId = null;
    }
  }
  _createLoop(interval) {
    if (this._sourceId) {
      GLib.Source.remove(this._sourceId);
      this._sourceId = null;
    }
    this._sourceId = GLib.timeout_add_seconds(
      GLib.PRIORITY_DEFAULT,
      interval,
      () => {
        this._sync();
        return GLib.SOURCE_CONTINUE;
      }
    );
  }
  async _sync() {
    if (this._menu && this._menu._items.size == 0) {
      await this._menu._setup(this);
    }
    const { error, output } = await exec([
      "fw-fanctrl",
      "--output-format=JSON",
      "print",
      "all"
    ]);
    if (error) {
      this._menu?._setFailed();
      return this.logger.error("Error checking state", error);
    }
    const state = JSON.parse(output);
    this._menu?._setFanState(state);
  }
};
export {
  FwFanCtrl as default
};
