// src/extension.ts
import * as Main from "resource:///org/gnome/shell/ui/main.js";

// src/utils.ts
import Gio from "gi://Gio";
function toError(thing) {
  return thing instanceof Error ? thing : new Error(`${thing}`);
}
async function exec(argv, input = null, cancellable = null) {
  let flags = Gio.SubprocessFlags.STDOUT_PIPE;
  if (input !== null) flags |= Gio.SubprocessFlags.STDIN_PIPE;
  const proc = new Gio.Subprocess({
    argv,
    flags
  });
  try {
    proc.init(cancellable);
  } catch (error) {
    return { error: toError(error), output: null };
  }
  return await new Promise((resolve) => {
    proc.communicate_utf8_async(input, cancellable, (proc2, res) => {
      try {
        const output = proc2.communicate_utf8_finish(res)[1];
        const status = proc2.get_exit_status();
        if (status === 0) {
          resolve({ output, error: null });
        } else {
          resolve({
            output: null,
            error: new Error(
              `Command failed with status ${status}. Output: "${output}"`
            )
          });
        }
      } catch (e) {
        resolve({
          output: null,
          error: toError(e)
        });
      }
    });
  });
}

// src/extension.ts
import GObject from "gi://GObject";
import GLib from "gi://GLib";
import Gio2 from "gi://Gio";
import {
  QuickMenuToggle,
  SystemIndicator
} from "resource:///org/gnome/shell/ui/quickSettings.js";
import {
  Extension,
  gettext as _
} from "resource:///org/gnome/shell/extensions/extension.js";
var QuickSettingsMenu = GObject.registerClass(
  class QuickSettingsMenu2 extends QuickMenuToggle {
    constructor(_extension) {
      super({
        title: _("ULauncher"),
        toggleMode: true,
        menuEnabled: false,
        gicon: Gio2.icon_new_for_string(
          `${_extension.path}/icons/ulauncher.svg`
        )
      });
      this._extension = _extension;
      this.connect("clicked", async () => {
        const newState = this.checked ? "start" : "stop";
        const { error } = await exec([
          "systemctl",
          "--user",
          newState,
          "--output=json",
          "ulauncher.service"
        ]);
        if (error) {
          return this._extension?.logger.error(
            `Error switching to ${newState}d state`,
            error
          );
        }
        await this._extension?._sync();
      });
    }
  }
);
var UlauncherToggle = class extends Extension {
  _indicator = null;
  _menu = null;
  _sourceId = null;
  logger;
  constructor(metadata) {
    super(metadata);
    this.logger = this.getLogger();
  }
  enable() {
    this._menu = new QuickSettingsMenu(this);
    this._indicator = new SystemIndicator();
    this._indicator.quickSettingsItems.push(this._menu);
    Main.panel.statusArea.quickSettings.addExternalIndicator(
      // @ts-expect-error who knows
      this._indicator
    );
    this._sourceId = GLib.timeout_add_seconds(
      GLib.PRIORITY_DEFAULT,
      5,
      () => {
        this._sync();
        return GLib.SOURCE_CONTINUE;
      }
    );
    this._sync();
  }
  disable() {
    this._indicator?.destroy();
    this._indicator = null;
    this._menu?.destroy();
    this._menu = null;
    if (this._sourceId) {
      GLib.Source.remove(this._sourceId);
      this._sourceId = null;
    }
  }
  async _sync() {
    const { output, error } = await exec([
      "systemctl",
      "--user",
      "list-units",
      "--output=json",
      "ulauncher.service"
    ]);
    if (error) {
      this.logger.error("Error getting ULauncher status:", error);
      return null;
    }
    const units = JSON.parse(output);
    if (units.length > 1) {
      const units2 = JSON.parse(output);
      this.logger.error("Unexpected number of units:", units2);
      return null;
    }
    const active = units.length ? units[0].active === "active" : false;
    this._menu.set_checked(active);
  }
};
export {
  UlauncherToggle as default
};
