import Gio from 'gi://Gio';
import { appInfoToAppConfig } from './apps.js';

class ExtensionUtils {
  #settings;
  #appConfigs;
  #configuredAppIds;
  #serializedAppConfigs;
  #settingsHandlerIds;
  #shouldDisableAnimations;

  constructor(settings) {
    this.#settings = settings;

    this.#settingsHandlerIds = [
      settings.connect(`changed::${'configs'}`, () => {
        this.#appConfigs = void 0;
        this.#configuredAppIds = void 0;
        this.#serializedAppConfigs = void 0;
      }),
      settings.connect(`changed::${'disable-animations'}`, () => {
        this.#shouldDisableAnimations = void 0;
      }),
    ];
  }

  get appConfigs() {
    return (this.#appConfigs ??= this.#getAppConfigs(this.serializedAppConfigs));
  }

  get configuredAppIds() {
    return (this.#configuredAppIds ??= this.#getConfiguredAppIds(this.serializedAppConfigs));
  }

  get serializedAppConfigs() {
    return (this.#serializedAppConfigs ??= this.#settings.get_strv('configs'));
  }

  get shouldDisableAnimations() {
    if (typeof this.#shouldDisableAnimations === 'undefined') {
      this.#shouldDisableAnimations = this.#settings.get_boolean('disable-animations');
    }

    return this.#shouldDisableAnimations;
  }

  #getAppConfigs(serializedAppConfigs) {
    return serializedAppConfigs.reduce((appConfigs, serializedAppConfig) => {
      const [appId, hotkey] = this.deserializeAppConfig(serializedAppConfig) ?? [];

      if (!appId) {
        return appConfigs;
      }

      const appInfo =
        /**
         * There might be a case where an appId cannot be matched with a desktop
         * file while executing this. This can happen if an app was uninstalled or
         * if the desktop file isn't available right now (e.g. for distrobox
         * apps). Don't filter them out here, if a user wants to remove an app,
         * they can do so from the extension settings.
         */
        Gio.DesktopAppInfo.new(appId);

      appConfigs.push(appInfoToAppConfig({ appId, appInfo, hotkey }));

      return appConfigs;
    }, []);
  }

  #getConfiguredAppIds(serializedAppConfigs) {
    return serializedAppConfigs.reduce((accumulator, serializedAppConfig) => {
      const [appId] = this.deserializeAppConfig(serializedAppConfig) ?? [];

      return appId ? [...accumulator, appId] : accumulator;
    }, []);
  }

  writeSerializedAppConfigs(serializedAppConfigs) {
    return this.#settings.set_strv('configs', serializedAppConfigs);
  }

  serializeAppConfig({ id, hotkey }) {
    return JSON.stringify([id, hotkey]);
  }

  deserializeAppConfig(serializedAppConfig) {
    try {
      const [appId, hotkey] = JSON.parse(serializedAppConfig);

      if (appId && typeof appId === 'string' && typeof hotkey === 'string') {
        return [appId, hotkey];
      }
    } catch {}

    return void 0;
  }

  getFilteredApps() {
    return Gio.AppInfo.get_all().filter((appInfo) => {
      if (appInfo.should_show()) {
        const appId = appInfo.get_id();

        if (appId) {
          return !this.configuredAppIds.includes(appId);
        }
      }

      return false;
    });
  }

  addAppConfig(appConfig) {
    return this.writeSerializedAppConfigs([
      ...this.serializedAppConfigs,
      this.serializeAppConfig(appConfig),
    ]);
  }

  removeAppConfigByAppId(targetAppId) {
    const serializedAppConfigs = [...this.serializedAppConfigs];

    const idx = serializedAppConfigs.findIndex((serializedAppConfig) => {
      const [currentAppId] = this.deserializeAppConfig(serializedAppConfig) ?? [];

      return currentAppId && currentAppId === targetAppId;
    });

    if (idx > -1) {
      serializedAppConfigs.splice(idx, 1);

      return this.writeSerializedAppConfigs(serializedAppConfigs);
    }

    return false;
  }

  updateAppConfigById(appId, properties) {
    for (const appConfig of this.appConfigs) {
      if (appConfig.id !== appId) {
        continue;
      }

      if (typeof properties.hotkey === 'string') {
        appConfig.hotkey = properties.hotkey;

        return this.writeSerializedAppConfigs(
          this.appConfigs.map((appConfig2) => {
            return this.serializeAppConfig(appConfig2);
          }),
        );
      }

      break;
    }

    return false;
  }

  writeShouldSkipAnimations(shouldSkipAnimations) {
    return this.#settings.set_boolean('disable-animations', shouldSkipAnimations);
  }

  dispose() {
    if (this.#settingsHandlerIds) {
      this.#settingsHandlerIds.forEach((handlerId) => {
        this.#settings.disconnect(handlerId);
      });
    }

    this.#appConfigs = void 0;
    this.#configuredAppIds = void 0;
    this.#serializedAppConfigs = void 0;
    this.#settingsHandlerIds = void 0;
    this.#shouldDisableAnimations = void 0;
  }
}

export { ExtensionUtils as default };
