import Gio from 'gi://Gio';
import { noop } from './noop.js';
import { wm } from './window-manager.js';

class InterfaceUtils {
  #settings;
  #isAnimationsEnabled;
  #originalShouldAnimate;
  #settingsHandlerIds;
  #timeout;

  constructor() {
    this.#originalShouldAnimate = wm._shouldAnimate;

    this.#settingsHandlerIds = [
      this.settings.connect(`changed::${'enable-animations'}`, () => {
        this.#isAnimationsEnabled = void 0;
      }),
    ];
  }

  get isAnimationsEnabled() {
    if (typeof this.#isAnimationsEnabled === 'undefined') {
      this.#isAnimationsEnabled = this.settings.get_boolean('enable-animations');
    }

    return this.#isAnimationsEnabled;
  }

  get settings() {
    return (this.#settings ??= new Gio.Settings({ schema_id: 'org.gnome.desktop.interface' }));
  }

  #clearTimeout() {
    if (typeof this.#timeout !== 'undefined') {
      clearTimeout(this.#timeout);
      this.#timeout = void 0;
    }
  }

  #overrideShouldAnimate() {
    wm._shouldAnimate = noop;
  }

  #restoreShouldAnimate() {
    wm._shouldAnimate = this.#originalShouldAnimate;
  }

  /**
   * From https://extensions.gnome.org/extension/119/disable-window-animations/
   */
  maybeSuppressAnimation(shouldSuppress, callback) {
    if (!shouldSuppress || !this.isAnimationsEnabled || typeof wm._shouldAnimate !== 'function') {
      callback();

      return;
    }

    this.#clearTimeout();
    this.#overrideShouldAnimate();
    callback();

    this.#timeout = setTimeout(() => {
      this.#restoreShouldAnimate();
    }, 100);
  }

  dispose() {
    this.#clearTimeout();
    this.#restoreShouldAnimate();

    if (this.#settingsHandlerIds) {
      this.#settingsHandlerIds.forEach((handlerId) => {
        this.settings.disconnect(handlerId);
      });
    }

    this.#settings = void 0;
    this.#originalShouldAnimate = void 0;
    this.#settingsHandlerIds = void 0;
    this.#isAnimationsEnabled = void 0;
    this.#timeout = void 0;
  }
}

export { InterfaceUtils as default };
