import Atk from "gi://Atk";
import Clutter from "gi://Clutter";
import Gio from "gi://Gio";
import GLib from "gi://GLib";
import GObject from "gi://GObject";
import St from "gi://St";

import {
  Extension,
  gettext as _,
} from "resource:///org/gnome/shell/extensions/extension.js";

// import * as DND from 'resource:///org/gnome/shell/ui/dnd.js';
import * as Main from "resource:///org/gnome/shell/ui/main.js";
import * as PanelMenu from "resource:///org/gnome/shell/ui/panelMenu.js";
import * as PopupMenu from "resource:///org/gnome/shell/ui/popupMenu.js";

async function getContent(pipe) {
  const decoder = new TextDecoder("utf-8");

  return new Promise(async (resolve, reject) => {
    try {
      await pipe.read_bytes_async(GLib.MAXINT16, 0, null, async (_, eres) => {
        try {
          const data = await pipe.read_bytes_finish(eres);
          resolve(decoder.decode(data.get_data()));
        } catch (e) {
          console.error("log: error in read_bytes_finish", e);
          reject(e);
        }
      });
    } catch (e) {
      console.error("log: error in read_bytes_finish", e);
      reject(e);
    }
  });
}

/**
 * @param {string[]} cmdline
 * @param {any} _env
 * @param {(stdout, stderr, exit_status)=>boolean} callback
 * @returns
 */
function spawn(command, env, callback) {
  const envFilter = ["SUBSHELL", "EXPAND_HOME"];
  let cmdline = [...command];

  if (env.SUBSHELL) {
    cmdline = ["/bin/bash", "-c", ...cmdline];
  }

  if (env.EXPAND_HOME) {
    cmdline = cmdline
      .map((x) => x.replace("$HOME", GLib.get_home_dir()))
      .map((x) => x.replace("~", GLib.get_home_dir()));
  }
  console.debug("spawn", cmdline.join(" "), " with env ", JSON.stringify(env));

  const flags =
    Gio.SubprocessFlags.STDERR_PIPE | Gio.SubprocessFlags.STDOUT_PIPE;
  let launcher = new Gio.SubprocessLauncher(callback ? { flags } : {});
  Object.keys(env)
    .filter((key) => !envFilter.indexOf(key) < 0)
    .map((key) => [key, env[key]])
    .forEach(([key, value]) => launcher.setenv(key, env[key], true));

  let process = launcher.spawnv(cmdline);
  if (callback) {
    process.wait_async(null, async () => {
      const stdout_pipe = process.get_stdout_pipe();
      const stdout_content = await getContent(stdout_pipe);

      const stderr_pipe = process.get_stderr_pipe();
      const stderr_content = await getContent(stderr_pipe);

      callback(stdout_content, stderr_content, process.get_exit_status());

      stderr_pipe.close(null);
      stdout_pipe.close(null);
    });
  }

  return process.get_identifier();
}

function run(command, env, wait = false) {
  if (!wait) {
    return new Promise((resolve) => {
      const id = spawn(command, env);
      resolve({ command: command.join(" "), id, stdout: "", stderr: "" });
    });
  } else {
    return new Promise((resolve) => {
      spawn(command, env, (stdout, stderr, exit_status) => {
        resolve({ command: command.join(" "), exit_status, stdout, stderr });
      });
    });
  }
}

class ApplicationsMenu extends PopupMenu.PopupMenu {
  constructor(sourceActor, arrowAlignment, arrowSide, button) {
    super(sourceActor, arrowAlignment, arrowSide);
    this._button = button;
  }
}

class ApplicationsButton extends PanelMenu.Button {
  static {
    GObject.registerClass(this);
  }

  constructor(menuConfig) {
    super(1.0, null, false);

    this.setMenu(new ApplicationsMenu(this, 1.0, St.Side.TOP, this));
    Main.panel.menuManager.addMenu(this.menu);

    // At this moment applications menu is not keyboard navigable at
    // all (so not accessible), so it doesn't make sense to set as
    // role ATK_ROLE_MENU like other elements of the panel.
    this.accessible_role = Atk.Role.LABEL;

    this._label = new St.Label({
      text: _(menuConfig.label),
      y_expand: true,
      y_align: Clutter.ActorAlign.CENTER,
    });

    this.add_child(this._label);
    this.name = "panelApplications";
    this.label_actor = this._label;

    Main.overview.connectObject(
      "showing",
      () => this.add_accessible_state(Atk.StateType.CHECKED),
      "hiding",
      () => this.remove_accessible_state(Atk.StateType.CHECKED),
      this
    );

    this.addMenuItems(this, menuConfig.entries, "-");
  }

  addMenuItems(menu, entries, d) {
    try {
      for (const entry of entries) {
        if (entry.type == "command") {
          this.addMenuItem(menu, entry, d + "-");
        } else if (entry.type == "toggle") {
          this.addToggleItem(menu, entry, d + "-");
        } else if (entry.type == "submenu") {
          this.addSubMenu(menu, entry, d + "-");
        }
      }
    } catch (e) {
      console.error("log: error in menu item creation", e);
    }
  }

  //? https://github.com/andreabenini/gnome-plugin.custom-menu-panel/blob/main/custom-menu-panel%40AndreaBenini/config.js
  addToggleItem(main, entry, d) {
    run(entry.commands.state, entry.env ?? {}, true)
      .then((result) => {
        let item;
        let state = result.exit_status == 0;
        item = main.menu.addAction(entry.label, () => {
          run(
            state ? entry.commands.off : entry.commands.on,
            entry.env ?? {},
            entry.catchExitCode ?? false
          )
            .then(() =>
              item.setOrnament(
                (state = !state)
                  ? PopupMenu.Ornament.CHECK
                  : PopupMenu.Ornament.NONE
              )
            )
            .then((result) => console.debug("log: toggle state result", result))
            .catch((err) => console.error("log: toggle state error", err));
        });
        item.setOrnament(
          state ? PopupMenu.Ornament.CHECK : PopupMenu.Ornament.NONE
        );
      })
      .catch((err) => console.error("log: toggle state error", err));
  }

  addMenuItem(main, entry, d) {
    main.menu.addAction(entry.label, () => {
      run(entry.command, entry.env ?? {}, entry.catchExitCode ?? false).then(
        (result) => console.debug("log: toggle state result", result)
      );
    });
  }

  addSubMenu(main, entry, d) {
    const subMenu = new PopupMenu.PopupSubMenuMenuItem(entry.label);
    subMenu.label.get_clutter_text().set_use_markup(true);
    main.menu.addMenuItem(subMenu, d);
    subMenu.menu.actor.keepMenuOpen = true;
    this.addMenuItems(subMenu, entry.entries, d);

    subMenu.menu.connect("open-state-changed", (menu, value) => {
      if (value) {
        if (main.setSubmenuShown) main.setSubmenuShown(true);
      }
    });

    main.menu.connect("menu-closed", (menu, value) => {
      if (!value) {
        subMenu.setSubmenuShown(false);
      }
    });
  }

  _onDestroy() {
    super._onDestroy();
  }
}

function loadConfiguration() {
  const filename = GLib.build_filenamev([
    GLib.get_user_config_dir(),
    "custom-menu.json",
  ]);
  if (!GLib.file_test(filename, GLib.FileTest.EXISTS)) {
    console.debug(
      "log: no config file found at ",
      filename,
      GLib.file_test(filename, GLib.FileTest.EXISTS)
    );
    return {
      label: `create ${filename}`,
      entries: [],
    };
  }

  const [ok, inhaltRaw, tag] = GLib.file_get_contents(filename);
  if (ok) {
    const decoder = new TextDecoder("utf-8");
    const inhalt = decoder.decode(inhaltRaw);
    return JSON.parse(inhalt);
  } else {
    console.error("log: error in loading config", tag);
    return {
      label: "error",
      entries: [],
    };
  }
}

export default class CustomMenuExtension extends Extension {
  enable() {
    console.debug("log: hi there");
    this.appsMenuButton = new ApplicationsButton(loadConfiguration());
    const index = Main.sessionMode.panel.left.indexOf("activities") + 1;
    Main.panel.addToStatusArea(
      "fiurthorn-custom-menu",
      this.appsMenuButton,
      index,
      "left"
    );
  }

  disable() {
    console.debug("log: bye there");
    Main.panel.menuManager.removeMenu(this.appsMenuButton.menu);
    this.appsMenuButton.destroy();
    delete this.appsMenuButton;
  }
}
