// src/extension.ts
import St4 from "gi://St";
import Gio4 from "gi://Gio";
import GLib3 from "gi://GLib";
import GObject3 from "gi://GObject";
import * as Main2 from "resource:///org/gnome/shell/ui/main.js";
import * as PanelMenu from "resource:///org/gnome/shell/ui/panelMenu.js";
import * as PopupMenu2 from "resource:///org/gnome/shell/ui/popupMenu.js";
import { Extension } from "resource:///org/gnome/shell/extensions/extension.js";

// src/floating_window.ts
import St2 from "gi://St";
import Clutter from "gi://Clutter";
import Gio2 from "gi://Gio";
import * as Main from "resource:///org/gnome/shell/ui/main.js";

// src/image_loader.ts
import St from "gi://St";
import Gio from "gi://Gio";
import Soup from "gi://Soup";
import GLib from "gi://GLib";
function _createIcon(imageBox, cacheFilePath, imageSize) {
  const gicon = Gio.icon_new_for_string(cacheFilePath);
  imageBox.add_child(new St.Icon({
    gicon,
    height: imageSize
  }));
}
async function loadImage(url, uuid, log2, handler) {
  const cacheDir = GLib.build_pathv("/", [GLib.get_user_cache_dir(), uuid]);
  if (!GLib.file_test(cacheDir, GLib.FileTest.IS_DIR)) {
    GLib.mkdir_with_parents(cacheDir, 493);
  }
  const checksum = GLib.compute_checksum_for_string(GLib.ChecksumType.MD5, url, url.length);
  if (checksum === null) {
    logError(new Error("Failed to compute checksum for URL"), "Image loader");
    return;
  }
  const urlHash = checksum;
  const cacheFilePath = GLib.build_filenamev([cacheDir, urlHash]);
  const cacheFile = Gio.File.new_for_path(cacheFilePath);
  try {
    if (cacheFile.query_exists(null)) {
      handler(cacheFilePath);
    } else {
      downloadImage(url, cacheFilePath).then((_) => {
        handler(cacheFilePath);
      }).catch((e) => {
        log2(["Error during image downloading: " + e]);
      });
    }
  } catch (e) {
    log2(["Error during image loading: " + e]);
  }
}
async function loadGicon(url, uuid, icon, log2) {
  loadImage(url, uuid, log2, (cacheFilePath) => {
    icon.gicon = Gio.icon_new_for_string(cacheFilePath);
  });
}
async function loadPopupMenuGicon(url, uuid, menuitem, log2) {
  loadImage(url, uuid, log2, (cacheFilePath) => {
    menuitem.icon.gicon = Gio.icon_new_for_string(cacheFilePath);
  });
}
async function loadWebImage(url, uuid, imageBox, imageSize, log2) {
  loadImage(url, uuid, log2, (cacheFilePath) => _createIcon(imageBox, cacheFilePath, imageSize));
}
async function downloadImage(url, destinationPath) {
  const session = new Soup.Session();
  try {
    const message = Soup.Message.new("GET", url);
    const bytes = await session.send_and_read_async(message, 0, null);
    if (message.status_code !== Soup.Status.OK) {
      throw new Error(`Failed to download image. Status: ${message.status_code} - ${message.reason_phrase}`);
    }
    const file = Gio.File.new_for_path(destinationPath);
    await file.replace_contents_bytes_async(bytes, null, false, Gio.FileCreateFlags.NONE, null, null);
    return destinationPath;
  } finally {
    session.abort();
  }
}

// src/floating_window.ts
var PADDING = 10;
var FloatingScoreWindow = class {
  _extensionPath;
  _uuid;
  _log;
  _settings;
  _windowActor;
  _windowIndex;
  _mainBox;
  _windowWidth;
  _windowHeight;
  constructor(windowIndex, extensionPath, uuid, log2, settings) {
    this._extensionPath = extensionPath;
    this._uuid = uuid;
    this._windowIndex = windowIndex;
    this._log = log2;
    this._settings = settings;
    this._windowWidth = this._settings.get_int("live-window-size-x");
    this._windowHeight = this._settings.get_int("live-window-size-y");
    const closeButtonContainer = new St2.Bin({
      child: this._closeButton(),
      y_align: Clutter.ActorAlign.START,
      x_align: Clutter.ActorAlign.END,
      x_expand: true
    });
    closeButtonContainer.hide();
    this._mainBox = new St2.BoxLayout({
      vertical: true,
      style_class: "main-box"
    });
    const mainBoxContainer = new St2.Bin({
      child: this._mainBox,
      y_align: Clutter.ActorAlign.START,
      x_align: Clutter.ActorAlign.CENTER,
      x_expand: true,
      y_expand: true
    });
    const windowContentContainer = new St2.BoxLayout({
      vertical: true,
      y_expand: true,
      x_expand: true
    });
    windowContentContainer.add_child(closeButtonContainer);
    windowContentContainer.add_child(mainBoxContainer);
    this._windowActor = new St2.Bin({
      style_class: "floating-score-window",
      reactive: true,
      can_focus: true,
      child: windowContentContainer,
      //track_hover: true,
      width: this._windowWidth,
      height: this._windowHeight
    });
    Main.uiGroup.add_child(this._windowActor);
    this.updatePosition();
  }
  _closeButton() {
    const closeButton = new St2.Button({
      style_class: "floating-window-close-button",
      reactive: true,
      can_focus: true,
      child: new St2.Icon({
        icon_name: "window-close-symbolic",
        icon_size: 16
      })
    });
    closeButton.connect("clicked", () => {
      console.log("Close clicked");
    });
    return closeButton;
  }
  _createSeparator() {
    let separator = new St2.BoxLayout({
      style_class: "separator"
      // The CSS class for styling the line
      // You can also set a fixed height here, but it's best done in CSS
      // height: 2, 
    });
    return separator;
  }
  _openURL(url) {
    Gio2.AppInfo.launch_default_for_uri(url, null);
  }
  _addEventHeader(eventHeader, match) {
    const event = match.event;
    const eventType = new St2.BoxLayout({ style_class: "event-type" });
    const url = match.event.eventTypeUrl;
    if (!url) {
      const eventTypeLabel = new St2.Label({ text: match.event.type });
      eventType.add_child(eventTypeLabel);
    } else {
      loadWebImage(url, this._uuid, eventType, 40, this._log);
    }
    eventHeader.add_child(eventType);
    const eventDescription = new St2.BoxLayout({ vertical: true });
    const eventNameLabel = new St2.Label({ text: match.event.title, style_class: "event-text" });
    eventDescription.add_child(eventNameLabel);
    const locationBox = new St2.BoxLayout();
    const eventLocationLabel = new St2.Label({ text: `${match.event.city}, ${match.event.country}`, style_class: "event-text" });
    locationBox.add_child(eventLocationLabel);
    eventDescription.add_child(locationBox);
    if (event.countryCode) {
      const iconPath = `${this._extensionPath}/flags/${event.countryCode.toLowerCase()}.svg`;
      const gicon = Gio2.icon_new_for_string(iconPath);
      const flagIcon = new St2.Icon({ gicon, icon_size: 16, style_class: "player-flag" });
      locationBox.add_child(flagIcon);
    }
    eventHeader.add_child(eventDescription);
    const eventDetails = new St2.BoxLayout({ vertical: true, x_expand: true, xAlign: Clutter.ActorAlign.END, yAlign: Clutter.ActorAlign.START });
    if (event.surface) {
      const surface = new St2.Label({ text: `${event.surface}/${event.indoor ? "Indoor" : "Outdoor"}`, style_class: "event-text" });
      eventDetails.add_child(surface);
    }
    if (event.prizeMoney && event.prizeMoneyCurrency) {
      eventDetails.add_child(new St2.Label({ text: `Prize Money: ${event.prizeMoneyCurrency} ${event.prizeMoney}`, style_class: "event-text" }));
    }
    if (event.singlesDrawSize && event.singlesDrawSize > 0 && event.doublesDrawSize && event.doublesDrawSize > 0) {
      eventDetails.add_child(new St2.Label({ text: `Draw: ${event.singlesDrawSize}/${event.doublesDrawSize}`, style_class: "event-text" }));
    }
    eventHeader.add_child(eventDetails);
  }
  _addMatchHeader(box, match) {
    const matchHeader = new St2.BoxLayout({ style_class: "match-header-box" });
    const matchRound = new St2.Label({
      text: `${match.roundName}`,
      style_class: "match-header-label round-label"
    });
    matchHeader.add_child(matchRound);
    const matchStatus = new St2.Label({
      text: match.displayStatus,
      style_class: `match-header-label match-status-${match.displayStatus.toLowerCase()}`,
      y_align: Clutter.ActorAlign.START
    });
    matchHeader.add_child(matchStatus);
    const matchCourt = new St2.Label({
      text: match.courtName ?? "Unknown",
      style_class: "match-header-label",
      y_align: Clutter.ActorAlign.START
    });
    matchHeader.add_child(matchCourt);
    const matchDuration = new St2.Label({
      text: match.matchTotalTime ?? "Unknown",
      style_class: "match-header-label",
      x_expand: true,
      x_align: Clutter.ActorAlign.END,
      y_align: Clutter.ActorAlign.START
    });
    matchHeader.add_child(matchDuration);
    box.add_child(matchHeader);
  }
  _addTeam(team, isDoubles, row) {
    const teamBox = new St2.BoxLayout({ vertical: false, style_class: "team-box team-row" });
    const playerInfoBox = new St2.BoxLayout({ vertical: true });
    team.players.forEach((p) => {
      const playerRow = new St2.BoxLayout({ style_class: "player-row" });
      const playerImage = new St2.BoxLayout({ style_class: "player-image" });
      playerRow.add_child(playerImage);
      const playerImageUrl = p.headUrl;
      if (playerImageUrl) {
        loadWebImage(playerImageUrl, this._uuid, playerImage, 25, this._log);
      }
      const iconPath = `${this._extensionPath}/flags/${p.countryCode.toLowerCase()}.svg`;
      const gicon = Gio2.icon_new_for_string(iconPath);
      const flagIcon = new St2.Icon({ gicon, icon_size: 16, style_class: "player-flag" });
      playerRow.add_child(flagIcon);
      playerRow.add_child(new St2.Label({ text: team.entryType ? `[${team.entryType}] ` : "" }));
      let name = `<b>${p.firstName} ${p.lastName}</b>`;
      if (team.seed) {
        name += ` (${team.seed})`;
      }
      const playerLabel = new St2.Label();
      playerLabel.clutter_text.set_markup(name);
      const playerButton = new St2.Button({
        style_class: "link-button",
        reactive: true,
        can_focus: true,
        track_hover: true
      });
      const playerUrl = `https://www.atptour.com/en/players/${p.firstName.toLowerCase()}-${p.lastName.toLowerCase()}/${p.id}/overview`;
      playerButton.set_child(playerLabel);
      playerButton.connect("button-press-event", this._openURL.bind(this, playerUrl));
      playerRow.add_child(playerButton);
      playerInfoBox.add_child(playerRow);
    });
    teamBox.add_child(playerInfoBox);
    row.add_child(teamBox);
  }
  _addScore(team, alignment, gameScoreStyle, isServing, row) {
    const serviceBox = new St2.BoxLayout({ style_class: "service-box", y_align: alignment });
    if (isServing) {
      const iconPath = `${this._extensionPath}/icons/tennis-icon.png`;
      const gicon = Gio2.icon_new_for_string(iconPath);
      const serviceIcon = new St2.Icon({
        gicon,
        icon_size: 16
      });
      serviceBox.add_child(serviceIcon);
    }
    row.add_child(serviceBox);
    const gameScoreLabel = new St2.Label({
      text: team.gameScore != null ? String(team.gameScore) : "",
      style_class: `game-score-box`,
      y_align: alignment
    });
    row.add_child(gameScoreLabel);
    this._formatSetScores(team.setScores).forEach((scoreText) => {
      if (!scoreText) {
        return;
      }
      const scoreLabel = new St2.Label({ style_class: "scores-box", y_align: alignment });
      scoreLabel.clutter_text.set_markup(scoreText);
      row.add_child(scoreLabel);
    });
  }
  _addMatchScoreRows(box, match) {
    const row1 = new St2.BoxLayout({ style_class: "match-content-box" });
    this._addTeam(match.team1, match.isDoubles, row1);
    this._addScore(match.team1, Clutter.ActorAlign.END, "game-score-box-top", match.server == 0, row1);
    box.add_child(row1);
    box.add_child(this._createSeparator());
    const row2 = new St2.BoxLayout({ style_class: "match-content-box" });
    this._addTeam(match.team2, match.isDoubles, row2);
    this._addScore(match.team2, Clutter.ActorAlign.START, "game-score-box-bot", match.server == 1, row2);
    box.add_child(row2);
  }
  _addExtrasRows(box, match) {
    if (match.umpireLastName && match.umpireFirstName) {
      const row = new St2.BoxLayout();
      const umpireLabel = new St2.Label({
        text: `Ump: ${match.umpireFirstName} ${match.umpireLastName}`,
        x_expand: true,
        style_class: "small-text-label"
      });
      row.add_child(umpireLabel);
      box.add_child(row);
    }
    const message = new St2.Label({ text: match.message, style_class: "small-text-label" });
    box.add_child(message);
  }
  updateContent(match) {
    this._mainBox.remove_all_children();
    if (!match) {
      this._windowActor.hide();
      return;
    }
    this._windowActor.show();
    const eventHeader = new St2.BoxLayout();
    this._addEventHeader(eventHeader, match);
    this._mainBox.add_child(eventHeader);
    const box = new St2.BoxLayout({ vertical: true, style_class: "sub-main-box" });
    this._addMatchHeader(box, match);
    box.add_child(this._createSeparator());
    this._addMatchScoreRows(box, match);
    box.add_child(this._createSeparator());
    this._addExtrasRows(box, match);
    this._mainBox.add_child(box);
  }
  updatePosition() {
    const primary = Main.layoutManager.primaryMonitor;
    const x = primary.x + primary.width - this._windowWidth - PADDING;
    const y = primary.y + primary.height - PADDING - (this._windowIndex + 1) * (this._windowHeight + PADDING);
    this._windowActor.set_position(x, y);
  }
  hide() {
    this._windowActor.hide();
  }
  destroy() {
    Main.uiGroup.remove_child(this._windowActor);
    this._windowActor.destroy();
  }
  _formatSetScores(scores) {
    if (!scores || scores.length === 0) {
      return [""];
    }
    return scores.filter((s) => s.score).map((s) => {
      let scoreString = `${s.score}`;
      if (s.tiebrake) {
        scoreString += `<sup>${s.tiebrake}</sup>`;
      }
      return scoreString;
    });
  }
};

// src/fetcher.ts
import Soup2 from "gi://Soup";
import GLib2 from "gi://GLib";
import GObject from "gi://GObject";

// src/atp_fetcher.ts
var AtpFetcher = class _AtpFetcher {
  static atp_url = "https://app.atptour.com/api/v2/gateway/livematches/website?scoringTournamentLevel=tour";
  static atp_challenger_url = "https://app.atptour.com/api/v2/gateway/livematches/website?scoringTournamentLevel=challenger";
  _build_req;
  _data_fetcher;
  constructor(build_req, data_fetcher) {
    this._build_req = build_req;
    this._data_fetcher = data_fetcher;
  }
  _get_player_data(p) {
    return {
      id: p["PlayerId"],
      countryCode: p["PlayerCountry"],
      country: p["PlayerCountryName"],
      firstName: p["PlayerFirstName"],
      lastName: p["PlayerLastName"],
      headUrl: `https://www.atptour.com/-/media/alias/player-headshot/${p["PlayerId"]}`,
      displayName: `${p["PlayerFirstName"]} ${p["PlayerLastName"]}`
    };
  }
  _get_set_score(s) {
    return {
      score: s["SetScore"],
      tiebrake: s["TieBreakScore"],
      stats: s["Stats"]
    };
  }
  _get_set_scores(t) {
    let scores = [];
    t["SetScores"].forEach((s) => {
      scores.push(this._get_set_score(s));
    });
    return scores;
  }
  _formatSetScores(team1Scores, team2Scores) {
    if (!team1Scores || !team2Scores || team1Scores.length === 0 || team2Scores.length === 0) {
      return "";
    }
    const scores = [];
    for (let i = 0; i < team1Scores.length && i < team2Scores.length; i++) {
      const score1 = team1Scores[i].score;
      const score2 = team2Scores[i].score;
      if (!score1 || !score2) {
        continue;
      }
      let scoreString = `${score1}-${score2}`;
      const tiebreak1 = team1Scores[i].tiebrake;
      const tiebreak2 = team2Scores[i].tiebrake;
      if (tiebreak1 || tiebreak2) {
        const tiebreakScore = tiebreak1 || tiebreak2;
        scoreString += `(${tiebreakScore})`;
      }
      scores.push(scoreString);
    }
    return scores.join(", ");
  }
  _get_atp_team_data(t, matchType) {
    const players = [];
    players.push(this._get_player_data(t["Player"]));
    if (matchType == "doubles") {
      players.push(this._get_player_data(t["Partner"]));
    }
    return {
      players,
      entryType: t["EntryType"],
      seed: t["Seed"],
      gameScore: t["GameScore"],
      setScores: this._get_set_scores(t),
      displayName: players.map((p) => p.lastName).join("/")
    };
  }
  _get_event_type_url(tour, eventType) {
    if (tour == "ATP") {
      return ["1000", "500", "250"].includes(eventType) ? `https://www.atptour.com/assets/atpwt/images/tournament/badges/categorystamps_${eventType}.png` : void 0;
    } else {
      if (eventType == "CH") {
        return "http://www.atptour.com/assets/atpwt/images/tournament/badges/categorystamps_ch.png";
      }
    }
  }
  _get_match_display_status(ms) {
    let status = "";
    if (ms == "F") {
      status = "Finished";
    } else if (ms == "P") {
      status = "Live";
    } else if (["C", "D", "M", "W"].includes(ms)) {
      status = "Paused";
    } else {
      status = ms;
    }
    return status;
  }
  fetchData(tour, callback) {
    const msg = this._build_req(tour == "ATP" ? _AtpFetcher.atp_url : _AtpFetcher.atp_challenger_url);
    const tennisEvents = [];
    this._data_fetcher(msg, (jsonData) => {
      if (jsonData == null) {
        return callback(void 0);
      }
      const jsonEvents = jsonData["Data"]["LiveMatchesTournamentsOrdered"];
      jsonEvents.forEach((e) => {
        const matches = [];
        const matchMapping = {};
        const event = {
          year: e["EventYear"],
          id: String(e["EventId"]),
          title: e["EventTitle"],
          countryCode: e["EventCountryCode"],
          country: e["EventCountry"],
          location: e["EventLocation"],
          city: e["EventCity"],
          startDate: e["EventStartDate"],
          endDate: e["EventEndDate"],
          type: e["EventType"],
          isLive: e["IsLive"],
          tour,
          matches,
          matchMapping,
          eventTypeUrl: this._get_event_type_url(tour, e["EventType"]),
          name: e["EventTitle"],
          surface: "",
          indoor: false,
          singlesDrawSize: -1,
          doublesDrawSize: -1,
          prizeMoney: -1,
          prizeMoneyCurrency: "",
          status: ""
        };
        tennisEvents.push(event);
        e["LiveMatches"].forEach((m) => {
          const matchType = m["Type"];
          const team1 = this._get_atp_team_data(m["PlayerTeam"], matchType);
          const team2 = this._get_atp_team_data(m["OpponentTeam"], matchType);
          const match = {
            id: m["MatchId"],
            isDoubles: m["IsDoubles"],
            roundName: m["RoundName"],
            courtName: m["CourtName"],
            courtId: m["CourtId"],
            matchTotalTime: m["MatchTimeTotal"],
            matchStateReasonMessage: m["MatchStateReasonMessage"],
            message: m["ExtendedMessage"],
            status: m["MatchStatus"],
            server: m["ServerTeam"],
            winnerId: m["WinningPlayerId"],
            umpireFirstName: m["UmpireFirstName"],
            umpireLastName: m["UmpireLastName"],
            lastUpdate: m["LastUpdated"],
            team1,
            team2,
            event,
            hasFinished: m["MatchStatus"] == "F",
            isLive: m["MatchStatus"] == "P",
            displayName: `${team1.displayName} vs ${team2.displayName}`,
            displayStatus: this._get_match_display_status(m["MatchStatus"]),
            displayScore: this._formatSetScores(team1.setScores, team2.setScores),
            roundId: m["RoundName"],
            matchTimeStamp: ""
          };
          matches.push(match);
          matchMapping[m["MatchId"]] = match;
        });
      });
      callback(tennisEvents);
    });
  }
};

// src/wta_fetcher.ts
var WtaFetcher = class _WtaFetcher {
  static wta_all_events_url_template = "https://api.wtatennis.com/tennis/tournaments/?page=0&pageSize=20&excludeLevels=ITF&from={from-date}&to={to-date}";
  static wta_event_url_template = "https://api.wtatennis.com/tennis/tournaments/{event-id}/{year}/matches?from={from-date}&to={to-date}";
  _build_req;
  _data_fetcher;
  constructor(build_req, data_fetcher) {
    this._build_req = build_req;
    this._data_fetcher = data_fetcher;
  }
  _replace_from_to_date(template) {
    const today = /* @__PURE__ */ new Date();
    const yesterday = new Date(today);
    yesterday.setDate(today.getDate() - 1);
    const tomorrow = new Date(today);
    tomorrow.setDate(today.getDate() + 1);
    return template.replace("{from-date}", yesterday.toISOString().slice(0, 10)).replace("{to-date}", tomorrow.toISOString().slice(0, 10));
  }
  _get_all_events_url() {
    return this._replace_from_to_date(_WtaFetcher.wta_all_events_url_template);
  }
  _get_event_url(eventId, year) {
    const template = this._replace_from_to_date(_WtaFetcher.wta_event_url_template);
    return template.replace("{event-id}", eventId).replace("{year}", year.toString());
  }
  _get_player(p, suffix) {
    const firstName = p[`PlayerNameFirst${suffix}`];
    const lastName = p[`PlayerNameLast${suffix}`];
    return {
      id: p[`PlayerID${suffix}`],
      countryCode: p[`PlayerCountry${suffix}`],
      country: p[`PlayerCountry${suffix}`],
      firstName,
      lastName,
      headUrl: "",
      displayName: `${firstName} ${lastName}`
    };
  }
  _get_players(t, matchType, team) {
    const players = [];
    players.push(this._get_player(t, team));
    if (matchType !== "S") {
      players.push(this._get_player(t, `${team}2`));
    }
    return players;
  }
  _get_set_scores(t, team) {
    const setScores = [];
    for (let i = 1; i <= 5; i++) {
      setScores.push({
        score: t[`ScoreSet${i}${team}`],
        tiebrake: t[`ScoreTbSet${i}`],
        stats: void 0
      });
    }
    return setScores;
  }
  _get_team_data(t, matchType, team) {
    const players = this._get_players(t, matchType, team);
    return {
      players,
      entryType: t[`EntryType${team}`],
      seed: t[`Seed${team}`],
      gameScore: t[`Point${team}`],
      setScores: this._get_set_scores(t, team),
      displayName: players.map((p) => p.lastName).join("/")
    };
  }
  _get_event_type_url(level) {
    switch (level) {
      case "WTA 1000":
        return "https://www.wtatennis.com/resources/v7.8.3/i/elements/1000k-tag.svg";
      case "WTA 500":
        return "https://www.wtatennis.com/resources/v7.8.3/i/elements/500k-tag.svg";
      case "WTA 250":
        return "https://www.wtatennis.com/resources/v7.8.3/i/elements/250k-tag.svg";
      case "WTA 125":
        return "https://www.wtatennis.com/resources/v7.8.3/i/elements/125k-tag.svg";
    }
  }
  _get_round_name(roundId) {
    switch (roundId) {
      case "Q":
        return "Quarterfinal";
      case "S":
        return "Semifinal";
      case "F":
        return "Final";
      case "1":
        return "Qualifying(1)";
      case "2":
        return "Qualifying(2)";
      case "3":
        return "Qualifying(3)";
    }
    return roundId;
  }
  _get_match_status(status) {
    if (status == "U") {
      return "Upcoming";
    } else if (status == "F") {
      return "Finished";
    } else if (status == "P") {
      return "Live";
    } else if (status == "C") {
      return "Paused";
    } else if (status == "S") {
      return "Suspended";
    }
    return status;
  }
  _fetch_event_data(event, events, index, tennisEvents, callback) {
    const msg = this._build_req(this._get_event_url(event.id, event.year));
    this._data_fetcher(msg, (json_data) => {
      json_data["matches"].forEach((m) => {
        const team1 = this._get_team_data(m, m["DrawMatchType"], "A");
        const team2 = this._get_team_data(m, m["DrawMatchType"], "B");
        const tennisMatch = {
          id: m["MatchID"],
          isDoubles: m["DrawMatchType"] !== "S",
          roundId: m["RoundID"],
          roundName: this._get_round_name(m["RoundID"]),
          courtName: m["CourtName"],
          courtId: m["CourtID"],
          matchTotalTime: m["MatchTimeTotal"],
          matchTimeStamp: m["MatchTimeStamp"],
          matchStateReasonMessage: "",
          message: "",
          server: m["Serve"] == "A" ? 0 : m["Serve"] == "B" ? 1 : -1,
          winnerId: -1,
          umpireFirstName: "",
          umpireLastName: "",
          lastUpdate: "",
          team1,
          team2,
          event,
          status: m["MatchState"],
          hasFinished: m["MatchState"] == "F",
          isLive: m["MatchState"] == "P",
          displayName: `${team1.displayName} vs ${team2.displayName}`,
          displayStatus: this._get_match_status(m["MatchState"]),
          displayScore: m["ScoreString"]
        };
        event.matches.push(tennisMatch);
        event.matchMapping[tennisMatch.id] = tennisMatch;
      });
      this._process_event(events, index + 1, tennisEvents, callback);
    });
  }
  _process_event(events, index, tennisEvents, callback) {
    if (index == events.length) {
      return callback(tennisEvents);
    }
    const e = events[index];
    const event = {
      year: e["year"],
      id: String(e["tournamentGroup"]["id"]),
      name: e["tournamentGroup"]["name"],
      title: e["title"],
      countryCode: "",
      country: e["country"],
      location: e["country"],
      city: e["city"],
      startDate: e["startDate"],
      endDate: e["endDate"],
      surface: e["surface"],
      indoor: e["inOutdoor"] == 1,
      type: e["level"],
      isLive: e["status"] == "inProgress",
      tour: "WTA",
      singlesDrawSize: e["singlesDrawSize"],
      doublesDrawSize: e["doublesDrawSize"],
      prizeMoney: e["prizeMoney"],
      prizeMoneyCurrency: e["prizeMoneyCurrency"],
      status: e["status"],
      matches: [],
      matchMapping: {},
      eventTypeUrl: this._get_event_type_url(e["level"])
    };
    tennisEvents.push(event);
    this._fetch_event_data(event, events, index, tennisEvents, callback);
  }
  fetchData(callback) {
    const msg = this._build_req(this._get_all_events_url());
    const tennisEvents = [];
    this._data_fetcher(msg, (json_data) => {
      this._process_event(json_data["content"], 0, tennisEvents, callback);
    });
  }
};

// src/fetcher.ts
var LiveTennis = GObject.registerClass({
  GTypeName: "LiveTennis"
  // It's a good practice to provide a unique GTypeName
}, class LiveTennis2 extends GObject.Object {
  _atp_fetcher;
  _wta_fetcher;
  _atp_lock;
  _atp_challenger_lock;
  _wta_lock;
  _log;
  _settings;
  _atp_events;
  _atp_challenger_events;
  _wta_events;
  constructor(log2, settings) {
    super();
    this._atp_lock = false;
    this._atp_challenger_lock = false;
    this._wta_lock = false;
    this._log = log2;
    this._settings = settings;
    this._atp_events = {};
    this._atp_challenger_events = {};
    this._wta_events = {};
    this._atp_fetcher = new AtpFetcher(this._build_req.bind(this), this._fetch_data_common.bind(this));
    this._wta_fetcher = new WtaFetcher(this._build_req.bind(this), this._fetch_data_common.bind(this));
  }
  _build_req(url) {
    this._log(["Fetching url", url]);
    let request = Soup2.Message.new("GET", url);
    request.request_headers.append("Cache-Control", "no-cache");
    request.request_headers.append("User-Agent", "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/132.0.6831.62 Safari/537.36");
    return request;
  }
  _fetch_data_common(msg, handler) {
    const httpSession = new Soup2.Session();
    httpSession.timeout = 6e4;
    httpSession.send_and_read_async(msg, GLib2.PRIORITY_DEFAULT, null, (_, r) => {
      try {
        const bytes = httpSession.send_and_read_finish(r).get_data();
        if (bytes == null) {
          this._log(["Invalid empty response"]);
          return;
        }
        const response = new TextDecoder("utf-8").decode(bytes);
        if (msg.get_status() > 299) {
          this._log(["Remote server error: ", msg.get_status().toString(), response]);
          return;
        }
        const jsonData = JSON.parse(response);
        if (jsonData.length === 0) {
          this._log(["Remote server error:", response]);
          return;
        }
        this._log(["Received response"]);
        handler(jsonData);
      } catch (e) {
        this._log([`Error fetching data: ${e}`]);
        if (e instanceof Error && e.stack) {
          this._log(["Stack trace", e.stack]);
        }
        handler(null);
      } finally {
        httpSession.abort();
      }
    });
  }
  _fetch_atp_data_common(oldEvents, tour, callback) {
    this._atp_fetcher.fetchData(tour, (tennisEvents) => {
      if (!tennisEvents) {
        this._log([`Fetch for ${tour} received no data`]);
        return callback(oldEvents);
      }
      const tennisEventsMap = {};
      tennisEvents.forEach((e) => tennisEventsMap[e.id] = e);
      callback(tennisEventsMap);
    });
  }
  _fetch_atp_data(callback) {
    if (this._atp_lock) {
      return callback(this._atp_events);
    }
    this._atp_lock = true;
    return this._fetch_atp_data_common(this._atp_events, "ATP", (e) => {
      this._atp_lock = false;
      callback(e);
    });
  }
  _fetch_atp_challenger_data(callback) {
    if (this._atp_challenger_lock) {
      return callback(this._atp_challenger_events);
    }
    this._atp_challenger_lock = true;
    return this._fetch_atp_data_common(this._atp_challenger_events, "ATP-Challenger", (e) => {
      this._atp_challenger_lock = false;
      callback(e);
    });
  }
  _fetch_wta_data(callback) {
    if (this._wta_lock) {
      return callback(this._wta_events);
    }
    this._wta_lock = true;
    return this._wta_fetcher.fetchData((tennisEvents) => {
      this._wta_lock = false;
      if (!tennisEvents) {
        this._log(["Fetch for WTA received no data"]);
        return callback(this._wta_events);
      }
      const tennisEventsMap = {};
      tennisEvents.forEach((e) => tennisEventsMap[e.id] = e);
      callback(tennisEventsMap);
    });
  }
  _post_fetch_handler(oldEvents, newEvents, eventCallback, matchCallback) {
    for (const [eventId, oldEvent] of Object.entries(oldEvents)) {
      if (!(eventId in newEvents)) {
        eventCallback(0 /* DeleteTournament */, oldEvent);
      } else {
        const newEvent = newEvents[eventId];
        for (const [matchId, oldMatch] of Object.entries(oldEvent.matchMapping)) {
          if (!(matchId in newEvent.matchMapping)) {
            matchCallback(3 /* DeleteMatch */, oldEvent, oldMatch);
          }
        }
      }
    }
    for (const [eventId, newEvent] of Object.entries(newEvents)) {
      if (!(eventId in oldEvents)) {
        eventCallback(1 /* AddTournament */, newEvent);
        for (const match of newEvent.matches) {
          matchCallback(4 /* AddMatch */, newEvent, match);
        }
      } else {
        eventCallback(2 /* UpdateTournament */, newEvent);
        const oldEvent = oldEvents[eventId];
        for (const [matchId, newMatch] of Object.entries(newEvent.matchMapping)) {
          if (!(matchId in oldEvent.matchMapping)) {
            matchCallback(4 /* AddMatch */, newEvent, newMatch);
          } else {
            matchCallback(5 /* UpdateMatch */, newEvent, newMatch);
          }
        }
      }
    }
  }
  _process_tour_handler(oldEvents, newEvents, eventCallback, matchCallback, doneCallback) {
    this._post_fetch_handler(oldEvents, newEvents, eventCallback, matchCallback);
    doneCallback(newEvents);
  }
  _process_common(settingKey, oldEvents, fetcher, eventCallback, matchCallback, doneCallback) {
    if (this._settings.get_boolean(`enable-${settingKey}`)) {
      fetcher((newEvents) => {
        this._process_tour_handler(oldEvents, newEvents, eventCallback, matchCallback, doneCallback);
        this._log([`${settingKey} processed`]);
      });
    } else {
      this._process_tour_handler(oldEvents, {}, eventCallback, matchCallback, doneCallback);
      this._log([`${settingKey} not enabled`]);
    }
  }
  _process_atp(eventCallback, matchCallback, doneCallback) {
    this._process_common("atp", this._atp_events, this._fetch_atp_data.bind(this), eventCallback, matchCallback, (newEvents) => {
      this._atp_events = newEvents;
      doneCallback();
    });
  }
  _process_wta(eventCallback, matchCallback, doneCallback) {
    this._process_common("wta", this._wta_events, this._fetch_wta_data.bind(this), eventCallback, matchCallback, (newEvents) => {
      this._wta_events = newEvents;
      doneCallback();
    });
  }
  _process_atp_challenger(eventCallback, matchCallback, doneCallback) {
    this._process_common("atp-challenger", this._atp_challenger_events, this._fetch_atp_challenger_data.bind(this), eventCallback, matchCallback, (newEvents) => {
      this._atp_challenger_events = newEvents;
      doneCallback();
    });
  }
  query(eventCallback, matchCallback, doneCallback) {
    const tourHandlers = [
      this._process_atp.bind(this),
      this._process_atp_challenger.bind(this),
      this._process_wta.bind(this)
    ];
    let count = 0;
    const myDoneCallback = () => {
      count += 1;
      if (count == tourHandlers.length) {
        this._log(["Query processing done"]);
        if (doneCallback) {
          doneCallback();
        }
      }
    };
    tourHandlers.forEach((h) => h(eventCallback, matchCallback, myDoneCallback));
  }
});

// src/menuItem.ts
import Gio3 from "gi://Gio";
import Clutter2 from "gi://Clutter";
import GObject2 from "gi://GObject";
import St3 from "gi://St";
import * as PopupMenu from "resource:///org/gnome/shell/ui/popupMenu.js";
var CheckedMenuItem = GObject2.registerClass({
  Signals: {
    "toggle": { param_types: [Clutter2.Event.$gtype] }
  }
}, class CheckedMenuItem2 extends PopupMenu.PopupBaseMenuItem {
  _checked;
  constructor(text, checked) {
    super();
    this._checked = checked;
    const label = new St3.Label({ text });
    this.actor.add_child(label);
    this._updateOrnament();
    this.actor._delegate = this;
  }
  get checked() {
    return this._checked;
  }
  // prevents menu from being closed
  activate(event) {
    this._checked = !this._checked;
    this._updateOrnament();
    this.emit("toggle", event);
  }
  _updateOrnament() {
    this.setOrnament(this._checked ? PopupMenu.Ornament.CHECK : PopupMenu.Ornament.NONE);
  }
});
var MatchMenuItem = GObject2.registerClass({
  Signals: {
    "toggle": { param_types: [Clutter2.Event.$gtype] }
  }
}, class MatchMenuItem2 extends PopupMenu.PopupBaseMenuItem {
  _extensionPath;
  _container;
  _scoreContainer;
  _match;
  _checked;
  _uuid;
  _log;
  constructor(extensionPath, match, checked, uuid, log2) {
    super();
    this._extensionPath = extensionPath;
    this._container = new St3.BoxLayout();
    this._scoreContainer = new St3.BoxLayout({ x_expand: true, x_align: Clutter2.ActorAlign.END });
    this._match = match;
    this._checked = checked;
    this._uuid = uuid;
    this._log = log2;
    this.actor.add_child(this._container);
    this._updateOrnament();
    this._updateMenu();
    this.actor.add_child(this._scoreContainer);
    this.actor._delegate = this;
  }
  _updateMenu() {
    this._container.remove_all_children();
    this._scoreContainer.remove_all_children();
    this._addTeam(this._match.team1);
    this._container.add_child(new St3.Label({ text: " vs" }));
    this._addTeam(this._match.team2);
    if (this._match.displayScore) {
      this._scoreContainer.add_child(new St3.Label({ text: this._match.displayScore }));
    }
  }
  _addTeam(team) {
    team.players.forEach((p) => {
      const url = p.headUrl;
      if (url) {
        const icon = new St3.Icon({ style_class: "popup-menu-icon" });
        loadGicon(url, this._uuid, icon, this._log);
        this._container.add_child(icon);
      }
      const iconPath = `${this._extensionPath}/flags/${p.countryCode.toLowerCase()}.svg`;
      const gicon = Gio3.icon_new_for_string(iconPath);
      const flagIcon = new St3.Icon({ gicon, icon_size: 16, style_class: "player-flag" });
      this._container.add_child(flagIcon);
    });
    this._container.add_child(new St3.Label({ text: team.displayName }));
  }
  set match(match) {
    this._match = match;
    this._updateMenu();
  }
  get checked() {
    return this._checked;
  }
  set checked(checked) {
    this._checked = checked;
  }
  // prevents menu from being closed
  activate(event) {
    this._log(["Menuitem event activated", this._match.displayName]);
    this._checked = !this._checked;
    this._updateOrnament();
    this.emit("toggle", event);
  }
  _updateOrnament() {
    this.setOrnament(this._checked ? PopupMenu.Ornament.CHECK : PopupMenu.Ornament.NONE);
  }
});

// src/utit.ts
var SortedStringList = class {
  data = [];
  /**
   * Inserts a string into the sorted list and returns its new index.
   * @param item The string to insert.
   * @returns The index where the string was inserted.
   */
  insert(item) {
    const index = this.findInsertionIndex(item);
    this.data.splice(index, 0, item);
    return index;
  }
  /**
   * Removes the first occurrence of a string from the list.
   * @param item The string to remove.
   * @returns A boolean indicating whether the item was found and removed.
   */
  remove(item) {
    const index = this.findIndex(item);
    if (index !== -1) {
      this.data.splice(index, 1);
      return true;
    }
    return false;
  }
  /**
   * Performs a binary search to find the correct insertion index.
   * @param item The string to find the position for.
   * @returns The index where the item should be inserted to maintain order.
   */
  findInsertionIndex(item) {
    let low = 0;
    let high = this.data.length;
    while (low < high) {
      const mid = Math.floor((low + high) / 2);
      if (item > this.data[mid]) {
        low = mid + 1;
      } else {
        high = mid;
      }
    }
    return low;
  }
  /**
   * Retrieves the element at a specific index.
   * @param index The index to retrieve.
   * @returns The element at the specified index, or undefined if the index is out of bounds.
   */
  get(index) {
    return this.data[index];
  }
  /**
   * Finds the index of a string in the list using binary search.
   * @param item The string to find.
   * @returns The index of the item, or -1 if not found.
   */
  findIndex(item) {
    let low = 0;
    let high = this.data.length - 1;
    let index = -1;
    while (low <= high) {
      const mid = Math.floor((low + high) / 2);
      if (this.data[mid] === item) {
        index = mid;
        break;
      } else if (this.data[mid] < item) {
        low = mid + 1;
      } else {
        high = mid - 1;
      }
    }
    return index;
  }
  /**
   * Returns a copy of the sorted list.
   */
  toArray() {
    return [...this.data];
  }
};

// src/extension.ts
var ICON_SIZE = 22;
var _activeFloatingWindows = [];
var _dataFetchTimeout = null;
var _matchCycleTimeout = null;
var _currentMatchIndex = 0;
function _log(logs) {
  console.log("[Live Tennis]", logs.join(", "));
}
var LiveScoreButton = class extends PanelMenu.Button {
  _settings;
  _extensionPath;
  _uuid;
  _matchesMenuItems = /* @__PURE__ */ new Map();
  _tennisEvents;
  _tournamentHeaders = /* @__PURE__ */ new Map();
  _eventAutoItems = /* @__PURE__ */ new Map();
  _refreshItem;
  _refreshLabel;
  _settingsItem;
  _manuallyDeselectedMatches = /* @__PURE__ */ new Set();
  _matchCompletionTimings = /* @__PURE__ */ new Map();
  constructor(settings, extensionPath, uuid) {
    super(0, "Live Score Tracker", false);
    this._settings = settings;
    this._extensionPath = extensionPath;
    this._uuid = uuid;
    this._tennisEvents = new SortedStringList();
    const iconPath = `${this._extensionPath}/icons/tennis-icon.png`;
    const gicon = Gio4.icon_new_for_string(iconPath);
    this.add_child(new St4.Icon({
      gicon,
      style_class: "system-status-icon livescore-panel-button",
      icon_size: ICON_SIZE
    }));
    this._setupBaseMenu();
    this.setLastRefreshTime(null);
  }
  _setupBaseMenu() {
    this.menu.addMenuItem(new PopupMenu2.PopupSeparatorMenuItem());
    const toggleItem = new CheckedMenuItem("Enable live view", this._settings.get_boolean("enabled"));
    toggleItem.connect("toggle", () => this._settings.set_boolean("enabled", toggleItem.checked));
    this.menu.addMenuItem(toggleItem);
    this.menu.addMenuItem(new PopupMenu2.PopupSeparatorMenuItem());
    this._refreshItem = new PopupMenu2.PopupMenuItem("", { reactive: true });
    this._refreshItem.connect("activate", () => {
      this.emit("manual-refresh");
    });
    this._refreshLabel = new St4.Label({ style_class: "livescore-refresh-label" });
    this._refreshLabel.clutter_text.set_markup(`Last Refresh: <span weight='bold'>Never</span>`);
    this._refreshItem.actor.add_child(this._refreshLabel);
    this.menu.addMenuItem(this._refreshItem);
    this.menu.addMenuItem(new PopupMenu2.PopupSeparatorMenuItem());
    this._settingsItem = new PopupMenu2.PopupMenuItem("Settings");
    this._settingsItem.connect("activate", () => this.emit("open-prefs"));
    this.menu.addMenuItem(this._settingsItem);
  }
  setLastRefreshTime(time) {
    this._lastRefreshTime = time;
    if (this._refreshLabel) {
      const timeString = time ? time.format("%H:%M:%S") : "N/A";
      this._refreshLabel.clutter_text.set_markup(`Last Refresh: <span weight='bold'>${timeString}</span>`);
    }
  }
  uniqMatchId(event, match) {
    return `${event.id}-${match.id}`;
  }
  addTournament(event) {
    if (!this._tournamentHeaders.has(event.id)) {
      _log([`Adding tournament: ${event.title} (${event.id})`]);
      const position = this._tennisEvents.insert(event.title);
      const submenuItem = new PopupMenu2.PopupSubMenuMenuItem(event.title, true);
      const eventTypeUrl = event.eventTypeUrl;
      if (eventTypeUrl) {
        loadPopupMenuGicon(eventTypeUrl, this._uuid, submenuItem, _log);
      }
      this.menu.addMenuItem(submenuItem, position);
      this._tournamentHeaders.set(event.id, submenuItem);
      const autoEvents = this._settings.get_strv("auto-view-new-matches");
      const autoItem = new CheckedMenuItem("Auto add new matches", autoEvents.includes(event.id));
      autoItem.connect("toggle", () => this._toggleAutoSelection(event.id));
      submenuItem.menu.addMenuItem(autoItem);
      this._eventAutoItems.set(event.id, autoItem);
    }
  }
  _shouldAutoSelectMatch(matchId, event, match) {
    if (this._manuallyDeselectedMatches.has(matchId)) {
      return false;
    }
    if (match.isLive && this._settings.get_boolean("auto-select-live-matches")) {
      return true;
    }
    const autoEvents = this._settings.get_strv("auto-view-new-matches");
    if (autoEvents && autoEvents.length > 0 && autoEvents.includes(event.id)) {
      return true;
    }
    const countries = this._settings.get_strv("auto-select-country-codes");
    if (countries && countries.length > 0 && [...match.team1.players, ...match.team2.players].some((v) => countries.includes(v.countryCode))) {
      return true;
    }
    const names = this._settings.get_strv("auto-select-player-names").map((n) => n.toLowerCase());
    if (names && names.length > 0 && [...match.team1.players, ...match.team2.players].some(
      (v) => names.includes(v.firstName.toLowerCase()) || names.includes(v.lastName.toLowerCase()) || names.some((n) => v.displayName.includes(n))
    )) {
      return true;
    }
    return false;
  }
  addMatch(event, match) {
    if (!this._matchesMenuItems.has(this.uniqMatchId(event, match))) {
      _log(["Adding match", event.title, match.displayName, match.id]);
      const matchId = this.uniqMatchId(event, match);
      let currentSelection = this._settings.get_strv("selected-matches");
      if (!currentSelection.includes(matchId) && !match.hasFinished) {
        if (this._shouldAutoSelectMatch(matchId, event, match)) {
          _log(["Auto selected", match.displayName]);
          currentSelection.push(matchId);
          this._settings.set_strv("selected-matches", currentSelection);
        }
      }
      _log([event.title, event.id, match.displayName, match.id, String(match.hasFinished), String(currentSelection.includes(matchId))]);
      const menuItem = new MatchMenuItem(this._extensionPath, match, currentSelection.includes(matchId), this._uuid, _log);
      const submenuItem = this._tournamentHeaders.get(event.id);
      submenuItem.menu.addMenuItem(menuItem);
      menuItem.connect("toggle", () => this._toggleMatchSelection(matchId));
      this._matchesMenuItems.set(this.uniqMatchId(event, match), menuItem);
    } else {
      this.updateMatch(event, match);
    }
  }
  updateMatch(event, match) {
    _log(["Updating match", event.title, event.id, match.displayName, match.id]);
    const menuItem = this._matchesMenuItems.get(this.uniqMatchId(event, match));
    if (menuItem) {
      menuItem.match = match;
    }
    if (menuItem.checked) {
      const matchId = this.uniqMatchId(event, match);
      if (this._matchCompletionTimings.has(matchId)) {
        const now = /* @__PURE__ */ new Date();
        now.setMinutes(now.getMinutes() - this._settings.get_int("keep-completed-duration"));
        if (now > this._matchCompletionTimings.get(matchId)) {
          menuItem.checked = false;
          this._matchCompletionTimings.delete(matchId);
        }
      } else {
        this._matchCompletionTimings.set(matchId, /* @__PURE__ */ new Date());
      }
    }
  }
  removeTournament(event) {
    _log([`Removing tournament: ${event.title}`]);
    this._tennisEvents.remove(event.title);
    this.filterAutoEvents((s) => s !== event.id);
    const autoItem = this._eventAutoItems.get(event.id);
    if (autoItem) {
      autoItem.destroy();
      this._eventAutoItems.delete(event.id);
    }
    const header = this._tournamentHeaders.get(event.id);
    if (header) {
      header.destroy();
      this._tournamentHeaders.delete(event.id);
    }
  }
  removeMatch(event, match) {
    _log(["Removinging match", event.title, match.displayName]);
    const matchId = this.uniqMatchId(event, match);
    this.filterLiveViewMatches((id) => id !== matchId);
    this._manuallyDeselectedMatches.delete(matchId);
    const matchItem = this._matchesMenuItems.get(matchId);
    if (matchItem) {
      matchItem.destroy();
      this._matchesMenuItems.delete(matchId);
    }
  }
  _toggleSetting(key, toggleValue) {
    let stored;
    let currentValues = this._settings.get_strv(key);
    if (currentValues.includes(toggleValue)) {
      stored = false;
      currentValues = currentValues.filter((v) => v !== toggleValue);
    } else {
      stored = true;
      currentValues.push(toggleValue);
    }
    this._settings.set_strv(key, currentValues);
    return stored;
  }
  _toggleAutoSelection(eventId) {
    return this._toggleSetting("auto-view-new-matches", eventId);
  }
  _toggleMatchSelection(matchId) {
    this._manuallyDeselectedMatches.add(matchId);
    return this._toggleSetting("selected-matches", matchId);
  }
  _filterSetting(key, handler) {
    let currentSelection = this._settings.get_strv(key);
    currentSelection = currentSelection.filter(handler);
    this._settings.set_strv(key, currentSelection);
    return currentSelection;
  }
  filterLiveViewMatches(handler) {
    return this._filterSetting("selected-matches", handler);
  }
  filterAutoEvents(handler) {
    return this._filterSetting("auto-view-new-matches", handler);
  }
};
var GObjectLiveScoreButton = GObject3.registerClass({
  Signals: { "open-prefs": {}, "manual-refresh": {} }
}, LiveScoreButton);
var LiveScoreExtension = class extends Extension {
  _panelButton;
  _settings;
  _liveTennis;
  _currentMatchesData = [];
  constructor(metadata) {
    super(metadata);
  }
  async _fetchMatchData() {
    try {
      _log(["Starting _fetchMatchData"]);
      const matchIds = /* @__PURE__ */ new Set();
      const eventIds = /* @__PURE__ */ new Set();
      const matchesData = [];
      this._liveTennis.query(
        (r, e) => {
          if (!this._panelButton) return;
          if (r === 1 /* AddTournament */) {
            if (e.title) {
              eventIds.add(e.id);
              this._panelButton.addTournament(e);
            } else {
              _log(["Skipping event having null title", e.id]);
            }
          } else if (r === 2 /* UpdateTournament */) {
            eventIds.add(e.id);
          } else if (r === 0 /* DeleteTournament */) {
            this._panelButton.removeTournament(e);
          }
        },
        (r, e, m) => {
          if (!this._panelButton) return;
          const matchId = this._panelButton.uniqMatchId(e, m);
          if (r === 4 /* AddMatch */) {
            m.eventId = e.id;
            m.eventTitle = e.title;
            this._panelButton.addMatch(e, m);
            matchIds.add(matchId);
            matchesData.push(m);
          } else if (r == 5 /* UpdateMatch */) {
            this._panelButton.updateMatch(e, m);
            matchIds.add(matchId);
            matchesData.push(m);
          } else if (r === 3 /* DeleteMatch */) {
            this._panelButton.removeMatch(e, m);
          }
        },
        () => {
          this._panelButton?.filterAutoEvents((id) => eventIds.has(id));
          this._panelButton?.filterLiveViewMatches((id) => matchIds.has(id));
          this._currentMatchesData = matchesData;
          this._updateFloatingWindows(this._currentMatchesData);
          this._panelButton?.setLastRefreshTime(GLib3.DateTime.new_now_local());
        }
      );
      if (_dataFetchTimeout) {
        GLib3.source_remove(_dataFetchTimeout);
      }
      _dataFetchTimeout = GLib3.timeout_add_seconds(GLib3.PRIORITY_DEFAULT, this._settings.get_int("update-interval"), () => {
        this._fetchMatchData();
        return GLib3.SOURCE_CONTINUE;
      });
    } catch (e) {
      _log(["Error during data fetch", String(e)]);
      if (e instanceof Error) {
        if (e.stack) {
          _log(["Stack trace", e.stack]);
        }
      }
    }
  }
  _updateUI() {
    this._updateFloatingWindows(this._currentMatchesData);
  }
  _destroyLiveView() {
    _activeFloatingWindows.forEach((w) => w.destroy());
    _activeFloatingWindows = [];
  }
  _recreateUI() {
    this._destroyLiveView();
    this._updateUI();
  }
  _getSelectedMatches(matchesData) {
    const selectedMatchIds = this._settings.get_strv("selected-matches");
    const onlyLiveMatches = this._settings.get_boolean("only-show-live-matches");
    const filteredMatchData = matchesData.filter((m) => onlyLiveMatches && m.isLive && selectedMatchIds.includes(this._panelButton.uniqMatchId(m.event, m)));
    _log(["Live View Count", matchesData.length.toString(), selectedMatchIds.length.toString(), filteredMatchData.length.toString()]);
    return filteredMatchData;
  }
  _shouldHideLiveView(selectedMatches) {
    return !this._settings?.get_boolean("enabled") || this._settings?.get_boolean("auto-hide-no-live-matches") && selectedMatches.every((m) => !m.isLive);
  }
  _updateFloatingWindows(matchesData) {
    const selectedMatches = this._getSelectedMatches(matchesData);
    if (this._shouldHideLiveView(selectedMatches)) {
      _activeFloatingWindows.forEach((w) => w.hide());
      return;
    }
    const numWindows = this._settings.get_int("num-windows");
    _log(["Will create windows", numWindows.toString(), selectedMatches.length.toString()]);
    while (_activeFloatingWindows.length < numWindows) {
      _activeFloatingWindows.push(new FloatingScoreWindow(_activeFloatingWindows.length, this.path, this.uuid, log, this._settings));
    }
    while (_activeFloatingWindows.length > numWindows) {
      _activeFloatingWindows.pop()?.destroy();
    }
    this._destroyCycleTimeout();
    if (selectedMatches.length > numWindows) {
      this._cycleMatches(selectedMatches);
    } else {
      selectedMatches.forEach((match, i) => _activeFloatingWindows[i].updateContent(match));
      for (let i = selectedMatches.length; i < _activeFloatingWindows.length; i++) {
        _activeFloatingWindows[i].updateContent(void 0);
      }
    }
  }
  _destroyCycleTimeout() {
    if (_matchCycleTimeout) {
      GLib3.source_remove(_matchCycleTimeout);
      _matchCycleTimeout = null;
    }
  }
  _cycleMatches(matchesData) {
    const cycle = () => {
      const selectedMatches = this._getSelectedMatches(matchesData);
      if (this._shouldHideLiveView(selectedMatches)) {
        _activeFloatingWindows.forEach((w) => w.hide());
        return GLib3.SOURCE_REMOVE;
      }
      if (selectedMatches.length <= _activeFloatingWindows.length) {
        this._updateFloatingWindows(matchesData);
        this._destroyCycleTimeout();
        return GLib3.SOURCE_REMOVE;
      }
      for (let i = 0; i < _activeFloatingWindows.length; i++) {
        const matchToShow = selectedMatches[(_currentMatchIndex + i) % selectedMatches.length];
        _activeFloatingWindows[i].updateContent(matchToShow);
      }
      _currentMatchIndex = (_currentMatchIndex + 1) % selectedMatches.length;
      return GLib3.SOURCE_CONTINUE;
    };
    cycle();
    _matchCycleTimeout = GLib3.timeout_add_seconds(GLib3.PRIORITY_DEFAULT, this._settings.get_int("match-display-duration"), cycle);
  }
  enable() {
    this._settings = this.getSettings();
    this._liveTennis = new LiveTennis(_log, this._settings);
    this._panelButton = new GObjectLiveScoreButton(this._settings, this.path, this.uuid);
    this._panelButton.connect("open-prefs", () => this.openPreferences());
    this._panelButton.connect("manual-refresh", () => this._fetchMatchData());
    [
      "enabled",
      "num-windows",
      "selected-matches",
      "auto-view-new-matches",
      "match-display-duration",
      "enable-atp",
      "enable-wta",
      "enable-atp-challenger",
      "auto-hide-no-live-matches"
    ].forEach((k) => this._settings.connect(`changed::${k}`, () => this._updateUI()));
    ["live-window-size-x", "live-window-size-y"].forEach((k) => this._settings.connect(`changed::${k}`, () => this._recreateUI()));
    Main2.panel.addToStatusArea(this.uuid, this._panelButton);
    this._fetchMatchData();
  }
  disable() {
    if (_dataFetchTimeout) GLib3.source_remove(_dataFetchTimeout);
    this._destroyCycleTimeout();
    this._destroyLiveView();
    this._panelButton?.destroy();
    this._panelButton = void 0;
    this._settings = void 0;
    this._liveTennis = void 0;
  }
};
export {
  LiveScoreExtension as default
};
