const Gtk = imports.gi.Gtk;
const GObject = imports.gi.GObject;
const Gio = imports.gi.Gio;
const GLib = imports.gi.GLib;

const ExtensionUtils = imports.misc.extensionUtils;
const Me = ExtensionUtils.getCurrentExtension();

function init() {
}

function buildPrefsWidget() {
    const widget = new MyPrefsWidget();

    // GTK4 doesn't need show_all anymore, but for GTK3 compatibility
    if (widget.show_all) {
        widget.show_all();
    }

    return widget;
}

const MyPrefsWidget = GObject.registerClass(
class MyPrefsWidget extends Gtk.Box {
    _init() {
        super._init({
            orientation: Gtk.Orientation.HORIZONTAL,
            spacing: 12,
            hexpand: true,
            vexpand: true
        });

        this._settings = ExtensionUtils.getSettings('org.gnome.shell.extensions.crypto-stash');
        this._path = Me.path;

        this.loadCryptoRates();
    }

    loadCryptoRates() {
        this._loadingText = new Gtk.Label({
            visible: true,
            justify: Gtk.Justification.CENTER,
            label: 'Loading crypto rates',
            hexpand: true,
            vexpand: true
        });
        this.append(this._loadingText);

        this.availableCoins = ['BTC'];

        const HTTP = Me.imports.HTTP;
        const Globals = Me.imports.Globals;
        this._Globals = Globals;

        HTTP.getJSON(Globals.GET_CRYPTO_RATES_URL, (error, data) => {
            log('HTTP.getJSON callback fired');
            if (error) {
                log(`Error fetching crypto rates: ${error}`);
            } else if (data && data.hasOwnProperty('data') && data.data.length > 0) {
                log(`Received ${data.data.length} crypto rates`);
                this.availableCoins = data.data.map((c) => c.symbol).sort((a, b) => {
                    if (a > b) return 1;
                    if (a < b) return -1;
                    return 0;
                });
            }

            try {
                this.remove(this._loadingText);
                log('About to call render()');
                this.render();
                log('render() completed');
            } catch (e) {
                log(`Error in render: ${e}`);
                logError(e);
            }
        });
    }

    render() {
        const StashModel = Me.imports.StashModel;
        this._store = new StashModel.StashModel();

        const sidebar = new Gtk.Box({
            orientation: Gtk.Orientation.VERTICAL,
            width_request: 200
        });
        sidebar.append(this._getTreeView());
        sidebar.append(this._getToolbar());
        this.append(sidebar);

        this._configLayout = new Gtk.Box({
            orientation: Gtk.Orientation.HORIZONTAL,
            hexpand: true,
            width_request: 500
        });
        this.append(this._configLayout);

        const metadata = Me.metadata;

        this._introText = new Gtk.Label({
            visible: true,
            justify: Gtk.Justification.LEFT,
            label: `${this._Globals.SYMBOLS.wallet}\n\n${metadata['name']} v${metadata['tag'].toFixed(2)}\n\nAuthor: ${metadata['author']} - <a href="${metadata['author_url']}">${metadata['author_url']}</a>\n\nRepository: <a href="${metadata['url']}">${metadata['url']}</a>`,
            use_markup: true,
            xalign: 0,
            hexpand: true,
            vexpand: true
        });
        this._configLayout.append(this._introText);

        this._selection = this._treeView.get_selection();
        this._selection.connect('changed', this._onSelectionChanged.bind(this));
    }

    _getTreeView() {
        this._treeView = new Gtk.TreeView({
            model: this._store,
            headers_visible: false,
            reorderable: true,
            hexpand: false,
            vexpand: true
        });

        let label = new Gtk.TreeViewColumn({title: 'Stashes'});
        let renderer = new Gtk.CellRendererText();
        label.pack_start(renderer, true);
        label.add_attribute(renderer, 'text', 0);
        this._treeView.insert_column(label, 0);

        return this._treeView;
    }

    _onSelectionChanged() {
        let [isSelected, , iter] = this._selection.get_selected();

        if (isSelected) {
            this._showStashConfig(this._store.getConfig(iter));
            this._introText.visible = false;
        } else {
            this._showStashConfig(null);
            this._introText.visible = true;
        }

        this._updateToolbar();
    }

    _showStashConfig(config) {
        if (this._stashConfigView) {
            this._configLayout.remove(this._stashConfigView.widget);
            this._stashConfigView.destroy();
            this._stashConfigView = null;
        }

        if (config === null) {
            return;
        }

        const StashConfigView = Me.imports.StashConfigView;
        this._stashConfigView = new StashConfigView.StashConfigView(config, this.availableCoins);
        this._configLayout.append(this._stashConfigView.widget);
    }

    _getToolbar() {
        let toolbar = this._toolbar = new Gtk.Box({
            orientation: Gtk.Orientation.HORIZONTAL,
            spacing: 0
        });

        // add_css_class is GTK4, use get_style_context for GTK3
        if (toolbar.add_css_class) {
            toolbar.add_css_class('toolbar');
        } else {
            toolbar.get_style_context().add_class('toolbar');
        }

        let newButton = new Gtk.Button({
            icon_name: 'list-add-symbolic',
            has_frame: false
        });
        newButton.connect('clicked', this._addClicked.bind(this));
        toolbar.append(newButton);

        let delButton = this._delButton = new Gtk.Button({
            icon_name: 'list-remove-symbolic',
            has_frame: false
        });
        delButton.connect('clicked', this._delClicked.bind(this));
        toolbar.append(delButton);

        this._updateToolbar();

        return toolbar;
    }

    _updateToolbar() {
        let sensitive = false;

        if (this._selection) {
            let [isSelected] = this._selection.get_selected();
            sensitive = isSelected;
        }

        this._delButton.set_sensitive(sensitive);
    }

    _addClicked() {
        this._store.append();
        this._updateToolbar();
    }

    _delClicked() {
        let [isSelected, , iter] = this._selection.get_selected();

        if (isSelected) {
            this._store.remove(iter);
        }

        this._updateToolbar();
    }
});
