import Clutter from 'gi://Clutter';
import GLib from 'gi://GLib';
import GObject from 'gi://GObject';
import St from 'gi://St';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import { Constantes } from './types.js';
import Utils from './utils.js';
class GraphOverlay {
    constructor() {
        this.label = new St.Label({ style_class: 'label' });
        this.actor = new St.Bin({
            style_class: 'gsp-graph-overlay',
            reactive: true,
        });
        this.actor.add_child(this.label);
        Main.layoutManager.addChrome(this.actor);
        this.actor.hide();
    }
    destroy() {
        this.actor.destroy();
    }
}
export const DEFAULT_HORIZONTAL_GRAPH_OPTIONS = {
    updateInterval: Constantes.INDICATOR_UPDATE_INTERVAL,
    offsetX: 5,
    offsetY: 3,
    units: '',
    gridColor: 'grid-color',
    autoscale: true,
    showMax: true,
    fillAll: false,
    max: 0,
};
export default GObject.registerClass(class HorizontalGraph extends St.Bin {
    constructor(name, options) {
        super({
            style_class: 'gsp-color gsp-graph-area',
            reactive: true,
            trackHover: true,
            x_expand: true,
            y_expand: true,
            xAlign: Clutter.ActorAlign.START,
            yAlign: Clutter.ActorAlign.FILL,
        });
        this.timeout = 0;
        this.graphoverlay = new GraphOverlay();
        this.gridColor = Constantes.DEFAULT_GRID_COLOR;
        this.max = -1;
        this.options = DEFAULT_HORIZONTAL_GRAPH_OPTIONS;
        this.ready = true;
        this.renderStats = [];
        this.stats = {};
        this.styleChanged = false;
        this.drawing_area = new St.DrawingArea({
            reactive: true,
            xExpand: true,
            yExpand: true,
            xAlign: Clutter.ActorAlign.FILL,
            yAlign: Clutter.ActorAlign.FILL,
        });
        this.stack = [];
        this.last_stack = [];
        this.name = name;
        if (options) {
            this.options = { ...this.options, ...options };
        }
        this.drawing_area.connect('repaint', this.repaint.bind(this));
        this.add_child(this.drawing_area);
        this.connect('style-changed', this.updateStyles.bind(this));
        this.timeout = GLib.timeout_add(GLib.PRIORITY_DEFAULT, this.options.updateInterval, () => {
            if (this.ready) {
                this.drawing_area.queue_repaint();
            }
            return true;
        });
        if (!this.options.autoscale) {
            this.max = this.options.max;
            this.updateMaxLabel();
        }
    }
    enable() { }
    disable() { }
    destroy() {
        if (Utils.debugMode) {
            Utils.debug(`HorizontalGraph::destroy ${this.name}`);
        }
        this.ready = false;
        if (this.timeout !== 0) {
            GLib.source_remove(this.timeout);
            this.timeout = 0;
        }
        super.destroy();
    }
    addDataSet(name, color) {
        this.renderStats.push(name);
        this.stats[name] = {
            color: color,
            cairo_color: this.lookupColor(color, Constantes.DEFAULT_STATS_COLOR),
            values: [],
            scaled: [],
            max: -1,
        };
    }
    addDataPoint(name, value) {
        this.stats[name].values.push(value);
    }
    updateDataSetMax(name) {
        this.stats[name].max = this.stats[name].values.reduce((prev, cur) => {
            return Math.max(prev, cur);
        }, 0);
        if (this.max < this.stats[name].max &&
            (Utils.showMaxLines || !name.startsWith('network-max-'))) {
            this.max = this.stats[name].max;
            this.updateMaxLabel();
        }
    }
    updateMax() {
        let max = 0;
        this.renderStats.map(k => {
            if (Utils.showMaxLines || !k.startsWith('network-max-')) {
                max = Math.max(max, this.stats[k].max);
            }
            return k;
        });
        if (max < this.max) {
            this.max = max;
            this.updateMaxLabel();
        }
    }
    updateMaxLabel() {
        if (this.options.showMax) {
            this.graphoverlay.label.set_text(Utils.formatMetricPretty(this.max * (Utils.bitsPerSecond ? 8 : 1), Utils.bitsPerSecond ? 'b/s' : 'B/s'));
        }
    }
    lookupColor(name, defaultColor) {
        return Utils.lookupColor(this, name, defaultColor);
    }
    updateStyles() {
        if (this.is_mapped()) {
            this.gridColor = this.lookupColor(this.options.gridColor, this.gridColor);
            this.renderStats.map(k => {
                const stat = this.stats[k];
                stat.cairo_color = this.lookupColor(stat.color, Constantes.DEFAULT_STATS_COLOR);
                return k;
            });
        }
    }
    drawGridLines(cr, width, gridOffset, count, color) {
        for (let i = 1; i <= count; ++i) {
            cr.moveTo(0, i * gridOffset + 0.5);
            cr.lineTo(width, i * gridOffset + 0.5);
        }
        cr.setSourceRGBA(color.red, color.green, color.blue, color.alpha);
        cr.setLineWidth(1);
        cr.setDash([2, 1], 0);
        cr.stroke();
    }
    repaint(area) {
        const [width, height] = area.get_surface_size();
        const cr = area.get_context();
        const gridOffset = height / (Constantes.INDICATOR_NUM_GRID_LINES + 1);
        if (!this.ready ||
            Main.overview.visibleTarget ||
            !this.get_stage() ||
            !this.visible ||
            !cr)
            return;
        if (!this.styleChanged) {
            this.updateStyles();
            this.styleChanged = true;
        }
        this.drawGridLines(cr, width, gridOffset, Constantes.INDICATOR_NUM_GRID_LINES, this.gridColor);
        this.drawGridLines(cr, width, gridOffset / 2, Constantes.INDICATOR_NUM_GRID_LINES * 2 + 1, {
            red: this.gridColor.red,
            green: this.gridColor.green,
            blue: this.gridColor.blue,
            alpha: this.gridColor.alpha * 0.2,
        });
        this.renderStats.map(k => {
            const stat = this.stats[k];
            const new_width = width + 1;
            if (stat.values.length > new_width) {
                stat.values = stat.values.slice(stat.values.length - new_width, stat.values.length);
            }
            if (this.options.autoscale) {
                this.updateDataSetMax(k);
            }
            return k;
        });
        if (this.options.autoscale) {
            this.updateMax();
        }
        this.renderStats.map(k => {
            this.stats[k].scaled = this.stats[k].values.map(cur => {
                return cur / this.max;
            });
            return k;
        });
        let renders = this.renderStats;
        const mem_stack = Utils.memStack && this.renderStats[0].startsWith('mem-');
        if (mem_stack) {
            this.stack = new Array(this.stats[this.renderStats[0]].scaled.length).fill(0);
            renders = [
                'mem-cached-used',
                'mem-user-used',
                'mem-shared-used',
                'mem-buffer-used',
                'mem-locked-used',
            ];
        }
        for (let index = renders.length - 1; index >= 0; index--) {
            const k = renders[index];
            if (Utils.debugMode) {
                const scaled = this.stats[k].scaled[this.stats[k].scaled.length - 1];
                const value = this.stats[k].values[this.stats[k].values.length - 1];
                console.log('==> ' + index + ': ' + k + ' = ' + value + ' / ' + scaled);
                console.log('  |--> memStack.....: ' + Utils.memStack);
                console.log('  |--> mem_stack....: ' + mem_stack);
                console.log('  |--> showMaxLines.: ' + Utils.showMaxLines);
                console.log('  |--> bitsPerSecond: ' + Utils.bitsPerSecond);
            }
            if (!Utils.showMaxLines && k.startsWith('network-max-'))
                continue;
            const stat = this.stats[k];
            const outlineColor = this.lookupColor(stat.color, stat.cairo_color);
            if (this.max <= 0.00001) {
                continue;
            }
            if (index === 0 || this.options.fillAll) {
                if (mem_stack) {
                    this.last_stack = this.stack.slice();
                    this.stack.forEach((_, i) => (this.stack[i] += stat.scaled[i]));
                    this.plotDataSet(cr, height, this.stack);
                    this.closeStack(cr, height, this.last_stack);
                }
                else {
                    this.plotDataSet(cr, height, stat.scaled);
                    cr.lineTo(stat.scaled.length - 1, height);
                    cr.lineTo(0, height);
                }
                cr.setSourceRGBA(outlineColor.red, outlineColor.green, outlineColor.blue, outlineColor.alpha * 0.2);
                cr.closePath();
                cr.fill();
            }
            this.plotDataSet(cr, height, mem_stack ? this.stack : stat.scaled);
            cr.setSourceRGBA(outlineColor.red, outlineColor.green, outlineColor.blue, outlineColor.alpha);
            cr.setLineWidth(1.0);
            cr.setDash([], 0);
            cr.stroke();
        }
    }
    plotDataSet(cr, height, values) {
        cr.moveTo(0, (1 - (values[0] || 0)) * height);
        for (let k = 1; k < values.length; ++k) {
            cr.lineTo(k, (1 - values[k]) * height);
        }
    }
    closeStack(cr, height, values) {
        let k = values.length - 1;
        cr.lineTo(k, (1 - (values[k] || 0)) * height);
        for (k--; k > -1; k--) {
            cr.lineTo(k, (1 - values[k]) * height);
        }
    }
    setOverlayPosition(x, y) {
        this.graphoverlay.actor.set_position(x + 12 + this.options.offsetX, y + 12 + this.options.offsetY);
    }
    show() {
        this.ready = true;
        this.graphoverlay.actor.show();
        this.graphoverlay.actor.opacity = 0;
        const easeActor = this.graphoverlay.actor;
        easeActor.ease({
            opacity: 255,
            time: Constantes.ITEM_LABEL_SHOW_TIME,
            transition: Clutter.AnimationMode.EASE_OUT_QUAD,
        });
        super.show();
    }
    hide() {
        this.ready = false;
        this.graphoverlay.actor.hide();
        super.hide();
    }
});
