import Cairo from 'cairo';
import Atk from 'gi://Atk';
import Clutter from 'gi://Clutter';
import GLib from 'gi://GLib';
import GObject from 'gi://GObject';
import Shell from 'gi://Shell';
import St from 'gi://St';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import { adjustStyleClass } from './style.js';
import { Constantes } from './types.js';
import Utils from './utils.js';
export const DEFAULT_OPTIONS = {
    updateInterval: Constantes.INDICATOR_UPDATE_INTERVAL,
    barPadding: 1,
    barWidth: 8,
    gridColor: 'grid-color',
    decay: 0.2,
};
export default GObject.registerClass(class Indicator extends St.Widget {
    constructor(name, options) {
        super({
            reactive: true,
            canFocus: true,
            trackHover: true,
            styleClass: adjustStyleClass('panel-button gsp-color gsp-header'),
            accessibleName: name,
            accessibleRole: Atk.Role.MENU,
            layoutManager: new Clutter.BinLayout(),
            xExpand: true,
            yExpand: true,
            xAlign: Clutter.ActorAlign.START,
            yAlign: Clutter.ActorAlign.FILL,
        });
        this.barPadding = 0;
        this.barWidth = 0;
        this.gridColor = Constantes.DEFAULT_GRID_COLOR;
        this.renderStats = [];
        this.resized = false;
        this.scale_factor = 0;
        this.styleCached = false;
        this.timeout = 0;
        this.options = DEFAULT_OPTIONS;
        this.stats = {};
        this.currentLabels = {};
        this.currentValues = {};
        this.currentColors = {};
        this.datasetNames = [];
        this.name = name;
        this.box = new St.BoxLayout({
            xExpand: true,
            yExpand: true,
            xAlign: Clutter.ActorAlign.START,
            yAlign: Clutter.ActorAlign.FILL,
            style_class: 'gsp-indicator',
            reactive: true,
            track_hover: true,
        });
        this.drawing_area = new St.DrawingArea({
            reactive: true,
            xExpand: true,
            yExpand: true,
            xAlign: Clutter.ActorAlign.FILL,
            yAlign: Clutter.ActorAlign.FILL,
        });
        if (options) {
            this.options = { ...this.options, ...options };
        }
        this.scale_factor = St.ThemeContext.get_for_stage(Shell.Global.get().get_stage()).scale_factor;
        this.barPadding = this.options.barPadding * this.scale_factor;
        this.barWidth = this.options.barWidth * this.scale_factor;
        this.drawing_area.connect('repaint', this.repaint.bind(this));
        this.box.add_child(this.drawing_area);
        this.box.connect('notify::visible', this.onVisibilityChanged.bind(this));
        this.box.connect('style-changed', this.updateStyles.bind(this));
        this.box.connect('button-press-event', this.showSystemMonitorOrPrefs.bind(this));
        this.dropdownLayout = new Clutter.GridLayout();
        this.dropdown = new St.Widget({
            layout_manager: this.dropdownLayout,
            reactive: true,
            style_class: 'gsp-dropdown',
        });
        Main.layoutManager.addChrome(this.dropdown);
        this.dropdown.hide();
        this.add_child(this.box);
        this.remove_style_class_name('panel-button');
    }
    addDataSet(name, color) {
        this.renderStats.push(name);
        this.stats[name] = {
            color: color,
            cairo_color: this.lookupColor(color, Constantes.DEFAULT_STATS_COLOR),
            values: [],
            max: -1,
        };
    }
    addDataPointWithColor(name, value, color) {
        this.stats[name].color = color;
        this.stats[name].cairo_color = this.lookupColor(color, this.stats[name].cairo_color);
        this.stats[name].values.push(value);
    }
    addDataPoint(name, value) {
        this.stats[name].values.push(value);
    }
    onVisibilityChanged() {
        if (!this.box.visible) {
            this.dropdown.hide();
        }
    }
    resize() {
        this.box.set_width(this.renderStats.length * (this.barWidth + this.barPadding) + 1);
        this.resized = true;
    }
    enable() {
        this.graph?.enable();
        this._updateValues();
        this.resize();
        if (this.timeout === 0) {
            this.timeout = GLib.timeout_add(GLib.PRIORITY_DEFAULT, this.options.updateInterval, this._updateValues.bind(this));
        }
    }
    disable() {
        this.graph?.disable();
        if (this.timeout) {
            GLib.source_remove(this.timeout);
            this.timeout = 0;
        }
    }
    show() {
        this.enable();
    }
    hide() {
        this.disable();
    }
    showPopup() {
        if (this.graph) {
            this.graph.enable();
            this.dropdown.opacity = 0;
            this.dropdown.show();
            const monitorIndex = Main.layoutManager.primaryIndex;
            const monitor = Main.layoutManager.monitors[monitorIndex];
            const [stageX, stageY] = this.box.get_transformed_position();
            const itemWidth = this.box.allocation.x2 - this.box.allocation.x1;
            const itemHeight = this.box.allocation.y2 - this.box.allocation.y1;
            const labelWidth = this.dropdown.width;
            const xOffset = Math.floor((itemWidth - labelWidth) / 2);
            const node = this.dropdown.get_theme_node();
            const yOffset = node.get_length('-y-offset');
            const easeActor = this.dropdown;
            const y = stageY + itemHeight + yOffset;
            const x = Math.min(stageX + xOffset, monitor.x + monitor.width - 4 - Math.max(itemWidth, labelWidth));
            this.graph?.set_width(this.dropdown.width - 24);
            this.dropdown.set_position(x, y);
            easeActor.ease({
                opacity: 255,
                time: Constantes.ITEM_LABEL_SHOW_TIME,
                transition: Clutter.AnimationMode.EASE_OUT_QUAD,
                onComplete: () => {
                    if (this.graph) {
                        this.graph.setOverlayPosition(x, y);
                        this.graph.show();
                    }
                },
            });
        }
    }
    _updateValues() {
        this.updateValues();
        this.drawing_area.queue_repaint();
        return true;
    }
    showSystemMonitorOrPrefs(_actor, event) {
        if (event.get_button() !== 1) {
            const extension = Utils.extension;
            extension.openPreferences();
            return Clutter.EVENT_PROPAGATE;
        }
        const appSys = Shell.AppSystem.get_default();
        const systemMonitorSignature = [
            'org.gnome.SystemMonitor.desktop',
            'gnome-system-monitor.desktop',
            'gnome-system-monitor_gnome-system-monitor.desktop',
        ];
        for (const signature of systemMonitorSignature) {
            const app = appSys.lookup_app(signature);
            if (app) {
                app.activate();
                break;
            }
        }
        return Clutter.EVENT_PROPAGATE;
    }
    hidePopup() {
        if (this.graph) {
            this.graph.disable();
            const easeActor = this.dropdown;
            easeActor.ease({
                opacity: 0,
                time: Constantes.ITEM_LABEL_HIDE_TIME,
                transition: Clutter.AnimationMode.EASE_OUT_QUAD,
                onComplete: () => {
                    this.graph?.hide();
                    this.dropdown.hide();
                },
            });
        }
    }
    destroy() {
        if (Utils.debugMode) {
            Utils.debug(`Indicator::destroy ${this.name}`);
        }
        if (this.timeout !== 0) {
            if (Utils.debugMode) {
                Utils.debug(`Indicator::destroy ${this.name}, clear timeout`);
            }
            GLib.source_remove(this.timeout);
            this.timeout = 0;
        }
        this.box.destroy();
        this.graph?.destroy();
        super.destroy();
    }
    lookupColor(name, defaultColor) {
        return Utils.lookupColor(this, name, defaultColor);
    }
    updateValues() { }
    updateStyles() {
        if (this.box.is_mapped()) {
            this.gridColor = this.lookupColor(this.options.gridColor, this.gridColor);
            this.renderStats.map(k => {
                const stat = this.stats[k];
                stat.cairo_color = this.lookupColor(stat.color, stat.cairo_color);
                for (const value of stat.values) {
                    if (typeof value !== 'number') {
                        const indicatorStatValues = value;
                        for (const indicatorStatValue of indicatorStatValues.values) {
                            indicatorStatValue.cairo_color = this.lookupColor(indicatorStatValue.color, indicatorStatValue.cairo_color);
                        }
                    }
                }
                return k;
            });
        }
    }
    repaint(area) {
        const cr = area.get_context();
        if (Main.overview.visibleTarget || !this.box.get_stage() || !this.box.visible || !cr) {
            return;
        }
        if (!this.styleCached) {
            this.updateStyles();
            this.styleCached = true;
        }
        if (!this.resized) {
            this.resize();
        }
        const [width, height] = area.get_surface_size();
        const gridOffset = Math.floor(height / Constantes.INDICATOR_NUM_GRID_LINES + 2);
        for (let i = 1; i <= Constantes.INDICATOR_NUM_GRID_LINES + 2; i++) {
            const y = i * gridOffset;
            cr.moveTo(0, y);
            cr.lineTo(width + 2, y);
        }
        cr.setSourceRGBA(this.gridColor.red, this.gridColor.green, this.gridColor.blue, this.gridColor.alpha);
        cr.setAntialias(Cairo.Antialias.NONE);
        cr.setLineWidth(1);
        if (this.renderStats.length > 1) {
            cr.setDash([this.barWidth, this.barPadding], 0);
        }
        else {
            cr.setDash([], 0);
        }
        cr.stroke();
        this.renderStats.map(k => {
            const stat = this.stats[k];
            const keepNumStats = 3;
            if (stat.values.length > keepNumStats) {
                stat.values = stat.values.slice(stat.values.length - keepNumStats, stat.values.length);
            }
            return k;
        });
        for (let index = 0; index < this.renderStats.length; index++) {
            const currentStat = this.stats[this.renderStats[index]];
            const renderingStats = [];
            const currentStatValue = currentStat.values.length === 0 ? 0 : currentStat.values[0];
            if (typeof currentStatValue === 'number') {
                const cairo_color = this.lookupColor(currentStat.color, currentStat.cairo_color);
                renderingStats.push({
                    value: currentStatValue,
                    cairo_color: cairo_color,
                    color: {
                        red: cairo_color.red,
                        green: cairo_color.green,
                        blue: cairo_color.blue,
                        alpha: cairo_color.alpha * 0.8,
                    },
                });
            }
            else {
                const combined = currentStatValue;
                for (const value of combined.values) {
                    if (value.visible) {
                        const cairo_color = this.lookupColor(value.color, value.cairo_color);
                        renderingStats.push({
                            value: value.value,
                            cairo_color: cairo_color,
                            color: {
                                red: cairo_color.red,
                                green: cairo_color.green,
                                blue: cairo_color.blue,
                                alpha: cairo_color.alpha * 0.8,
                            },
                        });
                    }
                }
            }
            for (const renderedStat of renderingStats) {
                const color = renderedStat.color;
                const barHeight = height * renderedStat.value;
                const offsetX = index * (this.barWidth + this.barPadding);
                const offsetY = height - barHeight;
                cr.setAntialias(Cairo.Antialias.NONE);
                cr.setSourceRGBA(color.red, color.green, color.blue, color.alpha);
                cr.rectangle(offsetX, offsetY, this.barWidth, barHeight);
                cr.fill();
                cr.stroke();
            }
        }
    }
    buildPopup(datasetNames, graph, prefix) {
        let offsetY = 0;
        this.dropdownLayout.attach(graph, 0, offsetY, 3, 1);
        for (const dataset of datasetNames) {
            if (dataset.header) {
                const label = new St.Label({ style_class: 'title_label' });
                label.set_text(dataset.label);
                offsetY++;
                this.dropdownLayout.attach(label, 0, offsetY, 3, 1);
            }
            else {
                const keyName = `${prefix}-${dataset.name}-used`;
                const colorName = `${prefix}-${dataset.name}-color`;
                if (dataset.vue_meter) {
                    this.addDataSet(keyName, colorName);
                }
                graph.addDataSet(keyName, colorName);
                const label = new St.Label({
                    style_class: 'description_label',
                });
                const value = new St.Label({
                    style_class: 'value_label',
                });
                const box = new St.Bin({
                    style_class: `color_label bg-${prefix}-${dataset.name}-color`,
                    xExpand: false,
                    yExpand: false,
                    reactive: false,
                    xAlign: Clutter.ActorAlign.CENTER,
                    yAlign: Clutter.ActorAlign.CENTER,
                });
                this.currentColors[dataset.name] = box;
                this.currentLabels[dataset.name] = label;
                this.currentValues[dataset.name] = value;
                label.set_text(dataset.label);
                offsetY++;
                this.dropdownLayout.attach(box, 0, offsetY, 1, 1);
                this.dropdownLayout.attach(label, 1, offsetY, 1, 1);
                this.dropdownLayout.attach(value, 2, offsetY, 1, 1);
            }
        }
    }
});
