import GObject from 'gi://GObject';
import GTop from 'gi://GTop';
import { gettext as _ } from 'resource:///org/gnome/shell/extensions/extension.js';
import HorizontalGraph from '../horizontalgraph.js';
import Indicator from '../indicator.js';
import { Constantes } from '../types.js';
import Utils from '../utils.js';
export default GObject.registerClass(class CpuIndicator extends Indicator {
    constructor() {
        super('VueMeterMonitor.CpuIndicator');
        this.ncpu = 1;
        this.previousCpuInfos = new GTop.glibtop_cpu();
        this.percentUseCpu = [];
        this.datasetNames = [
            {
                name: 'none',
                label: _('Current:'),
                vue_meter: false,
                header: true,
                registre: '',
                color: Constantes.WHITE,
            },
            {
                name: 'user',
                label: _('User CPU usage'),
                vue_meter: false,
                header: false,
                registre: 'xcpu_user',
                color: Utils.fromStyles({
                    red: 10,
                    blue: 10,
                    green: 216,
                    alpha: 1,
                }),
            },
            {
                name: 'sys',
                label: _('System CPU usage'),
                vue_meter: false,
                header: false,
                registre: 'xcpu_sys',
                color: Utils.fromStyles({
                    red: 255,
                    blue: 20,
                    green: 20,
                    alpha: 1,
                }),
            },
            {
                name: 'total',
                label: _('Total CPU usage'),
                vue_meter: false,
                header: false,
                registre: 'xcpu_total',
                color: Utils.fromStyles({
                    red: 0,
                    blue: 154,
                    green: 62,
                    alpha: 1,
                }),
            },
        ];
        GTop.glibtop_get_cpu(this.previousCpuInfos);
        try {
            this.ncpu = GTop.glibtop_get_sysinfo().ncpu;
        }
        catch (e) {
            Utils.error(e);
        }
        for (let cpu = 0; cpu < this.ncpu; cpu++) {
            this.addDataSet('cpu_' + cpu, `cpu-color`);
            this.percentUseCpu[cpu] = {
                xcpu_total: 0,
                xcpu_user: 0,
                xcpu_sys: 0,
            };
        }
        this.graph = new HorizontalGraph('CpuIndicatorGraph', {
            autoscale: false,
            max: 100,
            units: '%',
            showMax: false,
        });
        this.buildPopup(this.datasetNames, this.graph, 'cpu');
        this.enable();
    }
    updateValues() {
        const cpu = new GTop.glibtop_cpu();
        const cpu_ttl_usage = {
            xcpu_total: 0,
            xcpu_user: 0,
            xcpu_sys: 0,
        };
        GTop.glibtop_get_cpu(cpu);
        for (let index = 0; index < this.ncpu; ++index) {
            const previous = {
                xcpu_total: this.previousCpuInfos.xcpu_total[index],
                xcpu_user: this.previousCpuInfos.xcpu_user[index],
                xcpu_sys: this.previousCpuInfos.xcpu_sys[index],
                xcpu_idle: this.previousCpuInfos.xcpu_idle[index],
            };
            const current = {
                xcpu_total: cpu.xcpu_total[index],
                xcpu_user: cpu.xcpu_user[index],
                xcpu_sys: cpu.xcpu_sys[index],
                xcpu_idle: cpu.xcpu_idle[index],
            };
            const statValues = {
                values: [],
            };
            for (const dataset of this.datasetNames) {
                if (dataset.header === false) {
                    const registre = dataset.registre;
                    const currentDelta = Math.max(current[registre] - previous[registre], 0);
                    let reading = 0;
                    if (dataset.name === 'total') {
                        const idle = Math.max(current.xcpu_idle - previous.xcpu_idle, 0);
                        if (currentDelta > 0) {
                            reading = 1.0 - idle / currentDelta;
                        }
                    }
                    else {
                        const totalDelta = Math.max(current.xcpu_total - previous.xcpu_total, 0);
                        if (totalDelta > 0) {
                            reading = currentDelta / totalDelta;
                        }
                    }
                    cpu_ttl_usage[registre] += reading;
                    const decayed_value = Math.min(this.percentUseCpu[index][registre] * this.options.decay, 0.999999999);
                    const statValue = {
                        visible: dataset.vue_meter || dataset.name === 'total',
                        value: Math.max(reading, decayed_value),
                        color: `cpu-${dataset.name}-color`,
                        cairo_color: dataset.color,
                    };
                    if (statValue.visible) {
                        statValues.values.push(statValue);
                    }
                    this.percentUseCpu[index][registre] = statValue.value;
                }
            }
            this.addDataPoint('cpu_' + index, statValues);
        }
        for (const dataset of this.datasetNames) {
            if (dataset.header === false) {
                const registre = dataset.registre;
                const keyName = `cpu-${dataset.name}-used`;
                const value = (cpu_ttl_usage[registre] * 100) / this.ncpu;
                const cpu_ttl_text = '%s%%'.format(Utils.formatMetricPretty(value));
                cpu_ttl_usage[registre] = value;
                this.graph?.addDataPoint(keyName, value);
                this.currentValues[dataset.name].set_text(cpu_ttl_text);
            }
        }
        this.previousCpuInfos = cpu;
    }
});
