import Gio from 'gi://Gio';
import GLib from 'gi://GLib';
import Config from './config.js';
class Utils {
    static release() {
        Utils.extension = null;
        Utils.metadata = null;
        Utils.settings = null;
        Config.settings = undefined;
    }
    static init(service, extension, metadata, settings) {
        Utils.extension = extension;
        Utils.metadata = metadata;
        Utils.settings = settings;
        Config.settings = settings;
        Utils.debugMode = Config.get_boolean('debug-mode');
        Utils.showMaxLines = Config.get_boolean('show-max-lines');
        Utils.memStack = Config.get_boolean('mem-stack');
        Utils.bitsPerSecond = Config.get_boolean('bits-per-second');
        if (service === 'extension') {
            Utils.settings.connect('changed::debug-mode', (sender, key) => {
                this.debugMode = sender.get_boolean(key);
                if (this.debugMode) {
                    this.clean_logFile();
                }
                else {
                    this.delete_logFile();
                }
            });
            if (this.debugMode) {
                this.clean_logFile();
            }
            Utils.settings.connect('changed::show-max-lines', (sender, key) => {
                this.showMaxLines = sender.get_boolean(key);
            });
            Utils.settings.connect('changed::mem-stack', (sender, key) => {
                this.memStack = sender.get_boolean(key);
            });
            Utils.settings.connect('changed::bits-per-second', (sender, key) => {
                this.bitsPerSecond = sender.get_boolean(key);
            });
        }
    }
    static delete_logFile() {
        try {
            const log = Utils.getLogFile();
            if (log) {
                if (log.query_exists(null))
                    log.delete(null);
            }
        }
        catch (e) {
            console.error(e);
        }
    }
    static clean_logFile() {
        try {
            const log = Utils.getLogFile();
            if (log) {
                if (log.query_exists(null))
                    log.delete(null);
                log.create_readwrite(Gio.FileCreateFlags.REPLACE_DESTINATION, null);
            }
        }
        catch (e) {
            console.error(e);
        }
    }
    static fromStyles(color) {
        return {
            red: color.red / 255,
            green: color.green / 255,
            blue: color.blue / 255,
            alpha: color.alpha / 255,
        };
    }
    static getLogFile() {
        try {
            const dataDir = GLib.get_user_cache_dir();
            const destination = GLib.build_filenamev([dataDir, 'vuemeter-system', 'debug.log']);
            const destinationFile = Gio.File.new_for_path(destination);
            if (destinationFile &&
                GLib.mkdir_with_parents(destinationFile.get_parent().get_path(), 0o755) === 0)
                return destinationFile;
        }
        catch (e) {
            console.error(e);
        }
        return null;
    }
    static logToFile(header, message) {
        if (this.debugMode) {
            const log = Utils.getLogFile();
            if (log) {
                try {
                    const date = new Date();
                    const time = date.toISOString().split('T')[1].slice(0, -1);
                    const outputStream = log.append_to(Gio.FileCreateFlags.NONE, null);
                    const buffer = new TextEncoder().encode(`${time}: ${header} - ${message}\n`);
                    outputStream.write_all(buffer, null);
                }
                catch (e) {
                    console.error(Utils.HEADER, e);
                }
            }
        }
    }
    static concat(...data) {
        let message = '';
        for (const str of data) {
            if (message.length === 0)
                message += str;
            else
                message += ' ' + str;
        }
        return message;
    }
    static log(...data) {
        const message = Utils.concat(...data);
        console.log(Utils.HEADER, message);
        Utils.logToFile('INFO', message);
    }
    static debug(...data) {
        const message = Utils.concat(...data);
        console.log(Utils.HEADER, message);
        Utils.logToFile('DEBUG', message);
    }
    static error(...data) {
        const message = Utils.concat(...data);
        console.log(Utils.HEADER, message);
        Utils.logToFile('ERROR', message);
    }
    static roundFloatingPointNumber(num) {
        const numStr = num.toString();
        const decimalIndex = numStr.indexOf('.');
        if (decimalIndex === -1)
            return num;
        const fractionLength = numStr.length - decimalIndex - 1;
        let precision = Math.min(10, fractionLength);
        if (fractionLength > 10)
            precision = fractionLength - 10;
        return Number(num.toFixed(precision));
    }
    static lookupColor(widget, name, defaultColor) {
        if (widget.get_stage()) {
            const themeNode = widget.get_theme_node();
            const [hasColor, color] = themeNode.lookup_color(name, true);
            if (hasColor) {
                if (Utils.debugMode) {
                    Utils.debug(`${widget.name}::lookupColor->${this.name}, name:${name} red: ${color.red}, blue: ${color.blue}, green: ${color.green}, alpha:${color.alpha}`);
                }
                defaultColor = {
                    red: color.red / 255.0,
                    blue: color.blue / 255.0,
                    green: color.green / 255.0,
                    alpha: color.alpha / 255.0,
                };
            }
            else {
                if (Utils.debugMode) {
                    Utils.debug(`${widget.name}::lookupColor->${this.name}, name:${name} not found, use red: ${defaultColor.red}, blue: ${defaultColor.blue}, green: ${defaultColor.green}, alpha:${defaultColor.alpha}`);
                }
            }
        }
        return defaultColor;
    }
    static formatMetricPretty(value, units) {
        let metricPrefix = '';
        if (value > 1024 * 1024) {
            value /= 1024 * 1024;
            metricPrefix = 'Mi';
        }
        else if (value > 1024) {
            value /= 1024;
            metricPrefix = 'Ki';
        }
        return '%0.2f %s%s'.format(value, metricPrefix, units || '');
    }
}
Utils.HEADER = 'vuemeter-system';
Utils.debugMode = false;
Utils.showMaxLines = true;
Utils.memStack = false;
Utils.bitsPerSecond = true;
export default Utils;
